#include "GUIInputRA6M3.h"

extern "C"
{
#include <ctype.h>
#include "task.h"
}

#define INVALID_AXIS_VALUE -1

#include "eC_String.h"
#include "GUITimer.h"
#include "GUITrace.h"

#include "GUIMemLeakWatcher.h"
#include "WindowsLeakWatcher.h"

#define QUEUED_EVENTS 1
#define DEBUG_TOUCH_EVENTS 0

static CGUIInputRA6M3* s_pkInputDevice = NULL;

extern "C"
{
// callback for touch-events
void touchevent_callback(touchevent_data_t* pkTouchData)
{
    if (
        (NULL != s_pkInputDevice) && (NULL != pkTouchData)
        )
    {
        s_pkInputDevice->AddEvent(*pkTouchData);
    }
}
}

CGUIInputRA6M3::CGUIInputRA6M3(
    const eC_Int& iTouchScreenX,
    const eC_Int& iTouchScreenY,
    const eC_Int& iTouchScreenWidth,
    const eC_Int& iTouchScreenHeight,
    const eC_Int& iTouchFilterDelta) :
    m_bEventReady(false),
    m_iTouchScreenX(iTouchScreenX),
    m_iTouchScreenY(iTouchScreenY),
    m_iTouchScreenWidth(iTouchScreenWidth),
    m_iTouchScreenHeight(iTouchScreenHeight),
    m_iTouchFilterDelta(iTouchFilterDelta),
    m_iLastDraggedX(INVALID_AXIS_VALUE),
    m_iLastDraggedY(INVALID_AXIS_VALUE),
    m_iLastPositionX(INVALID_AXIS_VALUE),
    m_iLastPositionY(INVALID_AXIS_VALUE)
{
    if (true == m_kSemaphore.Create("Semaphore"))
    {
        GUILOG(GUI_TRACE_DEBUG, "CGUIInputRA6M3: Semaphore created\n");
    }
    else
    {
        GUILOG(GUI_TRACE_DEBUG, "CGUIInputRA6M3: Semaphore NOT created\n");
    }

    s_pkInputDevice = this;
}

CGUIInputRA6M3::~CGUIInputRA6M3()
{
}

void CGUIInputRA6M3::CreateInstance(
    const eC_Int& iTouchScreenX,
    const eC_Int& iTouchScreenY,
    const eC_Int& iTouchScreenWidth,
    const eC_Int& iTouchScreenHeight,
    const eC_Int& iTouchFilterDelta)
{
    if (!CGUIComponentManager::GetInstance().HasInputMedia())
        CGUIComponentManager::GetInstance().SetInputMedia(new CGUIInputRA6M3(iTouchScreenX, iTouchScreenY, iTouchScreenWidth, iTouchScreenHeight, iTouchFilterDelta));
}

void CGUIInputRA6M3::AddEvent(touchevent_data_t kTouchEvent)
{
    eC_Bool bCreateEvent = false;

    if (
        (m_eLastTouchEvent.event != kTouchEvent.event) ||
        (abs(m_iLastPositionX - kTouchEvent.x) > m_iTouchFilterDelta) ||
        (abs(m_iLastPositionY - kTouchEvent.y) > m_iTouchFilterDelta)
        )
    {
        if (
            (m_eLastTouchEvent.event == kTouchEvent.event) ||
            (kTouchEvent.event == TOUCH_EVENT_MOVE)
            )
        {
            // just a move
            kTouchEvent.event = TOUCH_EVENT_MOVE;
            m_iLastPositionX = kTouchEvent.x;
            m_iLastPositionY = kTouchEvent.y;
        }
        else
        {
            switch (kTouchEvent.event)
            {
                case TOUCH_EVENT_UP:
                    m_iLastDraggedX = m_iLastDraggedY = INVALID_AXIS_VALUE;
                    m_iLastPositionX = m_iLastPositionY = INVALID_AXIS_VALUE;
                    break;

                case TOUCH_EVENT_DOWN:
                    m_iLastPositionX = kTouchEvent.x;
                    m_iLastPositionY = kTouchEvent.y;
                    break;


                default:
                    break;
            }            
        }

        m_eLastTouchEvent = kTouchEvent;
        bCreateEvent = true;
    }

    if (true == bCreateEvent)
    {
        // add event
#if QUEUED_EVENTS == 1
        m_kSemaphore.Enter();
        m_kEventQueue.AddAtEnd(kTouchEvent);
        m_kSemaphore.Leave();
#else
        m_kLastEvent = kTouchEvent;
        m_bEventReady = true;
#endif
        StopIdle();
    }
}

CGUIEvent* CGUIInputRA6M3::GetEvent(eC_UInt uiIdleTime)
{
    CGUIEvent *pEvent = NULL;
    const eC_UInt uiEndTime = GETTIMER.GetTime() + uiIdleTime;
    volatile eC_UInt uiTime = 0;

    // wait for driver until new event arrives or Idle time reached
    do
    {
        vTaskDelay((TickType_t)(10));
        uiTime = GETTIMER.GetTime();
    }
#if QUEUED_EVENTS == 1
    while ((uiTime < (uiEndTime)) && (!m_bStopIdle) && (m_kEventQueue.GetQuantity() == 0));
#else
    while ((uiTime < (uiEndTime)) && (!m_bStopIdle) && (!m_bEventReady));
#endif

    m_bStopIdle = false;
    GUI_TRY
    {
#if QUEUED_EVENTS == 1
        if (m_kEventQueue.GetQuantity() > 0)
        {
            m_kSemaphore.Enter();

            // Get the last event
            touchevent_data_t kTouchEvent;
            memset(&kTouchEvent, 0, sizeof(touchevent_data_t));
            m_kEventQueue.RemoveFront(kTouchEvent);
            
            if (kTouchEvent.event == TOUCH_EVENT_MOVE)
            {
                touchevent_data_t kNextEvent;
                while (m_kEventQueue.GetFirst(kNextEvent))
                {
                    if (kNextEvent.event == TOUCH_EVENT_MOVE)
                    {
                        m_kEventQueue.RemoveFront(kNextEvent);
                        kTouchEvent = kNextEvent;
                    }
                    else
                    {
                        break;
                    }
                }
            }

            m_kSemaphore.Leave();

            switch (kTouchEvent.event)
            {
                case TOUCH_EVENT_UP:
#if DEBUG_TOUCH_EVENTS == 1
                    GUILOG(GUI_TRACE_DEBUG, "Up-Event: " + eC_String(kTouchEvent.x) + " " + eC_String(kTouchEvent.y) + "\n");
#endif
                    pEvent = new CGUIMouseEvent(ET_LBUTTONUP, kTouchEvent.x, kTouchEvent.y);
                    break;

                case TOUCH_EVENT_DOWN:
#if DEBUG_TOUCH_EVENTS == 1
                    GUILOG(GUI_TRACE_DEBUG, "Down-Event: " + eC_String(kTouchEvent.x) + " " + eC_String(kTouchEvent.y) + "\n");
#endif
                    pEvent = new CGUIMouseEvent(ET_LBUTTONDOWN, kTouchEvent.x, kTouchEvent.y, true);
                    break;

                case TOUCH_EVENT_MOVE:
#if DEBUG_TOUCH_EVENTS == 1
                    GUILOG(GUI_TRACE_DEBUG, "Move-Event: " + eC_String(kTouchEvent.x) + " " + eC_String(kTouchEvent.y) + "\n");
#endif
                    pEvent = new CGUIMouseEvent(ET_MOUSEMOVE, kTouchEvent.x, kTouchEvent.y, true, false);
                    break;

                default:
                    break;
            }
        }
#else
        if (true == m_bEventReady)
        {
            switch (m_kLastEvent.event)
            {
            case TOUCH_EVENT_UP:
#if DEBUG_TOUCH_EVENTS == 1
                GUILOG(GUI_TRACE_DEBUG, "Up-Event: " + eC_String(m_kLastEvent.x) + " " + eC_String(m_kLastEvent.y) + "\n");
#endif
                pEvent = new CGUIMouseEvent(ET_LBUTTONUP, m_kLastEvent.x, m_kLastEvent.y);
                break;

            case TOUCH_EVENT_DOWN:
#if DEBUG_TOUCH_EVENTS == 1
                GUILOG(GUI_TRACE_DEBUG, "Down-Event: " + eC_String(m_kLastEvent.x) + " " + eC_String(m_kLastEvent.y) + "\n");
#endif
                pEvent = new CGUIMouseEvent(ET_LBUTTONDOWN, m_kLastEvent.x, m_kLastEvent.y, true);
                break;

            case TOUCH_EVENT_MOVE:
#if DEBUG_TOUCH_EVENTS == 1
                GUILOG(GUI_TRACE_DEBUG, "Move-Event: " + eC_String(m_kLastEvent.x) + " " + eC_String(m_kLastEvent.y) + "\n");
#endif
                pEvent = new CGUIMouseEvent(ET_MOUSEMOVE, m_kLastEvent.x, m_kLastEvent.y, true, false);
                break;

            default:
                break;
            }
            m_bEventReady = false;
        }
#endif
    }
    GUI_CATCH_ALL
    {
        GUILOG(GUI_TRACE_ERROR, "something went wrong while acquiring event.\n");
    }
    return pEvent;
}
