/*
* Copyright (C) 2004 TES Electronic Solutions GmbH,
* All Rights Reserved.
* This source code and any compilation or derivative thereof is the
* proprietary information of TES Electronic Solutions GmbH
* and is confidential in nature.
* Under no circumstances is this software to be exposed to or placed
* under an Open Source License of any type without the expressed
* written permission of TES Electronic Solutions GmbH
*
*############################################################
*/

/******************************************************************************
*   PROJECT:        Guiliani
*******************************************************************************
*
*    MODULE:        MyGUI_SR.cpp
*
*    Archive:       $URL: https://10.25.129.51:3690/svn/GSE/trunk/StreamRuntime/src/MyGUI_SR.cpp $
*
*    Date created:  2005
*
*
*
*    Author:        JRE
*
*******************************************************************************
*   MODIFICATIONS
*******************************************************************************
*    ID
*    --------------------------------------------------------------------------
*    $Id: MyGUI_SR.cpp 2159 2014-11-26 15:36:46Z christian.euler $
*
******************************************************************************/
#include "MyGUI_SR.h"

#include "GUIImage.h"
#include "GUILocalisationHandler.h"
#include "GUIImageStack.h"
#include "GUITextField.h"
#include "GUITransitionCmd.h"
#include "GUITransitionPush.h"
#include "GUIImage.h"
#include "GUIButton.h"
#include "GUIBaseMessageBox.h"
#include "GUIIconButton.h"
#include "GUIProperties.h"

#include "GUICommandHandler.h"

#include "CallApplicationAPICmd.h"

#include "GUIEasing.h"

#include "GUIBehaviourDecorator.h"
#include "GUIProfiler.h"

#include "GUIMemLeakWatcher.h"


static const eC_String DIALOG_SPLASH = "Splash";
static const eC_String DIALOG_MAIN = "Main";
static const eC_String DIALOG_SETTINGS = "Settings";
static const eC_String DIALOG_PAYMENT = "Payment";
static const eC_String DIALOG_PASSWORD = "Password";
static const eC_String DIALOG_ADJUSTMENT = "Adjustment";
static const eC_String DIALOG_PROCESSING = "Processing";

//ProductInformation main_info;

class IdleResetBehavior :public CGUIBehaviourDecorator
{

public:

    IdleResetBehavior(eC_Value& idleTimer) :
        m_vIdleTimer(idleTimer)
    {}

    virtual eC_Bool DoMouseMove(
        const eC_Value &vAbsX = eC_FromInt(-1),
        const eC_Value &vAbsY = eC_FromInt(-1))
    {
        m_vIdleTimer = 0;
        return true;
    }

    virtual eC_Bool DoClick(
        const eC_Value &vAbsX = eC_FromInt(-1),
        const eC_Value &vAbsY = eC_FromInt(-1))
    {
        m_vIdleTimer = 0;
        return true;
    }

    virtual eC_Bool DoDrag(
        const eC_Value &vDeltaX,
        const eC_Value &vDeltaY,
        const eC_Value &vAbsX,
        const eC_Value &vAbsY)
    {
        m_vIdleTimer = 0;
        return true;
    }

private:
    eC_Value & m_vIdleTimer;

};

// CGUIBehaviour 

CMyGUI::CMyGUI(
    eC_Value x, eC_Value y,
    eC_Value width, eC_Value height,
    ObjectHandle_t eID) :
    CStreamRuntimeGUI(x, y, width, height, eID),
    m_pkProduct(NULL), m_pkPassword(NULL),
    m_iMilk(20), m_iWater(20), m_iBean(20), m_iSugar(20),
    m_vSecondsForProccessing(0), m_vSecondsForIdle(0),
    m_vSeconds(0), m_vTime(60 * 60 * 8),
    m_eScreenForIdle(SCREEN_FIRST_MODE),
    m_bIdle(false),
    m_bEnglishActive(true),
    m_ubProfilingMode(0)
{
    GETTIMER.AddAnimationCallback(1000, this);
    IdleResetBehavior* pkIdleReset = new IdleResetBehavior(m_vSecondsForIdle);
    SetBehaviour(pkIdleReset);

    // initialize coffee states
    m_kProductInfo[0].eImageMode = DemoProduct::IM_BLACK_COFFEE_MODE;
    m_kProductInfo[0].eDisplayText = TXT_BLACK_COFFEE_HEADER;
    m_kProductInfo[0].iPrice = eC_ToInt(50);

    m_kProductInfo[1].eImageMode = DemoProduct::IM_WHITE_COFFEE_MODE;
    m_kProductInfo[1].eDisplayText = TXT_WHITE_COFFEE_HEADER;
    m_kProductInfo[1].iPrice = eC_ToInt(50);

    m_kProductInfo[2].eImageMode = DemoProduct::IM_ESPRESSO_MACH_MODE;
    m_kProductInfo[2].eDisplayText = TXT_ESPRESSO_MACHIATO_HEADER;
    m_kProductInfo[2].iPrice = eC_ToInt(50);

    m_kProductInfo[3].eImageMode = DemoProduct::IM_ESPRESSO_MODE;
    m_kProductInfo[3].eDisplayText = TXT_ESPRESSO_HEADER;
    m_kProductInfo[3].iPrice = eC_ToInt(50);

    m_kProductInfo[4].eImageMode = DemoProduct::IM_CAPPUCINO_MODE;
    m_kProductInfo[4].kDisplayString = eC_String("Cappuccino");
    m_kProductInfo[4].iPrice = eC_ToInt(50);

    m_kProductInfo[5].eImageMode = DemoProduct::IM_MOCHA_MODE;
    m_kProductInfo[5].kDisplayString = eC_String("Mocha");
    m_kProductInfo[5].iPrice = eC_ToInt(50);

    m_kProductInfo[6].eImageMode = DemoProduct::IM_LATTE_MACH_MODE;
    m_kProductInfo[6].kDisplayString = eC_String("Latte Macchiato");
    m_kProductInfo[6].iPrice = eC_ToInt(50);

    m_kProductInfo[7].eImageMode = DemoProduct::IM_HOT_CHOCOLATE_MODE;
    m_kProductInfo[7].eDisplayText = TXT_HOT_CHOCOLATE_HEADER;
    m_kProductInfo[7].iPrice = eC_ToInt(50);

    m_kProductInfo[8].eImageMode = DemoProduct::IM_HOT_WATER;
    m_kProductInfo[8].eDisplayText = TXT_HOT_WATER_HEADER;
    m_kProductInfo[8].iPrice = eC_ToInt(50);

    GETPROFILER.SetFont(FNT_AMBLE_LIGHT);
    GETPROFILER.SetTextColor(0xffffffff);

    ChangeLang(m_bEnglishActive);
}

CMyGUI::~CMyGUI()
{
    // Add application specific de-initialisation here if necessary
    Cleanup();
}

void CMyGUI::Cleanup()
{
    /* cleans up resources */
    if (m_pkProduct != NULL)
    {
        delete m_pkProduct;
        m_pkProduct = NULL;
    }
    if (m_pkPassword != NULL)
    {
        delete m_pkPassword;
        m_pkPassword = NULL;
    }
}

void CMyGUI::DoCallAPI(const eC_String& kAPI, const eC_String& kParam)
{
    GUILOGMESSAGE(" CallApplicationAPI called with API: " + kAPI + " and Param:" + kParam + "\n");

    // Add application API handlers here
    if (kAPI == "ChangeLang")
    {
        /*Change the language from the settings dialog*/
        m_bEnglishActive = !m_bEnglishActive;
        ChangeLang(m_bEnglishActive);
    }
    // select product
    else if (kAPI == "SelectProduct")
    {
        /*Sets the image and price*/
        if (NULL == m_pkProduct)
        {
            m_pkProduct = new DemoProduct(m_iWater, m_iMilk, m_iBean, m_iSugar);

            eC_Int iProductIndex;
            kParam.ToInt(iProductIndex);
            if ((iProductIndex >= 0) && (iProductIndex <= 8))
            {
                m_pkProduct->SetImageMode(m_kProductInfo[iProductIndex].eImageMode);
                if (m_kProductInfo[iProductIndex].eDisplayText == DUMMY_TEXT)
                {
                    m_pkProduct->SetProductText(m_kProductInfo[iProductIndex].kDisplayString);
                }
                else
                {
                    m_pkProduct->SetProductText(*(GETLOCALEHDL.GetText(m_kProductInfo[iProductIndex].eDisplayText)));
                }
                m_pkProduct->SetBasePrice(m_kProductInfo[iProductIndex].iPrice);
            }
            m_pkProduct->SetAdjustmentValue();
        }
    }
    //UpdatePicture
    else if (kAPI == "UpdatePicture")
    {
        /*Updates pictures, texts and sliders.*/
        if (kParam == "adj")
        {
            if (NULL != m_pkProduct)
            {
                // gray out adjustments
                GrayOutAdjSugar();
                GrayOutAdjMilk();
                GrayOutAdjFreness();
                // set adjustment slider
                SetAdjustmentsSliders();
                // add observers
                m_pkProduct->AddObservers();
            }
            m_pkProduct->m_bWereInAdjScreen = true;
            m_pkProduct->GrayOutBtnForwardAdj();
            // prapare screen must come after initialization of the product
            PrepareScreen(AID_ADJUSTMENT_SCREEN, SCREEN_ADJUSTMENT_MODE);
        }
        else if (kParam == "pay")
        {
            if (NULL != m_pkProduct)
            {
                // save adjustments info
                SaveAdjustmentSliders();
                UpdatePrice(AID_PAYMENT_SCREEN);
            }
            // prapare screen must come after initialization of the product
            PrepareScreen(AID_PAYMENT_SCREEN, SCREEN_PAYMENT_MODE);
        }
        else if (kParam == "proc")
        {
            PrepareScreen(AID_PROCESSING_SCREEN, SCREEN_PROCCESSING_MODE);
        }
        else if (kParam == "pass")
        {
            PrepareScreen(AID_PASSWORD_SCREEN, SCREEN_PASSWORD_MODE);
            // make sure password is reseted when you go from idle to password
            if (m_pkPassword != NULL)
            {
                m_pkPassword->CleanPassword();
                m_pkPassword->LetItGo();
            }
        }
        else if (kParam == "setting")
        {
            PrepareScreen(AID_SETTINGS_SCREEN, SCREEN_SETTINGS_MODE);
        }
        else if (kParam == "main")
        {
            PrepareScreen(AID_MAIN_SCREEN, SCREEN_MAIN_MODE);

            // make the right flag is displayed
            BackToMainLangFlag(m_bEnglishActive);

            // gray out unavailible products
            GrayOutUnavailible();
            m_vSecondsForProccessing = 0;
        }
    }
    //Back from pay
    else if (kAPI == "BackFromPay")
    {
        if (!(NULL == m_pkProduct))
        {
            if (m_pkProduct->m_bWereInAdjScreen)
            {
                GetBackToAdjScreen();
            }
            else
            {
                GetBackToMainScreen();
            }
        }
    }
    // refill
    else if (kAPI == "Refill")
    {
        Refill();
        SetTopBar(AID_SETTINGS_SCREEN);
    }
    // password
    else if (kAPI == "UpdatePassword")
    {
        if (NULL == m_pkPassword)
        {
            m_pkPassword = new DemoPassword();
        }
        if (NULL != m_pkPassword)
        {
            eC_Int iPasswordIndex;
            kParam.ToInt(iPasswordIndex);
            if ((iPasswordIndex >= 0) && (iPasswordIndex <= 11))
            {
                UpdatePassword(iPasswordIndex);
            }
        }
    }
    // payment
    else if (kAPI == "PaymentMethod")
    {
        if (NULL != m_pkProduct)
        {
            SelectPaymentMethod(kParam);
        }
    }
    else if (kAPI == "Clean")
    {
        Cleanup();
    }
    else if (kAPI == "Wake")
    {
        WakeFromIdle();
    }

    else if (kAPI == "PasswordPrep")
    {
        if (NULL == m_pkPassword)
        {
            m_pkPassword = new DemoPassword();
        }
        // let go of the forward button
        m_pkPassword->LetItGo();
    }
    else if (kAPI == "Help")
    {
        eC_String kHelpText = "";
        ObjectHandle_t eParent = NO_HANDLE;

        if (kParam == "Main")
        {
            kHelpText = "Click for fast-selection of beverage.\nLong-click for adjustments.";
            eParent = AID_MAIN_SCREEN;
        }
        else if (kParam == "Password")
        {
            kHelpText = "Password is 12345";
            eParent = AID_PASSWORD_SCREEN;
        }

        GETPROPHDL.SetGlobalColorProperty(GUI_PROP_DEFAULT_COLOR, 0xFF989898);
        GETPROPHDL.SetGlobalColorProperty(GUI_PROP_DEFAULT_HIGHLIGHT_COLOR, 0xFFA5A5A5);
        GETPROPHDL.SetGlobalColorProperty(GUI_PROP_DEFAULT_PRESSED_COLOR, 0xFF656565);
        CGUICompositeObject* pkParent = dynamic_cast<CGUICompositeObject*>(GETGUI.GetObjectByID(eParent));

        if (NULL != pkParent)
        {
            eC_Value vMessageBoxWidth = eC_FromInt(300);
            eC_Value vMessageBoxHeight = eC_FromInt(250);
            eC_Value vMessageBoxXPos = eC_Div((pkParent->GetWidth() - vMessageBoxWidth), 2);
            eC_Value vMessageBoxYPos = eC_Div((pkParent->GetHeight() - vMessageBoxHeight), 2);
            CGUIBaseMessageBox* pkMessageBox = new CGUIBaseMessageBox(pkParent, vMessageBoxXPos, vMessageBoxYPos, vMessageBoxWidth, vMessageBoxHeight, "", true);
            if (NULL != pkMessageBox)
            {
                pkMessageBox->SetBackgroundColor(0xFFEFEFEF);
                pkMessageBox->SetTextFont(FNT_AMBLE_LIGHT);
                pkMessageBox->SetText(kHelpText);
                pkMessageBox->SetButtonFont(FNT_AMBLE_LIGHT);
            }
        }
    }
    else if (kAPI == "Info")
    {
        switch (m_ubProfilingMode)
        {
            case 0:
                GETPROFILER.SetActive(true);
                GETPROFILER.SetDisplayActive(true);
                GETPROFILER.SetFPSActive(true);
                m_ubProfilingMode = 1;
                break;

            case 1:
                GETPROFILER.SetActive(true);
                GETPROFILER.SetDisplayActive(true);
                GETPROFILER.SetFPSActive(true);
                GETPROFILER.SetCPUProfilingActive(true);
                m_ubProfilingMode = 2;
                break;

            case 2:
                GETPROFILER.SetActive(true);
                GETPROFILER.SetDisplayActive(true);
                GETPROFILER.SetFPSActive(true);
                GETPROFILER.SetCPUProfilingActive(true);
                GETPROFILER.SetMemoryProfilingActive(true);
                m_ubProfilingMode = 3;
                break;

            case 3:
                GETPROFILER.SetActive(false);
                GETPROFILER.SetDisplayActive(false);
                GETPROFILER.SetFPSActive(false);
                GETPROFILER.SetCPUProfilingActive(false);
                GETPROFILER.SetMemoryProfilingActive(false);
                m_ubProfilingMode = 0;
                break;
        }
        GETGUI.InvalidateArea();
    }
}

/* ######################################################FUNCTIONS####################################################### */

// prepare screen
void CMyGUI::PrepareScreen(const ObjectHandle_t& eObjectID, ScreenMode_enum modeTmp)
{
    /* visually prepares upcoming screen */
    UpdatePicture(eObjectID);
    SetTopBar(eObjectID);
    SetScreenMode(modeTmp);

    m_vSecondsForIdle = 0;
}

// CHANGE LANGUAGE

void CMyGUI::ChangeLang(eC_Bool bEnglishActive)
{
    /* changes the language and picture of the flag in the main dialogue */
    ImageResource_t eImageID = IMG_ENGLISH_FLAG;
    eC_String kLanguageFile = "German.lng";
    if (bEnglishActive)
    {
        eImageID = IMG_GERMAN_FLAG;
        kLanguageFile = "English.lng";
    }

    GETLOCALEHDL.LoadLocalisationFile(kLanguageFile);
    CGUICompositeObject *pkDialogCompObj = dynamic_cast<CGUICompositeObject*>(GETGUI.GetObjectByID(AID_MAIN_SCREEN));
    if (NULL != pkDialogCompObj)
    {
        CGUIIconButton* pkLanguageButton = dynamic_cast<CGUIIconButton*>(pkDialogCompObj->GetObjectByID(BTN_LANGUAGE));
        if (NULL != pkLanguageButton)
        {
            pkLanguageButton->SetIconImages(eImageID, eImageID, eImageID, eImageID, eImageID);
        }
    }
    GETGUI.InvalidateArea();
}

// UPDATE PICTURE
void CMyGUI::UpdatePicture(const ObjectHandle_t& eObjectID)
{
    /* updates picture according to the given dialogue object id */
    if ((eObjectID == AID_ADJUSTMENT_SCREEN || eObjectID == AID_PAYMENT_SCREEN || eObjectID == AID_PROCESSING_SCREEN) && NULL != m_pkProduct)
    {
        m_pkProduct->SetPictureToObjectID(eObjectID, IMG_DYNAMIC);
        m_pkProduct->SetProductText(eObjectID);
    }

    if (eObjectID == AID_MAIN_SCREEN)
    {
        SetRightTime(AID_MAIN_SCREEN);
    }
    else if (eObjectID == AID_PASSWORD_SCREEN)
    {
        SetRightTime(AID_PASSWORD_SCREEN);
    }
    else if (eObjectID == AID_ADJUSTMENT_SCREEN)
    {
        SetRightTime(AID_ADJUSTMENT_SCREEN);
    }
    else if (eObjectID == AID_PAYMENT_SCREEN)
    {
        SetRightTime(AID_PAYMENT_SCREEN);
    }
    else if (eObjectID == AID_SETTINGS_SCREEN)
    {
        SetRightTime(AID_SETTINGS_SCREEN);
    }
    else if (eObjectID == AID_PROCESSING_SCREEN)
    {
        SetRightTime(AID_PROCESSING_SCREEN);
    }
}

// UPDATE PRICE
void CMyGUI::UpdatePrice(const ObjectHandle_t& eObjectID)
{
    /* updates price according to the selection in the payment dialogue */

    m_pkProduct->UpdatePrice(m_pkProduct->GetImageMode());
    m_pkProduct->SetPriceToObjectID(eObjectID, TXT_PREIS);
}

// GRAY OUT ADJUSTMENTS SCREEN
void CMyGUI::GrayOutAdjSugar()
{
    /* grays our sugar in the adjustments dialogue for the following products */
    if (m_pkProduct->GetImageMode() == DemoProduct::IM_HOT_WATER)
    {
        m_pkProduct->GrayOutObjectAdjustmentsSugar();
    }
}

void CMyGUI::GrayOutAdjMilk()
{
    /* grays our milk in the adjustments dialogue for the following products */
    if (m_pkProduct->GetImageMode() == DemoProduct::IM_BLACK_COFFEE_MODE)
    {
        m_pkProduct->GrayOutObjectAdjustmentsMilk();
    }
    else if (m_pkProduct->GetImageMode() == DemoProduct::IM_ESPRESSO_MODE)
    {
        m_pkProduct->GrayOutObjectAdjustmentsMilk();
    }
    else if (m_pkProduct->GetImageMode() == DemoProduct::IM_MOCHA_MODE)
    {
        m_pkProduct->GrayOutObjectAdjustmentsMilk();
    }
    else if (m_pkProduct->GetImageMode() == DemoProduct::IM_HOT_WATER)
    {
        m_pkProduct->GrayOutObjectAdjustmentsMilk();
    }
}

void CMyGUI::GrayOutAdjFreness()
{
    /* grays our freness in the adjustments dialogue for the following products */
    if (m_pkProduct->GetImageMode() == DemoProduct::IM_HOT_CHOCOLATE_MODE)
    {
        m_pkProduct->GrayOutObjectAdjustmentsFreness();
    }
    else if (m_pkProduct->GetImageMode() == DemoProduct::IM_HOT_WATER)
    {
        m_pkProduct->GrayOutObjectAdjustmentsFreness();
    }
}

// ADJUSTMENT SLIDERS
void CMyGUI::SetAdjustmentsSliders()
{
    /* sets the adjusment sliders in the adjustment dialogue */
    m_pkProduct->SetSizeDialog();
    m_pkProduct->SetSugarDialog();
    m_pkProduct->SetMilkDialog();
    m_pkProduct->SetFrenessDialog();
}

void CMyGUI::SaveAdjustmentSliders()
{
    /* saves the information from the adjusment sliders in the adjustment dialogue */
    m_pkProduct->SaveAllAdjustmentsData();
}

// PASSWORD SCREEN
void CMyGUI::UpdatePassword(eC_Int tmpNumber)
{
    /* upadtes password in the password dialogue */
    m_pkPassword->ButtonPress(tmpNumber);
}

void CMyGUI::CleanPassword()
{
    /* cleans password in the password dialogue */
    m_pkPassword->CleanPassword();
}

// PAYMENT METHOD
void CMyGUI::SelectPaymentMethod(eC_String stringTmp)
{
    /* grays out the forward button when the payment method is selected */
    if (stringTmp == "money")
    {
        m_pkProduct->SelectPaymentMethodMoney();
    }
    else if (stringTmp == "card")
    {
        m_pkProduct->SelectPaymentMethodCard();
    }
    else if (stringTmp == "smart")
    {
        m_pkProduct->SelectPaymentMethodSmart();
    }
    else if (stringTmp == "code")
    {
        m_pkProduct->SelectPaymentMethodCode();
    }

    CGUICompositeObject *pkDialogCompObj = dynamic_cast<CGUICompositeObject*>(GETGUI.GetObjectByID(AID_PAYMENT_SCREEN));
    if (NULL != pkDialogCompObj)
    {

        CGUIButton* pkGuiButtonTmp = dynamic_cast<CGUIButton*>(pkDialogCompObj->GetObjectByID(BTN_FORWARD));
        if (NULL != pkGuiButtonTmp)
        {
            pkGuiButtonTmp->SetGrayedOut(false);
        }
    }
}

// PAYMENT SCREEN BACK ARROW LOGIC
void CMyGUI::GetBackToAdjScreen()
{
    /* goes back to adjustments dialog from the payment dialog */
    CGUITransitionCmd* pkAdjScreenTransition = new CGUITransitionCmd(CGUITransitionCmd::PUSH_FROM_LEFT, AID_PAYMENT_SCREEN, DIALOG_ADJUSTMENT, 500);

    CallApplicationAPICmd* pkCallApplicationApiTmp = new CallApplicationAPICmd("UpdatePicture", "adj");
    pkAdjScreenTransition->AddAdditionalCmd(pkCallApplicationApiTmp);

    GETCMDHDL.Execute(pkAdjScreenTransition);
}

void CMyGUI::GetBackToMainScreen()
{
    /* goes back to main dialog from the payment dialog */
    CGUITransitionCmd* pMainScreenTransition = new CGUITransitionCmd(CGUITransitionCmd::PUSH_FROM_LEFT, AID_PAYMENT_SCREEN, DIALOG_MAIN);
    pMainScreenTransition->SetTransitionTime(500);

    CallApplicationAPICmd* pkCallApplicationApiTmpClean = new CallApplicationAPICmd("Clean", "");
    pMainScreenTransition->AddAdditionalCmd(pkCallApplicationApiTmpClean);
    CallApplicationAPICmd* pkCallApplicationApiTmpUpd = new CallApplicationAPICmd("UpdatePicture", "main");
    pMainScreenTransition->AddAdditionalCmd(pkCallApplicationApiTmpUpd);

    GETCMDHDL.Execute(pMainScreenTransition);
}

void CMyGUI::GetToMainScreen()
{
    /* goes to main dialog from the adjusments screen */
    CGUITransitionCmd* pMainScreenTransition = new CGUITransitionCmd(CGUITransitionCmd::PUSH_FROM_RIGHT, AID_FIRST_SCREEN, DIALOG_MAIN);
    pMainScreenTransition->SetTransitionTime(500);

    CallApplicationAPICmd* pkCallApplicationApiTmpUpd = new CallApplicationAPICmd("UpdatePicture", "main");
    pMainScreenTransition->AddAdditionalCmd(pkCallApplicationApiTmpUpd);

    GETCMDHDL.Execute(pMainScreenTransition);
}

void CMyGUI::GetToMainScreenWhenProcessed()
{
    /* goes to main dialog from the adjusments screen */
    CGUITransitionCmd* pMainScreenTransition = new CGUITransitionCmd(CGUITransitionCmd::PUSH_FROM_RIGHT, AID_PROCESSING_SCREEN, DIALOG_MAIN);
    pMainScreenTransition->SetTransitionTime(500);

    CallApplicationAPICmd* pkCallApplicationApiTmpUpd = new CallApplicationAPICmd("UpdatePicture", "main");
    pMainScreenTransition->AddAdditionalCmd(pkCallApplicationApiTmpUpd);

    GETCMDHDL.Execute(pMainScreenTransition);
}

// MAIN SCREEN UPADE FLAG

void CMyGUI::BackToMainLangFlag(eC_Bool bEnglishActive)
{
    /* if english is active the german flag will be displayed and vice versa */
    ImageResource_t eImageID = IMG_ENGLISH_FLAG;
    if (bEnglishActive)
        eImageID = IMG_GERMAN_FLAG;

    CGUICompositeObject *pkDialogCompObj = dynamic_cast<CGUICompositeObject*>(GETGUI.GetObjectByID(AID_MAIN_SCREEN));
    if (NULL != pkDialogCompObj)
    {
        CGUIIconButton* pkLanguageButton = dynamic_cast<CGUIIconButton*>(pkDialogCompObj->GetObjectByID(BTN_LANGUAGE));
        if (NULL != pkLanguageButton)
        {
            pkLanguageButton->SetIconImages(eImageID, eImageID, eImageID, eImageID, eImageID);
        }
    }
}

// DO ANIMATE
void CMyGUI::DoAnimate(const eC_Value & vTimes)
{
    /* animates dialog */
    m_vTime += vTimes;
    m_vSeconds += vTimes;

    // processing screen
    CGUITextField *pkDialogCompObj = static_cast<CGUITextField*>(GETGUI.GetObjectByID(TXT_PREPARING));
    if (pkDialogCompObj != NULL)
    {
        m_vSecondsForProccessing += vTimes;
        if (m_vSecondsForProccessing >= 5)
        {
            if (!(pkDialogCompObj->IsInvisible()))
            {
                if (m_pkProduct != NULL)
                {
                    // sets txt_preparing to invisible
                    PaymentComplete();
                    SetTopBar(AID_PROCESSING_SCREEN);
                }
                Cleanup();
                m_vSecondsForProccessing = 0;
            }
            else
            {
                // only works if txt_preparing is invisible
                GetToMainScreenWhenProcessed();
                m_vSecondsForProccessing = 0;
            }
        }
    }

    // very first transition from the frist screen
    if ((m_vSeconds >= 5) && (m_eScreenForIdle == SCREEN_FIRST_MODE))
    {
        GetToMainScreen();
    }
    else if (m_vSeconds >= 60)
    {
        // clamp if over 24 iHours
        if (m_vTime >= 24 * 60 * 60)
        {
            m_vTime = 0;
        }
        m_vSeconds -= 60;

        CGUITextField* pkTextClock = static_cast<CGUITextField*>(GETGUI.GetObjectByID(TXT_CLOCK));
        if (NULL != pkTextClock)
        {
            pkTextClock->SetValue(CleanTimeFormat());
        }

        // for settings menu
        pkTextClock = static_cast<CGUITextField*>(GETGUI.GetObjectByID(TXT_CLOCK_SMALL));
        if (NULL != pkTextClock)
        {
            pkTextClock->SetValue(CleanTimeFormat());
        }
    }

    // idle
    m_vSecondsForIdle += vTimes;
    if (m_vSecondsForIdle >= 30 && m_bIdle == false)
    {
        if (m_eScreenForIdle == SCREEN_MAIN_MODE)
        {
            SleepFromScreen(AID_MAIN_SCREEN);
            m_bIdle = true;
        }
        else if (m_eScreenForIdle == SCREEN_ADJUSTMENT_MODE)
        {
            SleepFromScreen(AID_ADJUSTMENT_SCREEN);
            m_bIdle = true;
        }
        else if (m_eScreenForIdle == SCREEN_SETTINGS_MODE)
        {
            SleepFromScreen(AID_SETTINGS_SCREEN);
            m_bIdle = true;
        }
        else if (m_eScreenForIdle == SCREEN_PASSWORD_MODE)
        {
            SleepFromScreen(AID_PASSWORD_SCREEN);
            m_bIdle = true;
        }
        else if (m_eScreenForIdle == SCREEN_PAYMENT_MODE)
        {
            SleepFromScreen(AID_PAYMENT_SCREEN);
            m_bIdle = true;
        }
    }
}

// TIME

void CMyGUI::SetRightTime(const ObjectHandle_t& eObjectID)
{
    /* changes the value of the textfiel in the dialog */
    CGUICompositeObject *pkDialogCompObj = dynamic_cast<CGUICompositeObject*>(GETGUI.GetObjectByID(eObjectID));
    if (NULL != pkDialogCompObj)
    {
        CGUITextField* pkTextClock = static_cast<CGUITextField*>(pkDialogCompObj->GetObjectByID(TXT_CLOCK));

        if (NULL != pkTextClock)
        {
            pkTextClock->SetValue(CleanTimeFormat());
        }

        if (eObjectID == AID_SETTINGS_SCREEN)
        {
            pkTextClock = static_cast<CGUITextField*>(pkDialogCompObj->GetObjectByID(TXT_CLOCK_SMALL));
            if (NULL != pkTextClock)
            {
                pkTextClock->SetValue(CleanTimeFormat());
            }
        }
    }
}

eC_String CMyGUI::CleanTimeFormat()
{
    /* returns appropriate string for the time display */
    eC_Int iHours = 0;
    eC_Int iMinutes = 0;

    // calculate horas
    iMinutes = (eC_ToInt(m_vTime) / 60) % 60;
    iHours = eC_ToInt(m_vTime) / (60 * 60);

    char text[8];
    sprintf(text, "%02i:%02i", iHours, iMinutes);
    eC_String tmpStringDef(text);

    // comprende
    return tmpStringDef;
}

// Payment Complete
void CMyGUI::PaymentComplete()
{
    /* changes text and picture when payment is completed in the processing dialog */
    CGUICompositeObject *pkDialogCompObj = dynamic_cast<CGUICompositeObject*>(GETGUI.GetObjectByID(AID_PROCESSING_SCREEN));
    if (NULL != pkDialogCompObj)
    {
        // TEXT
        CGUITextField* pkTextDone = static_cast<CGUITextField*>(pkDialogCompObj->GetObjectByID(TXT_DONE));
        if (NULL != pkTextDone)
        {
            pkTextDone->SetInvisible(false);
        }

        // set to null for later use
        pkTextDone = static_cast<CGUITextField*>(pkDialogCompObj->GetObjectByID(TXT_PREPARING));
        if (NULL != pkTextDone)
        {
            pkTextDone->SetInvisible(true);
        }

        pkTextDone = static_cast<CGUITextField*>(pkDialogCompObj->GetObjectByID(TXT_PRODUCT_TEXT));
        if (NULL != pkTextDone)
        {
            pkTextDone->SetInvisible(true);
        }
        // IMG
        CGUIImage* pkImageDone = static_cast<CGUIImage*>(pkDialogCompObj->GetObjectByID(IMG_DONE));

        if (NULL != pkImageDone)
        {
            pkImageDone->SetInvisible(false);
        }

        pkImageDone = static_cast<CGUIImage*>(pkDialogCompObj->GetObjectByID(IMG_DYNAMIC));

        if (NULL != pkImageDone)
        {
            pkImageDone->SetInvisible(true);
        }
        // Animated image
        CGUIAnimatedImage* pkAnimImg = static_cast<CGUIAnimatedImage*>(pkDialogCompObj->GetObjectByID(ANIM_PROCESSING));
        if (NULL != pkAnimImg)
        {
            pkAnimImg->SetInvisible(true);
        }
        // Button
        CGUIButton* pkButton = static_cast<CGUIButton*>(pkDialogCompObj->GetObjectByID(BTN_BACK_TO_MAIN));
        if (NULL != pkButton)
        {
            pkButton->SetInvisible(false);
        }
    }

    // substracts resources from resource pull after payment
    if (m_pkProduct != NULL)
    {
        m_iWater -= m_pkProduct->GetSize();
        m_iMilk -= m_pkProduct->GetMilk();
        m_iBean -= m_pkProduct->GetBean();
        m_iSugar -= m_pkProduct->GetSugar();
    }
}

// TOP BAR

void CMyGUI::SetTopBar(const ObjectHandle_t& eObjectID)
{
    /* sets the small objects on the top bar according to the changes made in the last dialog  */

    // Get the dialog of eObjectID, which is a composite object
    CGUICompositeObject *pkDialogCompObj = dynamic_cast<CGUICompositeObject*>(GETGUI.GetObjectByID(eObjectID));
    if (NULL != pkDialogCompObj)
    {
        CGUIImageStack* pkGuiImgStackTmp = dynamic_cast<CGUIImageStack*>(pkDialogCompObj->GetObjectByID(IMGSTACK_SMALL_WATER));
        if (NULL != pkGuiImgStackTmp)
        {
            if (m_iWater > 10)
            {
                pkGuiImgStackTmp->SetAnimationIndex(0);
            }
            else if (m_iWater <= 10 && m_iWater > 0)
            {
                pkGuiImgStackTmp->SetAnimationIndex(1);
            }
            else
            {
                pkGuiImgStackTmp->SetAnimationIndex(2);
            }
        }

        pkGuiImgStackTmp = dynamic_cast<CGUIImageStack*>(pkDialogCompObj->GetObjectByID(IMGSTACK_SMALL_SUGAR));
        if (NULL != pkGuiImgStackTmp)
        {
            if (m_iSugar > 10)
            {
                pkGuiImgStackTmp->SetAnimationIndex(0);
            }
            else if (m_iSugar <= 10 && m_iSugar > 0)
            {
                pkGuiImgStackTmp->SetAnimationIndex(1);
            }
            else
            {
                pkGuiImgStackTmp->SetAnimationIndex(2);
            }
        }

        pkGuiImgStackTmp = dynamic_cast<CGUIImageStack*>(pkDialogCompObj->GetObjectByID(IMGSTACK_SMALL_BEAN));
        if (NULL != pkGuiImgStackTmp)
        {
            if (m_iBean > 10)
            {
                pkGuiImgStackTmp->SetAnimationIndex(0);
            }
            else if (m_iBean <= 10 && m_iBean > 0)
            {
                pkGuiImgStackTmp->SetAnimationIndex(1);
            }
            else
            {
                pkGuiImgStackTmp->SetAnimationIndex(2);
            }
        }

        pkGuiImgStackTmp = dynamic_cast<CGUIImageStack*>(pkDialogCompObj->GetObjectByID(IMGSTACK_SMALL_MILK));
        if (NULL != pkGuiImgStackTmp)
        {
            if (m_iMilk > 10)
            {
                pkGuiImgStackTmp->SetAnimationIndex(0);
            }
            else if (m_iMilk <= 10 && m_iMilk > 0)
            {
                pkGuiImgStackTmp->SetAnimationIndex(1);
            }
            else
            {
                pkGuiImgStackTmp->SetAnimationIndex(2);
            }
        }
    }
}

void CMyGUI::SleepFromScreen(const ObjectHandle_t& eObjectID)
{
    /* goes to first dialogue screen from eObjectID dialogue */
    CGUITransitionCmd* pMainScreenTransition = new CGUITransitionCmd(CGUITransitionCmd::PUSH_FROM_RIGHT, eObjectID, DIALOG_SPLASH);
    pMainScreenTransition->SetTransitionTime(500);

    GETCMDHDL.Execute(pMainScreenTransition);
}

void CMyGUI::WakeFromIdle()
{
    // always go back to main-screen from idle
    CGUITransitionCmd* pkScreenTransition = new CGUITransitionCmd(CGUITransitionCmd::PUSH_FROM_RIGHT, AID_FIRST_SCREEN, DIALOG_MAIN);
    CallApplicationAPICmd* pkCallApplicationApiTmpUpd = new CallApplicationAPICmd("UpdatePicture", "main");
    pkScreenTransition->AddAdditionalCmd(pkCallApplicationApiTmpUpd);
    m_bIdle = false;

    pkScreenTransition->SetTransitionTime(500);
    GETCMDHDL.Execute(pkScreenTransition);
}

void CMyGUI::GrayOutUnavailible()
{
    /* grays out unavailible products in main screen (availability is determined by availible resources (water,bean,sugar,milk)) */
    CGUICompositeObject *pkDialogCompObj = dynamic_cast<CGUICompositeObject*>(GETGUI.GetObjectByID(AID_MAIN_SCREEN));

    if (NULL != pkDialogCompObj)
    {
        // everything needs to be set manually because this happens before the class product is created
        CGUIButton* pkButtonTmp = static_cast<CGUIButton*>(pkDialogCompObj->GetObjectByID(BTN_BLACK_COFFE));
        if (NULL != pkButtonTmp)
        {
            if (m_iWater < 2 || m_iSugar < 3 || m_iBean < 2)
            {
                pkButtonTmp->SetGrayedOut(true);
            }
        }

        pkButtonTmp = static_cast<CGUIButton*>(pkDialogCompObj->GetObjectByID(BTN_WHITE_COFFE));
        if (NULL != pkButtonTmp)
        {
            if (m_iWater < 2 || m_iSugar < 3 || m_iMilk < 2 || m_iBean < 2)
            {
                pkButtonTmp->SetGrayedOut(true);
            }
        }

        pkButtonTmp = static_cast<CGUIButton*>(pkDialogCompObj->GetObjectByID(BTN_ESPRESSO_MACH));
        if (NULL != pkButtonTmp)
        {
            if (m_iWater < 2 || m_iSugar < 3 || m_iMilk < 2 || m_iBean < 2)
            {
                pkButtonTmp->SetGrayedOut(true);
            }
        }

        pkButtonTmp = static_cast<CGUIButton*>(pkDialogCompObj->GetObjectByID(BTN_ESPRESSO));
        if (NULL != pkButtonTmp)
        {
            if (m_iWater < 1 || m_iSugar < 3 || m_iBean < 1)
            {
                pkButtonTmp->SetGrayedOut(true);
            }
        }

        pkButtonTmp = static_cast<CGUIButton*>(pkDialogCompObj->GetObjectByID(BTN_CAPPUCINO));
        if (NULL != pkButtonTmp)
        {
            if (m_iWater < 2 || m_iSugar < 3 || m_iMilk < 2 || m_iBean < 2)
            {
                pkButtonTmp->SetGrayedOut(true);
            }
        }

        pkButtonTmp = static_cast<CGUIButton*>(pkDialogCompObj->GetObjectByID(BTN_MOCHA));
        if (NULL != pkButtonTmp)
        {
            if (m_iWater < 1 || m_iSugar < 3 || m_iBean < 1)
            {
                pkButtonTmp->SetGrayedOut(true);
            }
        }

        pkButtonTmp = static_cast<CGUIButton*>(pkDialogCompObj->GetObjectByID(BTN_LATTE_MACH));
        if (NULL != pkButtonTmp)
        {
            if (m_iWater < 3 || m_iSugar < 3 || m_iMilk < 2 || m_iBean < 3)
            {
                pkButtonTmp->SetGrayedOut(true);
            }
        }

        pkButtonTmp = static_cast<CGUIButton*>(pkDialogCompObj->GetObjectByID(BTN_HOT_CHOCK));
        if (NULL != pkButtonTmp)
        {
            if (m_iWater < 3 || m_iSugar < 3 || m_iMilk < 2)
            {
                pkButtonTmp->SetGrayedOut(true);
            }
        }

        pkButtonTmp = static_cast<CGUIButton*>(pkDialogCompObj->GetObjectByID(BTN_HOT_WATER));
        if (NULL != pkButtonTmp)
        {
            if (m_iWater < 2)
            {
                pkButtonTmp->SetGrayedOut(true);
            }
        }
    }
}
