/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#if !defined(EXAMPLE_CONTROL_H)
#define EXAMPLE_CONTROL_H

#include "GUIObject.h"
#include "GUIColorPropertyObserver.h"

/** An example CGUIObject implementation that draws a rectangle with
    configureable border width and configurable colors.
  */
class ExampleControl : public CGUIObject, public CGUIColorPropertyObserver
{
public:
    /** Constructor.
        @param pkParent Pointer to the designated parent object.
        @param vX X-position relative to its parent object, i.e. the x-offset from the left border of the parent object
        @param vY Y-position relative to its parent object, i.e. the y-offset from the upper border of the parent object
        @param vWidth Width of the object
        @param vHeight Height of the object
        @param uiInnerColor Color to be used for drawing this control's center
               rectangle (everything except the border) in ARGB format.
        @param uiBorderColor Color to be used for drawing this control's frame
               in ARGB format.
        @param vBorderWidth The width of the border in pixels.
        @param eID Object identifier of this object (choose NO_HANDLE if none is required).
    */
    ExampleControl(
        CGUICompositeObject* const pkParent,
        const eC_Value& vX, const eC_Value& vY,
        const eC_Value& vWidth, const eC_Value& vHeight,
        eC_UInt uiInnerColor, eC_UInt uiBorderColor,
        eC_Value vBorderWidth = eC_FromInt(1),
        const ObjectHandle_t &eID = NO_HANDLE);

    // method used by the factory to create new instance
    static CGUIObject* Create()
    {
        return new ExampleControl(NULL, eC_FromInt(0), eC_FromInt(0), eC_FromInt(100), eC_FromInt(100), 0xffff0000, 0xff00ff00);
    }

    /// Returns the inner color.
    eC_UInt GetInnerColor() const;

    /// Returns the border color.
    eC_UInt GetBorderColor() const;

    /// Returns the border width.
    eC_Value GetBorderWidth() const;

    /** Sets a new inner color and invalidates this control's area to have it
        repainted by the framework.
        @param uiInnerColor The new color to be used for the inner-color.
      */
    void SetInnerColor(const eC_UInt& uiInnerColor);

    /** Sets a new inner color and invalidates this control's area to have it
        repainted by the framework.
        @param eInnerColor The new property to be used for the inner-color.
      */
    void SetInnerColor(const GlobalProperty_t& eInnerColor);

    /** Sets a new center color and invalidates this control's area to have it
        repainted by the framework.
        @param uiBorderColor The new color to be used for the border-color.
      */
    void SetBorderColor(const eC_UInt& uiBorderColor);

    /** Sets a new center color and invalidates this control's area to have it
        repainted by the framework.
        @param eBorderColor The new property to be used for the border-color.
      */
    void SetBorderColor(const GlobalProperty_t& eBorderColor);

    /** Sets a new border width and invalidates this control's area to have it
        repainted by the framework.
        @param vBorderWidth The new border width.
      */
    void SetBorderWidth(eC_Value vBorderWidth);

#if defined(GUILIANI_STREAM_GUI)
    /** Reads all object attributes from streaming file.
        This method is called by CGUIFactoryManager after one of the registered
        factories has created an instance of this class.
    */
    virtual void ReadFromStream();
#endif

#if defined(GUILIANI_WRITE_GUI)
    /** Writes all object attributes to the streaming file. A CGUIStreamWriter
        has to be initialized first.
        @param bWriteClassID This flag is used to select if writing of ControlID,
               leading and trailing tags is performed.
    */
    virtual void WriteToStream(const eC_Bool bWriteClassID=false);
#endif

protected:
    /// Called by the Guiliani framework when this control should paint itself.
    virtual eC_Bool DoDraw();

private:
    static const eC_UInt INDEX_INNERCOLOR;
    static const eC_UInt INDEX_BORDERCOLOR;

    /// The width of the border in (sub-) pixels.
    eC_Value m_vBorderWidth;

    static eC_Bool ms_bInitialized;
};

#endif // EXAMPLE_CONTROL_H
