/****************************************************************************
* copyright:       TES Electronic Solutions GmbH
* project:         eDBG
* file name:       eDBG.h
* %version:        134.0 %
*
* author:          hh04005
* creation date:   Thu May 04 17:09:10 2006
* description:

* %date_modified:  % (%derived_by: hh04005 %)
*
* history:
* date         | name          | description of modifications
*-------------------------------------------------------------------------
  05-08-03     | hh04031       | Exchange fileheader with CM Synergy-Tags

******************************************************************************/

#ifndef EDBG_DEBUG__H_
#define EDBG_DEBUG__H_

//---------------------------------------------------------------
#include "eC_Config.h"
#include "eDBG_Config.h"
#include "eDBG_Statistic.h"

//#define  PROFILE_BLIT
#if defined PROFILE_BLIT
#include "eDBG_RenderCallProfiler.h"
#endif //PROFILE_BLIT

//---------------------------------------------------------------
#ifdef DOCXX

/**
@name Component_eDBG
@memo
The <B>e</B>mbedded <B>D</B>e<B>B</B>u<B>G</B>ging and profiling component.
@doc
Debugging and Profiling macros and functions can be enabled and disabled in serveral steps. Calls to disabled functions are removed at
compiletime. The output generated by the debugfunctions can be redirected to several devices, including platform specific ones. <BR>
Within the <I>eDBG_Config.h</I> file ("<I>$(eGML_ROOT)/comps/eDBG/code</I>" folder) you can select the following levels of debug output:
<PRE>
eDBG_DEBUG_LEVEL <B>0</B> - all debugmacros are removed
eDBG_DEBUG_LEVEL <B>1</B> - asserts only
eDBG_DEBUG_LEVEL <B>2</B> - <B>eGML</B> intern level1 logs, asserts and debugprints
eDBG_DEBUG_LEVEL <B>3</B> - all <B>eGML</B> intern logs, asserts and debugprints
<BR>
eDBG_DEBUG_APPLICATION_LEVEL <B>0</B> - no application logs
eDBG_DEBUG_APPLICATION_LEVEL <B>1</B> - level1 application logs
eDBG_DEBUG_APPLICATION_LEVEL <B>2</B> - level1&2 application logs
</PRE>
To get more information about {@link Component_eDBG eDBG} and its handling, look at the \Ref{Tutorial_eDBG}.
@see eDBG_DEBUG_LEVEL, eDBG_DEBUG_APPLICATION_LEVEL
@see Tutorial_eDBG
*/

//@{

//@Include: ../../comps/eDBG/code/eDBG_Config.h

/**
@memo
Assert (level1).
@doc
Asserts are used to embed preconditions into the code.
When compiling with \Ref{eDBG_DEBUG_LEVEL} = 0 calls to this function are removed. <BR>
The output made on the debugtarget will look like this:
<PRE>
sourcefile.cpp(line) Assert: yourmessage
</PRE>
formatstring is optional parameters but it is recommended that you include the values of some critical variables
into the assert output to make debugging easier.
@param expr If value is set to <B>false</B> the assert is triggered.
@param format Standard <I>printf</I> syntax (note that you do not have to include newline codes).
@param ... Any number of additional parameters referenced by the format string.
@see eDBG_DP, eDBG_LOG, eDBG_LOG2
*/
void eDBG_ASSERT(bool expr, const char *format, ...);

/**
@memo
Debugprint (level2).
@doc
Works like standard printf but the output is written to debugtarget.
When compiling with \Ref{eDBG_DEBUG_LEVEL} < 1 calls to this function are removed by the compiler. <BR>
DP's are meant to be used to output additional information while debugging a certain piece of code,
or logging general information during initialisation/shutdown. When detecting a warning or minor error
condition you should rather use <TT>LOG</TT> instead of <TT>DP</TT>. <BR>
The output made on the debugtarget will look like this:
<PRE>
DP: yourmessage
</PRE>
@param format Standard <I>printf</I> syntax (note that you do not have to include newline codes).
@param ... Any number of additional parameters referenced by the format string.
@see eDBG_LOG, eDBG_LOG2, eDBG_ASSERT
*/
void eDBG_DP(const char *format, ...);

/**
@memo
Warning / Information (applevel1).
@doc
Log works like printf and is similar to \Ref{eDBG_DP} but it will output the sourcefile and linenumber
along with your message. It is meant to be used to log nonfatal error conditions or warnings detected
by the application. LOG is not used internally by the <B>eGML</B> (there is an intern log function instead)
so you can turn application and library output on and off independently.
Level1 application logoutput is removed when compiling with \Ref{eDBG_DEBUG_APPLICATION_LEVEL} = 0. <BR>
The output made on the debugtarget will look like this:
<PRE>
sourcefile.cpp(line) logmessage
</PRE>
@param format Standard <I>printf</I> syntax (note that you do not have to include newline codes).
@param ... Any number of additional parameters referenced by the format string.
@see eDBG_DP, eDBG_LOG2, eDBG_ASSERT
*/
void eDBG_LOG(const char *format, ...);

/**
@memo
Warning / Information (applevel2).
@doc
Same as \Ref{eDBG_LOG} except that it has a lower priority. In order to be able to reduce the logoutput to
the most important information you can disable <TT>LOG2</TT> but still include <TT>LOG</TT>. <TT>LOG2</TT> is not used internally by
the <B>eGML</B> (there is an intern log function instead) so you can turn application and library output
on and off independently.
Level2 application logoutput is removed when compiling with \Ref{eDBG_DEBUG_APPLICATION_LEVEL} < 2. <BR>
The output made on the debugtarget will look like this:
<PRE>
sourcefile.cpp(line) logmessage
</PRE>
@param format Standard <I>printf</I> syntax (note that you do not have to include newline codes).
@param ... Any number of additional parameters referenced by the format string.
@see eDBG_DP, eDBG_LOG, eDBG_ASSERT
*/
void eDBG_LOG2(const char *format, ...);

//@Include: ../../comps/eDBG/code/eDBG_Statistic.h

//@}

#endif

//---------------------------------------------------------------
// !!! INTERNAL USE ONLY !!!

//---------------------------------------------------------------
#define eDBG_NOP if (1) {;} else eDBG_Nop

//---------------------------------------------------------------
// level0 - no debug macros
//
#if (eDBG_DEBUG_LEVEL==0)

#define eDBG_DEBUG_INIT
#define eDBG_DEBUG_EXIT

#define eDBG_ASSERT			if (1) {;} else eDBG_Assert
#define eDBG_DP				eDBG_NOP
#define eDBG_ILOG			eDBG_NOP
#define eDBG_ILOG2			eDBG_NOP
#define eDBG_MARK		

inline void eDBG_Assert(bool expr, const char *format, ...) { ; }
inline void eDBG_Assert(bool expr) { ; }
inline void eDBG_Nop(const char *format, ...) { ; }

#undef eDBG_DEBUG_APPLICATION_LEVEL
#define eDBG_DEBUG_APPLICATION_LEVEL 0

//---------------------------------------------------------------
// get position information
//
#else
#define eDBG_MARK			eDBG_DebugMark(__FILE__, __LINE__)

extern const char *g_eDBGL_acFile;
extern unsigned int g_eDBG_uiLine;

inline void eDBG_DebugMark(const char *acFile, unsigned int uiLine)
{
    g_eDBGL_acFile = acFile;
    g_eDBG_uiLine = uiLine;
}

extern const char *EC_FNKT(eDBG_AddPos)(const char *acString);

#endif
//---------------------------------------------------------------
// level1 - asserts only
//
#if (eDBG_DEBUG_LEVEL>0)

#define eDBG_DEBUG_INIT		eDBG_DEBUG_INIT
#define eDBG_DEBUG_EXIT		eDBG_DEBUG_EXIT

extern void eDBG_DEBUG_INIT();
extern void eDBG_DEBUG_EXIT();

#define eDBG_ASSERT			eDBG_MARK, eDBG_Assert

extern void EC_FNKT(eDBG_Assert)(bool expr, const char *format, ...);
inline void eDBG_Assert(bool expr) { eDBG_Assert(expr, "failed"); }

#endif
#if (eDBG_DEBUG_LEVEL==1)

#define eDBG_DP				eDBG_NOP
#define eDBG_ILOG			eDBG_NOP
#define eDBG_ILOG2			eDBG_NOP

inline void eDBG_Nop(const char *format, ...) { ; }

#if (eDBG_DEBUG_APPLICATION_LEVEL>0)
extern void EC_FNKT(eDBG_Log)(const char *format, ...);
#endif

#endif
//---------------------------------------------------------------
// level2 - asserts, debugprints and <B>eGML</B> level1 logs 
//
#if (eDBG_DEBUG_LEVEL>1)

#define eDBG_DP				eDBG_DebugPrint
#define eDBG_ILOG			eDBG_MARK, eDBG_Log

extern void EC_FNKT(eDBG_DebugPrint)(const char *format, ...);
extern void EC_FNKT(eDBG_Log)(const char *format, ...);

#endif
#if (eDBG_DEBUG_LEVEL==2)

#define eDBG_ILOG2			eDBG_NOP

inline void eDBG_Nop(const char *format, ...) { ; }

#endif
//---------------------------------------------------------------
// level3 - asserts, debugprints, <B>eGML</B> level 1 and 2 logs
//
#if (eDBG_DEBUG_LEVEL>2)

#define eDBG_ILOG2			eDBG_MARK, eDBG_Log

#endif
#if (eDBG_DEBUG_LEVEL==3)

inline void eDBG_Nop(const char *format, ...) { format; /* To avoud warnings */ }

#endif

//---------------------------------------------------------------
// applevel0 - no application logs
//
#if (eDBG_DEBUG_APPLICATION_LEVEL==0)

#define eDBG_LOG			eDBG_NOP
#define eDBG_LOG2			eDBG_NOP

//---------------------------------------------------------------
// applevel1 - only level 1 application logs
//
#elif (eDBG_DEBUG_APPLICATION_LEVEL==1)

#define eDBG_LOG			eDBG_MARK, eDBG_Log
#define eDBG_LOG2			eDBG_NOP

//---------------------------------------------------------------
// applevel2 - level 1 ans 2 application logs
//
#elif (eDBG_DEBUG_APPLICATION_LEVEL==2)

#define eDBG_LOG			eDBG_MARK, eDBG_Log
#define eDBG_LOG2			eDBG_MARK, eDBG_Log

#endif


//---------------------------------------------------------------
#endif

