/*
* Copyright (C) 2004 TES Electronic Solutions GmbH,
* All Rights Reserved.
* This source code and any compilation or derivative thereof is the
* proprietary information of TES Electronic Solutions GmbH
* and is confidential in nature.
* Under no circumstances is this software to be exposed to or placed
* under an Open Source License of any type without the expressed
* written permission of TES Electronic Solutions GmbH
*
*############################################################
*/

#include "StreamRuntimeGUI.h"
#include "StreamRuntimeConfig.h"

#include "GUIProperties.h"
#include "GUILocalisationHandler.h"
#include "GUIResourceFileHandler.h"
#include "GUIResourceManager.h"
#include "GUIPropertyResource.h"

#include "GfxWrap.h"
#include "GUIFactoryManager.h"
#include "CustomExtensionFactory.h"
#include "GUITrace.h"

#ifdef GUILIANI_STREAM_GUI
#include "GUIStreamReaderBinary.h"
#include "GUIStreamReaderXML.h"
#endif

#ifdef GUILIANI_ANIMATION_ACTIVE
#include "GUIAnimationHandler.h"
#endif

#include "CustomExtensionFuncs.h"
#include "CustomStandardFactory.h"

#include "GUIException.h"
#include "GUIDataPool.h"
#include "GUIEvent.h"
#include "GUIFileSysWrap.h"

#include "GUIBitmapWriter.h"

#include "GUIMemLeakDetection.h"
#include "GUIMemLeakWatcher.h"

namespace NStreamRuntime
{
    CStreamRuntimeGUI::CStreamRuntimeGUI(
        eC_Value x, eC_Value y,
        eC_Value width, eC_Value height,
        ObjectHandle_t eID) :
        CGUI(x, y, width, height, eID),
        m_bScreenshotActive(false),
        m_kScreenshotPrefix("screenshot"),
        m_eScreenshotKey(GK_S),
        m_uiScreenshotModifier(CGUIEvent::GKM_CONTROL)
#ifdef GUILIANI_STREAM_GUI
        ,m_pkStreamReader(NULL, false)
#endif
    {
        Init();
    }

    CStreamRuntimeGUI::~CStreamRuntimeGUI()
    {
#ifdef GUILIANI_STREAM_GUI
        CGUIFactoryManager::DeleteInstance();
#endif
    }

    void CStreamRuntimeGUI::Init()
    {
        // Dynamically registered resources shall not be automatically unloaded if their request count reaches zero.
        // In a Gui created within GSE basically ALL resources will be dynamic.
        GETRESMANAGER.KeepDynamicImages(true);
        GETRESMANAGER.KeepDynamicFonts(true);
        GETRESMANAGER.KeepDynamicSounds(true);
        GETRESMANAGER.KeepDynamicGeneralResources(true);

        SetFocussable(true);
        try
        {
#ifdef GUILIANI_STREAM_GUI
            try
            {
                CGUIFactoryManager::CreateInstance();
                // Add the factory for Guiliani's standard widgets first
#ifdef STREAMRUNTIME_CUSTOM_STANDARD_FACTORY_EXISTS
                GETFACTORY.AddUserFactory(new CustomStandardFactory());
#else
                GETFACTORY.AddUserFactory(new CGUIStandardFactory());
#endif
                // Add factory for custom extensions
                GETFACTORY.AddUserFactory(GETEXTENSIONSFACTORY);
                // Dependant on StreamRuntime configuration the necessary StreamReader is
                // created (XML or binary streaming mode).
                GETFACTORY.SetDefaultStreamingMode(GETRUNTIMECONFIG.GetStreamingMode());

                switch (GETFACTORY.GetDefaultStreamingMode())
                {
                case FILE_TYPE_XML:
                    // XML based dialog files.
                    m_pkStreamReader = new CGUIStreamReaderXML();
                    break;

                case FILE_TYPE_BINARY:
                    // Guiliani binary based dialog files.
                    m_pkStreamReader = new CGUIStreamReaderBinary();
                    break;

                case FILE_TYPE_CUSTOM:
                    GUILOG(GUI_TRACE_ERROR, "unknown file-type.\n");
                    break;
                }
            }
            catch (...)
            {
                GUILOG_THROW_EXCEPTION(CGUIException(), "Init: CGUIFactoryManager and StreamReaders could not be created.\n");
            }
#endif

            // Now let's start the GUI:
            // let the factory create the objects, getting the arguments for the construction from the streaming file
            StreamGUI();

            // Trigger an initial refresh of the screen
            GETGFX.InvalidateRect(0, 0, eC_FromInt(GETGFX.GetPhysicalScreenWidth()), eC_FromInt(GETGFX.GetPhysicalScreenHeight()));
        }
        catch (...)
        {
            // As the CGUI destructor functionality is not automatically called when leaving via an exception
            // we clean up here explicitly
            RemoveAllChildren();
            throw;
        }
    }

    void CStreamRuntimeGUI::StreamGUI()
    {
#ifdef GUILIANI_STREAM_GUI
        // since version 4 the filenames are not completely stored
        if (GETRUNTIMECONFIG.GetVersion() >= 4)
        {
            GETRESMANAGER.SetPredictResourceFileExtension(true);
            GETRESMANAGER.SetAddResourceFilePrefix(true);
        }

        // Load the initial language
        eC_String kFileName = GETRUNTIMECONFIG.GetLanguageFileName();
        if (!kFileName.IsEmpty())
        {
            GETLOCALEHDL.LoadLocalisationFile(kFileName);
        }

        // Register images, fonts, sounds etc. via resource files (if configured)
        kFileName = GETRUNTIMECONFIG.GetPropertyFileName();
        if (!kFileName.IsEmpty())
        {
            GETPROPHDL.ReadGlobalPropertiesFromFile(kFileName);
        }

        kFileName = GETRUNTIMECONFIG.GetImageFileName();
        if (!kFileName.IsEmpty())
        {
            GETRESMANAGER.RegisterImageResourcesFromFile(kFileName);
        }

        kFileName = GETRUNTIMECONFIG.GetFontFileName();
        if (!kFileName.IsEmpty())
        {
            GETRESMANAGER.RegisterFontResourcesFromFile(kFileName);
        }

        kFileName = GETRUNTIMECONFIG.GetSoundFileName();
        if (!kFileName.IsEmpty())
        {
            GETRESMANAGER.RegisterSoundResourcesFromFile(kFileName);
        }

        kFileName = GETRUNTIMECONFIG.GetGenResFileName();
        if (!kFileName.IsEmpty())
        {
            GETRESMANAGER.RegisterGeneralResourcesFromFile(kFileName);
        }

        // Load the DataPool file
        kFileName = GETRUNTIMECONFIG.GetDataPoolFileName();
        if (!kFileName.IsEmpty())
        {
            CGUIDataPool::ReadDataPoolFromFile(kFileName);
        }

#ifdef GUILIANI_ANIMATION_ACTIVE
        kFileName = GETRUNTIMECONFIG.GetAnimationsFileName();
        if (!kFileName.IsEmpty())
        {
            GETANIMATIONHDL.ReadAnimationsFromFile(kFileName);
        }
#endif

        // Load initial dialog
        kFileName = GETRUNTIMECONFIG.GetDialogFileName();
        if (kFileName.IsEmpty())
        {
            GUILOG(GUI_TRACE_ERROR, "StreamGUI: No dialog name configured.\n");
        }
        else
        {
            RemoveAllChildren();
            AddObject(GETFACTORY.LoadDialogFromFile(kFileName));
        }
#endif
    }

    eC_Bool CStreamRuntimeGUI::DoDraw()
    {
        // If the configuration requests us to fill the background in a given color
        if (!GETRUNTIMECONFIG.IsBackgroundEnabled())
            return false;
        // Set color
        GETGFX.SetForegroundColor(GETRUNTIMECONFIG.GetBackgroundColor());
        // Fill the GUI area with this color
        GETGFX.FilledRect(GetAbsRect());
        return true;
    }

    eC_Bool CStreamRuntimeGUI::DoKeyDown(const GUIKeyIdentifier_t& eKeyID, const eC_UInt& uiModifiers)
    {
        if (
            (m_bScreenshotActive) &&
            (eKeyID == m_eScreenshotKey) &&
            (uiModifiers & m_uiScreenshotModifier)
            )
        {
            CGUIBitmapWriter::CreateScreenshot(m_kScreenshotPrefix);
        }

#ifdef GUILIANI_LEAK_DETECTION
        if (
            (eKeyID == GK_M) &&
            (uiModifiers & CGUIEvent::GKM_SHIFT)
            )
        {
            LEAK_DETECTION.WriteResultsToFile();
        }
#endif

        return false;
    }

    void CStreamRuntimeGUI::EnableScreenshot(
        const eC_String& kScreenshotPrefix,
        const GUIKeyIdentifier_t& eScreenshotKey,
        const eC_UInt& uiScreeenshotModifier)
    {
        m_bScreenshotActive = true;
        m_kScreenshotPrefix = kScreenshotPrefix;
        m_eScreenshotKey = eScreenshotKey;
        m_uiScreenshotModifier = uiScreeenshotModifier;
    }
}
