/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef STREAMRUNTIMEBASE_H
#define STREAMRUNTIMEBASE_H

#include "eC_String.h"

#include "StreamRuntimeConfig.h"

/** The NStreamRuntime Namespace contains generic helper classes for initializing a default StreamRuntime application.
    The code herein typically remains widely unchanged across most StreamRuntimes.
    Feel free to remove these classes from your own application and replace them with custom code, if you wish.*/
namespace NStreamRuntime
{
    class CStreamRuntimeBase
    {
    public:
        CStreamRuntimeBase();

        virtual ~CStreamRuntimeBase();

        /**
        process the commandline-parameters and set internal settings
        @param uiCommandCount number of parameters
        @param pkCommandLine list of commandline-parameters
        */
        void ProcessCommandLine(eC_UInt uiCommandCount, eC_String* pkCommandLine);

        /**
        load specified configuration-file (default: StreamRuntimeConfig.xml)
        */
        eC_Bool LoadConfiguration();

        /**
        setup all needed wrapper-classes for specific platform
        */
        virtual void ConstructWrapperClasses() = 0;

        /**
        cleanup all instantiated wrapper-classes.
        */
        virtual void DestructWrapperClasses();

        void CreateGUI();

        void DeInit();

        virtual void UpdateWindowTitle() {}

    protected:
        /* getter-methods are protected for safety. only usable in ConstructWrapperClasses-method */

        eC_String GetConfigurationFile() const
        {
            return m_kConfigurationFile;
        }

        /**
        get requested width of screen. will be overwritten by commandline
        */
        eC_Int GetScreenWidth() const
        {
            if (m_iScreenWidth == -1)
                return GETRUNTIMECONFIG.GetScreenWidth();
            else
                return m_iScreenWidth;
        }

        /**
        get requested height of screen. will be overwritten by commandline
        */
        eC_Int GetScreenHeight() const
        {
            if (m_iScreenHeight == -1)
                return GETRUNTIMECONFIG.GetScreenHeight();
            else
                return m_iScreenHeight;
        }

        eC_Bool IsWindowResizable() const
        {
            return GETRUNTIMECONFIG.IsWindowResizable();
        }

        eC_String GetKeyboardDevice() const
        {
            return m_kKeyboardDevice;
        }

        eC_String GetMouseDevice() const
        {
            return m_kMouseDevice;
        }

        eC_String GetRemoteDevice() const
        {
            return m_kRemoteDevice;
        }

        eC_String GetTouchDevice() const
        {
            return m_kTouchDevice;
        }

        eC_Int GetTouchSizeX() const
        {
            return m_iTouchSizeX;
        }

        eC_Int GetTouchSizeY() const
        {
            return m_iTouchSizeY;
        }

        eC_Int GetTouchOffsetX() const
        {
            return m_iTouchOffsetX;
        }

        eC_Int GetTouchOffsetY() const
        {
            return m_iTouchOffsetY;
        }

        eC_Bool IsScreenshotEnabled() const
        {
            return m_bScreenshotsEnabled;
        }
        
        eC_Bool IsInputDebugModeEnable() const
        {
            return m_bInputDebugMode;
        }

    private:
        /**
        initialize common parts of the Guiliani-framework, e.g. Tracing
        */
        void CommonInit();

        void PrintUsage();

    private:
        eC_Bool m_bUsagePrinted;        ///< help was printed
        eC_String m_kLicense;           ///< license to be used
        eC_String m_kLicenseFile;       ///< license-file to be used
        eC_String m_kConfigurationFile; ///< name of configuration-file
        eC_Int m_iScreenWidth;          ///< screen width
        eC_Int m_iScreenHeight;         ///< screen height
        eC_String m_kKeyboardDevice;    ///< name of keyboard-device to use
        eC_String m_kMouseDevice;       ///< name of mouse-device to use
        eC_String m_kTouchDevice;       ///< name of touch-device to use
        eC_String m_kRemoteDevice;      ///< name of remote-device to use
        eC_Int m_iTouchSizeX;           ///< maximum horizontal size of touch-device
        eC_Int m_iTouchSizeY;           ///< maximum vertical size of touch-device
        eC_Int m_iTouchOffsetX;         ///< horizontal offset for touch-device
        eC_Int m_iTouchOffsetY;         ///< vertical offset for touch-device
        eC_Bool m_bScreenshotsEnabled;  ///< enable screenshots
        eC_Bool m_bInputDebugMode;      ///< enable debug-mode for input
        eC_Char* m_pkLicense;           ///< license-string
    };
}

#endif
