/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef GUIINPUTMEDIA__H_
#define GUIINPUTMEDIA__H_

#include "eC_Types.h"
#include <cassert>

/// Retrieve pointer to inputmedia instance
#define GETINPUTMEDIA        CGUIInputMedia::GetInstance()

class CGUI;
class CGUIObject;
class CGUIEvent;

/** Base-class for all input devices that are attachable to the GUI.
    @see @ref sec_porting_input "Porting User Input for Guiliani"
*/
class CGUIInputMedia
{
public:
    /** return pointer to inputmedia instance
    @return instance
    */
    static inline CGUIInputMedia& GetInstance() { assert(NULL != ms_pInputMedia); return *ms_pInputMedia; }

    /// Deletes this input media.
    static void DeleteInstance();

    /** Retrieves a platform-specific event and creates an appropriate Guiliani event.
        The supplied parameter specifies the time in milliseconds until the next processing of
        animation timers or commands is required. An input media may use this to avoid unnecessary
        event polling. It MUST return from GetEvent after the specified time, though.
        @param uiIdleTime specifies the maximum idle time in ms.
        @return Pointer to a newly generated Guiliani event, or NULL if nothing happened.
    */
    virtual CGUIEvent* GetEvent(eC_UInt uiIdleTime) = 0;

    /** Forces the input media to quit idling immediately.
        Called by the framework if the input media is requested to stop idling immediately.
        Classes derived from GUIInputMedia should implement StopIdle in such a way, that it interrupts any
        idling/sleeping code in GetEvent().
    */
    virtual void StopIdle() = 0;
    
    /** Activates debug-mode for input-device
        Will output messages for every input and can be used for calibrating input-device
        @param bDebugMode true for active debug-mode
    */
    void SetDebugMode(const eC_Bool& bDebugMode);
    
    /** Return if debug-mode for input-device is active
        @return true if active
    */
    eC_Bool GetDebugMode();

    /** set the size for the input-device
    @param iSizeX width
    @param iSizeY height
    */
    virtual void SetSize(const eC_Int& iSizeX, const eC_Int& iSizeY);

    /** get the size of the input-device
    @param riSizeX width
    @param riSizeY height
    */
    virtual void GetSize(eC_Int& riSizeX, eC_Int& riSizeY) const;

    /** set the offset for the input-device
    @param iOffsetX x-offset
    @param iOffsetY y-offset
    */
    virtual void SetOffset(const eC_Int& iOffsetX, const eC_Int& iOffsetY);

    /** get the offset of the input-device
    @param riOffsetX x-offset
    @param riOffsetY y-offset
    */
    virtual void GetOffset(eC_Int& riOffsetX, eC_Int& riOffsetY) const;

protected:
    CGUIInputMedia();

    virtual ~CGUIInputMedia();
    
    /// Pointer to input media
    static CGUIInputMedia       *ms_pInputMedia;
    
    eC_Bool m_bDebugMode;
    eC_Int m_iSizeX;
    eC_Int m_iSizeY;
    eC_Int m_iOffsetX;
    eC_Int m_iOffsetY;

private:

    /** Copy-constructor.
        Dummy implementation to avoid unintended use of compiler-generated default    */
    CGUIInputMedia(const CGUIInputMedia& kSource);

    /** Operator= method.
        Dummy implementation to avoid unintended use of compiler-generated default    */
    CGUIInputMedia& operator=(const CGUIInputMedia& kSource);
};

#endif
