/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef __FNT_WRAP_GLYPHLIB__H__
#define __FNT_WRAP_GLYPHLIB__H__

#include "eC_TArray.h"
#include "FntWrap.h"

#include "GUIImageData.h"

#include "gui_font.h"

/// GlyphLib (uGuiliani font) implementation of the Font Wrapper

/**
<h3>Platform Specific Limitations </h3>
*/
// @guiliani_doxygen limitations CFntWrapGlyphLib
/**
As the CFntWrapGlyphLib uses a pre-compiled ROMFS to store the glyph bitmaps, it is not possible to change the number of used fonts via SetNOFFonts()
*/
// @endguiliani_doxygen
/**
See @ref subsec_limitations_CFntWrapGlyphLib "CFntWrapGlyphLib Platform Specific Limitations"
*/

/** This class is used to wrap the funtionality of the GlyphLib
*/
class CFntWrapGlyphLib : public CFntWrap
{
public:
    /// entry for font
    struct FontEntry_s
    {
        gui_font_t Font; ///< font-id
        void* Memory; ///< memory
    };

public:
    void RequiredSpace(const eC_String * const pkText, eC_Value &vWidth, eC_Value &vHeight);

    void SetFont(const FontResource_t &eID);

    void LoadFont(const FontResource_t &eFontID);

    /** Unloading is not supported by GlyphLib
    @param eFontID FontID to unload
    */
    void UnloadFont(const FontResource_t &eFontID);

protected:
    CFntWrapGlyphLib();

    /** Render a text
    @param lpString text
    @param eOperation Use RENDER_TEXT to render the text or GET_REQUIRED_SPACE to calculate the required space.
    @param vX1 x-position
    @param vY1 y-position
    @param vWidth the width needed for the text
    @param vHeight the height needed for the text
    @param vWidthMax the maximum width available
    @param uiNumFittingChars maximum number of characters which will fit
    */
    virtual void RenderTextInternal(
        const eC_String *lpString,
        const TextOperation_t &eOperation,
        const eC_Value &vX1,
        const eC_Value &vY1,
        eC_Value &vWidth,
        eC_Value &vHeight,
        eC_Value vWidthMax,
        eC_UInt &uiNumFittingChars);

    /** Renders a glyph to the screen.
    This function takes the 8 bit alpha image provided by freetype adds color and blits it to the screen.
    @param vX1 The x coordinate where the glyph image is rendered.
    @param vY1 The y coordinate where the glyph image is rendered.
    @param kGlyphImage The glyph image containing a 8 bit alpha image of the glyph.
    */
    virtual void RenderGlyphToScreen(const eC_Value &vX1, const eC_Value &vY1, CGUIImageData& kGlyphImage) = 0;

    void Text(const eC_Value &vX1, const eC_Value &vY1, const eC_String * const lpString);

    virtual void GetGlyphMetrics(const eC_Char cChar, GlyphMetrics_t& kMetrics);

    virtual eC_Int GetAscender() const;
    virtual eC_Int GetDescender() const;
    virtual eC_Int GetInternalLeading() const;

    eC_Bool SetNOFFonts(const eC_UInt uiNOFFonts);

protected:
    eC_UInt m_uiCurrentFontIndex; ///< the currently used font-index
    eC_TArray<FontEntry_s> m_kFontsHandles; ///< handles of all loaded fonts

private:
    /** Copy-constructor. Should not be used.
    Dummy implementation to avoid unintended use of compiler-generated default
    */
    CFntWrapGlyphLib(const CFntWrapGlyphLib& kSource);

    /** Operator= method. Should not be used.
    Dummy implementation to avoid unintended use of compiler-generated default
    */
    CFntWrapGlyphLib& operator=(const CFntWrapGlyphLib& kSource);
};

#endif
