/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef DCWRAP__H_
#define DCWRAP__H_

#include "eC_Types.h"
#include "eC_TArray.h"

#include "GUILayerConfig.h"

#define GETDC CDCWrap::GetInstance()

class CGfxWrap;
class CGUIObject;

/**
    @brief Base-class for abstracting the use of display-controllers.

    Display-Controllers usually give access to certain specific functionality working on layers.
    By using layers content of different sources (e.g. videos or camera-input) can be displayed alongside with Guiliani-content.

    Each layer has its own portion of memory and attributes and will be processed seperately.

    By using layers complex GUI can be created containing multiple camera-inputs and Guiliani-controls.

    The normal feature-set of a display-controller covers:
    - moving/resizing of layers
    - alpha-blending
    - chroma-keying

    Note: on some systems alpha-blending and chroma-keying might share the same mechanism and
    might have a priorization which effect will be applied to the layer.
    please refer to the manual of the specific system you are using.

    This class is used to unify the access to each supported Display-Controller by using a single interface.

    All platform-specific processing will be done in the platform-specific implementation.

    To use different layers in Guiliani-projects, see CGUILayerContainer.

    @ingroup GUILIANI_GRAPHICS
*/
class CDCWrap
{
public:
    /** Create an instance
    @param uiScreenWidth width of screen
    @param uiScreenHeight height of screen
    @return true if successful
    */
    static eC_Bool CreateInstance(const eC_UInt& uiScreenWidth, const eC_UInt& uiScreenHeight);

    /// called to destroy and detach DC-Wrapper
    static void DeleteInstance();

    /**
    Initialize display
    @return 1 on success, else 0
    */
    virtual eC_UInt InitDisplay() { return 1; }

    /**
    De-Initialize display
    */
    virtual void DeInitDisplay() {}

    /**
    Get the internal index for given layer-id
    @param uiLayerID id of requested layer
    @param ruiLayerIndex internal index
    @return true if layer exists, otherwise false
    */
    eC_Bool GetLayerIndexForID(const eC_UInt& uiLayerID, eC_UInt& ruiLayerIndex) const;

    /**
    Set the given layer-id as the active
    all following access for rendering will be sent to the layer
    @param uiLayerID id of layer
    @param bMarkAsModified layer will be marked as modified upon switch (default: false)
    @return true if layer exists, otherwise false
    */
    eC_Bool SetActiveLayerID(const eC_UInt& uiLayerID, const eC_Bool& bMarkAsModified = false);

    /**
    Return the id of the currently active layer
    @return id of active layer
    */
    eC_UInt GetActiveLayerID() const;

    /**
    Return if layer was recently modified
    @param uiLayerID id of layer
    @return True if modified
    */
    eC_Bool IsLayerModified(const eC_UInt& uiLayerID) const;

    /**
    Refresh layer
    @param uiLayerID id of layer
    @param pkRenderBuffer currently used buffer for rendering
    @param pkDisplayBuffer currently used buffer for display
    @param uiWidth width of layer
    @param uiHeight height of layer
    @return true if successful
    */
    virtual eC_Bool Refresh(const eC_UInt& uiLayerID, void* pkRenderBuffer, void* pkDisplayBuffer, const eC_UInt& uiWidth, const eC_UInt& uiHeight);

    /**
    Add layer to internal configuration. should only be called during startup of application
    @param kLayerConfig layer-configuration
    */
    void AddLayer(const LayerConfiguration_t& kLayerConfig);

    /**
    Initialize layer
    this can be used to execute platform-specific code
    is called when layer gets added to DCWrap

    @param kLayerConfig layer-configuration
    */
    virtual void InitLayer(const LayerConfiguration_t& kLayerConfig) {}

    /**
    Return number of configured layers
    @return number of layers
    */
    eC_UInt GetNumberOfLayers() const;

    /**
    Get configuration for given layer. Note: this will need an index instead of id
    @param uiLayerIndex index of layer in internal data
    @param rkLayerConfig configuation of layer
    @return true if config was found
    */
    eC_Bool GetLayerConfig(const eC_UInt& uiLayerIndex, LayerConfiguration_t& rkLayerConfig) const;

    /**
    Get configuration for layer with given ID
    @param uiLayerID id of layer
    @param rkLayerConfig configuation of layer
    @return true if config was found
    */
    eC_Bool GetLayerConfigForID(const eC_UInt& uiLayerID, LayerConfiguration_t& rkLayerConfig) const;

    /**
    Get configuration for main-layer
    @param rkLayerConfig configuation of layer
    @return true if config was found
    */
    eC_Bool GetLayerConfigForMainLayer(LayerConfiguration_t& rkLayerConfig) const;

    /**
    Set the layer to visible
    @param uiLayerID id of layer
    @param bVisible visibility of layer
    */
    void SetLayerVisible(const eC_UInt& uiLayerID, const eC_Bool& bVisible);

    /**
    Get current visibility of layer
    @param uiLayerID id of layer
    @return true if visible, else false
    */
    eC_Bool IsLayerVisible(const eC_UInt& uiLayerID) const;

    /**
    Set the position for the given layer
    @param uiLayerID id of layer
    @param iXPos x-position
    @param iYPos y-position
    */
    void SetLayerPosition(const eC_UInt& uiLayerID, const eC_Int& iXPos, const eC_Int& iYPos);

    /**
    Get the current position for the given layer
    @param uiLayerID id of layer
    @param iXPos x-position
    @param iYPos y-position
    */
    void GetLayerPosition(const eC_UInt& uiLayerID, eC_Int& iXPos, eC_Int& iYPos);

    /**
    Set the visible size for the given layer
    @param uiLayerID id of layer
    @param uiWidth visible width
    @param uiHeight visible height
    */
    void SetLayerVisibleSize(const eC_UInt& uiLayerID, const eC_UInt& uiWidth, const eC_UInt& uiHeight);

    /**
    Get the visible size for the given layer
    @param uiLayerID id of layer
    @param uiWidth visible width
    @param uiHeight visible height
    */
    void GetLayerVisibleSize(const eC_UInt& uiLayerID, eC_UInt& uiWidth, eC_UInt& uiHeight);

    /**
    Set the visible position for the given layer
    @param uiLayerID id of layer
    @param iXPos x-position
    @param iYPos y-position
    */
    void SetLayerVisiblePos(const eC_UInt& uiLayerID, const eC_Int& iXPos, const eC_Int& iYPos);

    /**
    Set the alpha-value for blending between layer for the given layer.
    @param uiLayerID id of layer
    @param ubAlpha alpha-value
    */
    void SetLayerAlpha(const eC_UInt& uiLayerID, const eC_UByte& ubAlpha);

    /**
    Get the current alpha-value for the given layer.
    @param uiLayerID id of layer
    @return alpha-value
    */
    eC_UByte GetLayerAlpha(const eC_UInt& uiLayerID) const;

    /**
    Set the chroma-key for blending between layer for the given layer.
    @param uiLayerID id of layer
    @param bChromaKeyActive true if chroma-keying is active
    @param uiChromaKey the color for chroma-keying
    @param ubAlpha alpha-value for replacement
    */
    void SetLayerChromaKey(const eC_UInt& uiLayerID, const eC_Bool& bChromaKeyActive, const eC_UInt& uiChromaKey, const eC_UByte& ubAlpha);

    /**
    Get the current alpha-value for the given layer.
    @param uiLayerID id of layer
    @return chroma-key
    */
    eC_UByte GetLayerChromaKey(const eC_UInt& uiLayerID) const;

    /**
    Get the size of the buffer attached to the given layer.
    @param uiLayerID id of layer
    @param uiWidth width of layer
    @param uiHeight height of layer
    @param ubBPP bytes per pixel
    */
    void GetLayerBufferSize(const eC_UInt& uiLayerID, eC_UInt& uiWidth, eC_UInt& uiHeight, eC_UByte& ubBPP);

    /**
    Update layer according to reason.
    This will be called when attributes of the layer (position, size, etc) have changed
    @param kLayerConfig configuration of updated layer
    @param eUpdateReason reason of update
    */
    virtual void UpdateLayer(const LayerConfiguration_t& kLayerConfig, const LayerUpdate_t& eUpdateReason) {}

    /**
    Get the currently active buffer which is used for rendering
    @param uiLayerID id of layer
    @return address of buffer or NULL if layer was not found
    */
    void* GetRenderBuffer(const eC_UInt& uiLayerID) const;

    /**
    Get the currently active buffer which is used for display
    @param uiLayerID id of layer
    @return address of buffer or NULL if layer was not found
    */
    void* GetDisplayBuffer(const eC_UInt& uiLayerID) const;

    /**
    Refresh layer after content has been changed. This applies merely to layers containing user-defined content
    @param uiLayerID id of layer to refresh
    */
    void InvalidateLayer(const eC_UInt& uiLayerID);

    /**
    This will update the positions for all layers, BEFORE the redraw is done.
    */
    void UpdateLayerPositions();

    /**
    This will link an object to a specific layer. this info is used in UpdateLayerPositions
    @param uiLayerID id of layer
    @param pkObject pointer to object
    */
    void SetObjectForLayerID(const eC_UInt& uiLayerID, CGUIObject* pkObject);

    /**
    This will be called when all layers have been refreshed.
    To wait for V-sync this would be the right place.
    */
    virtual void EndOfRefresh() {}

    /**
    This will set the id of the main-layer
    @param uiLayerID id of layer
    */
    void SetMainLayer(const eC_UInt& uiLayerID);

    /**
    Return if layer is set to be used as main-layer
    @param uiLayerID id of layer
    @return True if main-layer
    */
    eC_Bool IsMainLayer(const eC_UInt& uiLayerID) const;

    /** Check if Wrapper is NULL-implementation
    @return true if NULL
    */
    virtual eC_Bool IsNull() { return false; }

    /// Destructor
    virtual ~CDCWrap(void);

    /** return graphics wrapper instance
    @return Graphics wrapper instance
    */
    static CDCWrap& GetInstance();

    /** Set the GfxWrapper
    @param pkGfxWrap
    */
    void SetGfxWrap(CGfxWrap* pkGfxWrap) { m_pkGfxWrap = pkGfxWrap; }

    /** Return the number of bytes per pixel for pixel-format
    @param uiPixFormat
    @return bpp
    */
    eC_UByte GetBPP(const eC_UInt& uiPixFormat) const;

    /** Set a device
    @param pkDevice
    */
    virtual void SetDevice(void* pkDevice) {}

    /** Return merge-buffer
    @return NULL by default
    */
    virtual void* GetMergeBuffer() const { return NULL; }

protected:
    /** create DC-Wrapper with specified size
    @param uiScreenWidth
    @param uiScreenHeight
    */
    CDCWrap(const eC_UInt& uiScreenWidth = 0, const eC_UInt& uiScreenHeight = 0);

    /** Copy-constructor.
        @param kSource Source object to be copied.
    */
    CDCWrap(const CDCWrap& kSource);

    /** Operator= method.
        @param kSource Source object to be copied.
        @return This instance.
    */
    CDCWrap& operator=(const CDCWrap& kSource);

protected:
    const eC_UInt m_uiScreenWidth; ///< width of screen
    const eC_UInt m_uiScreenHeight; ///< height of screen

    eC_TArray<LayerConfiguration_t> m_akLayers; ///< : *DCLayer_t, one DCLayer_t struct per layer storing general settings
    eC_UInt m_uiActiveLayerID; ///< : unsigned int = Active Layer
    eC_UInt m_uiActiveLayerIndex; ///< index of currentl< active layer

    static CDCWrap* ms_pkDCWrap; ///< instance of wrapper

    CGfxWrap* m_pkGfxWrap; ///< pointer to GfxWrapper

    eC_Int m_iMainLayerID; ///< id of main-layer

private:
    void Init() {}

    void DeInit() {}
};

#endif
