/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef GFXWRAPEGML__H_
#define GFXWRAPEGML__H_

#include "GfxWrap.h"
#include "eGML_Config.h"
#include "eGML_Platform.h"

class eGML;
class eImg_Bitmap;
class eGML_Screen;
class eGML_Bitmap;
class CDCWrap;

///  This is the eGML implementation of the graphics wrapper.

/** This graphics wrapper uses eGML_Bitmap methods for blitting. The blit flags
for all operations can be set by calling SetBlitFlags(). They are by default
initialized to eGML_Bitmap::BLIT_BILINEAR.

For additional details see description of base class CGfxWrap.
*/
class CGfxWrapeGML : public CGfxWrap
{
public:
    /** Singleton for creating a GfxWrapeGML object and change the pointer in
    base class to this graphics wrapper.
    @param uiScreenWidth The eGML screen width.
    @param uiScreenHeight The eGML screen height.
    @param eBitmapType bitmap type to create
    @param uiBufferCount number of buffers
    @param ppBuffers pointer to buffers to use
    @return Indicate whether by this call, an instance is created.
    The instance is created, if true, not created, otherwise.
    */
    static eC_Bool CreateInstance(
        const eC_UInt &uiScreenWidth = eGML_DEFAULT_SCREEN_WIDTH,
        const eC_UInt &uiScreenHeight = eGML_DEFAULT_SCREEN_HEIGHT,
        eC_Int eBitmapType = eGML_DEFAULT_BITMAP_TYPE,
        eC_UInt uiBufferCount = eGML_DEFAULT_BUFFERS,
        void** ppBuffers = NULL);

    /** Singleton for creating a GfxWrapeGML object and change the pointer in
    base class to this graphics wrapper.
    @param pkDCWrap display-controller-wrapper to be used (default: NULL)
    @return Indicate whether by this call, an instance is created.
    The instance is created, if true, not created, otherwise.
    */
    static eC_Bool CreateInstance(CDCWrap* pkDCWrap = NULL);

    virtual eC_Value SetLineWidth(const eC_Value &vWidth);

    virtual eC_Bool Refresh(const RefreshCall_t eRefreshCallType);

    /** Create a screen to be used with eGML-GfxWrap
    @param uiWidth width of screen
    @param uiHeight height of screen
    */
    void CreateScreen(
        const eC_UInt &uiWidth = eGML_DEFAULT_SCREEN_WIDTH,
        const eC_UInt &uiHeight = eGML_DEFAULT_SCREEN_HEIGHT)
    {
        CreateScreen(uiWidth, uiHeight, eGML_DEFAULT_BITMAP_TYPE, eGML_DEFAULT_BUFFERS);
    }

    /** create a screen with given height and attributes
    @param uiWidth screen-width
    @param uiHeight screen-height
    @param eBitmapType bitmap-type for screen
    @param uiBufferCount number of buffers to use
    @param ppBuffers buffers to use
    */
    void CreateScreen(
        const eC_UInt &uiWidth = eGML_DEFAULT_SCREEN_WIDTH,
        const eC_UInt &uiHeight = eGML_DEFAULT_SCREEN_HEIGHT,
        eC_Int eBitmapType = eGML_DEFAULT_BITMAP_TYPE,
        eC_UInt uiBufferCount = eGML_DEFAULT_BUFFERS,
        void** ppBuffers = NULL);

    /** create a screen with given height and attributes
    @param pkDCWrap display-controller-wrapper to be used (default: NULL)
    */
    virtual void CreateScreen(CDCWrap* pkDCWrap = NULL);

    virtual void DestroyScreen();

    ///@return the eGML screen pointer
    eGML_Screen* GetScreen(void);

    ///@return the eGML render bitmap pointer
    eGML_Bitmap* GetRenderBitmap();

    ///@param pRenderBitmap The target bitmap for subsequent render operations
    void SetRenderBitmap(eGML_Bitmap* pRenderBitmap);

    ///@param uiLayerID Switch to buffer for layer for subsequent render operations
    void SetTargetLayer(const eC_UInt& uiLayerID);

    virtual eC_UInt GetVirtualScreenWidth();
    virtual eC_UInt GetVirtualScreenHeight();
    virtual eC_UInt GetPhysicalScreenWidth();
    virtual eC_UInt GetPhysicalScreenHeight();

    virtual void SetForegroundColorImpl(
        const eC_UByte &ubRed,
        const eC_UByte &ubGreen,
        const eC_UByte &ubBlue,
        const eC_UByte &ubAlpha = 255);

    virtual void Line(const eC_Value &vX1, const eC_Value &vY1, const eC_Value &vX2, const eC_Value &vY2);

    virtual void Rect(const eC_Value &vX1, const eC_Value &vY1, const eC_Value &vX2, const eC_Value &vY2);

    virtual void FilledRect(const eC_Value &vX1, const eC_Value &vY1, const eC_Value &vX2, const eC_Value &vY2);

    /** Implementation of the CGfxWrap::Ellipse() interface for eGML.
    Note that eGML does currently only support filled circles (e.g. width and height must be identical), and
    the active foreground color must be fully opaque.
    @param vAbsX1 X-coordinate of bounding rectangle's top left corner
    @param vAbsY1 Y-Coordinate of bounding rectangle's top left corner
    @param vAbsX2 X-coordinate of bounding rectangle's bottom right corner
    @param vAbsY2 Y-Coordinate of bounding rectangle's bottom right corner
    @param bFilled Specifies whether the ellipse will be drawn outlined (false) or filled (true).
    */
    virtual void Ellipse(const eC_Value &vAbsX1, const eC_Value &vAbsY1, const eC_Value &vAbsX2, const eC_Value &vAbsY2, const eC_Bool &bFilled);

    virtual void Arc(
        const eC_Value &vX1,
        const eC_Value &vY1,
        const eC_Value &vX2,
        const eC_Value &vY2,
        const eC_Value& vStartAngle,
        const eC_Value& vEndAngle,
        const eC_Value& vRotationAngle,
        const eC_Bool &bFilled);

    virtual void Ring(
        const eC_Value& vX1,
        const eC_Value& vY1,
        const eC_Value& vX2,
        const eC_Value& vY2,
        const eC_Value& vRingWidth,
        const eC_Value& vStartAngle,
        const eC_Value& vEndAngle,
        const eC_Value& vRotationAngle,
        const eC_Bool & bFilled);

    virtual void SetCliprect(const CGUIRect &NewAbsClipRect);

    /** Returns a pointer to the image referenced by the specified ID.
    @param eID of the image to retrieve
    @return eImb_Bitmap pointer to the related bitmap   */
    virtual eImg_Bitmap* GetImage(const ImageResource_t &eID);

    /** Returns whether an image is present for the given ID.
    @param eID ID for which to check
    @return True if an image is present, False otherwise. */
    virtual eC_Bool ImageExists(const ImageResource_t eID) const;

    /** Enables/disables subpixel rendering.  (Disabled by default!)
    While subpixel rendering is enabled, blitting will be done with subpixel-accuracy (Resulting in a time-consuming RotBlit)
    @param bEnabled true to enable subpixel rendering, false to disable. */
    void SetSubpixelEnabled(const eC_Bool bEnabled);

    /** Sets the eGML blit flags for all blit operations.
    @param uiFlags Combination of eGML blit flags.
    @see GetBlitFlags()
    */
    void SetBlitFlags(eC_Flag uiFlags);

    /** Returns the currently set eGML blit flags.
    @return The current combination of eGML blit flags.
    @see SetBlitFlags()
    */
    eC_Flag GetBlitFlags() const;

    /** Enables/Disables NEON optimization
    if neon is available this setting is true by default
    @param bOptimizedBlit true to enable NEON optimization if available, false to disable
    */
    virtual void SetOptimizedBlit(const eC_Bool& bOptimizedBlit);

    /** Enables/Disables bilinear filtering during blit operation
    if bilinear is available this setting is true by default
    @param bFilteredBlit true to enable bilinear blit, false to disable bilinear filtering and use nearest neighbor texturing
    */
    virtual void SetFilteredBlit(const eC_Bool& bFilteredBlit);

    virtual eC_Bool DrawToScreen();

    virtual eC_UInt GetPixelFormat(const LayerPixFormat_t& ePixFormat) const;

protected:
    /** Sets the total number of images. This is called internally by the ResourceManager when dynamically
    allocated image resources require the total number of images to be increased.
    @param uiNOFImages The number of images
    @return True if internal storage has been successfully adapted to support the requested
    number of images. False otherwise.
    */
    virtual eC_Bool SetNOFImagesImpl(const eC_UInt uiNOFImages);

    /** Loads an image file from disk. Internally called by ResourceManager.
    @param psPath filename of image to load
    @param eID image-id which will be set
    */
    virtual void LoadImgImpl(const eC_String &psPath, const ImageResource_t &eID);

    /** Loads an image file from disk. Internally called by ResourceManager.
    @param pkMemory memory-location
    @param uiWidth
    @param uiHeight
    @param eImageType
    @param eID image-id which will be set
    */
    virtual void LoadImgImpl(
        void* pkMemory,
        const eC_UInt& uiWidth,
        const eC_UInt& uiHeight,
        const eC_UInt& eImageType,
        const ImageResource_t &eID);

    /** Unloads an image file. Internally called by ResourceManager.
    @param eID image-id to unload
    */
    virtual void UnloadImg(const ImageResource_t &eID);

    virtual void BlitImgExtImpl(
        const ImageResource_t &eID,
        const eC_UInt &uiSrcX, const eC_UInt &uiSrcY,
        const eC_UInt &uiSrcWidth, const eC_UInt &uiSrcHeight,
        const eC_Value &vDestX, const eC_Value &vDestY,
        const eC_Value &vDestWidth, const eC_Value &vDestHeight,
        const eC_UByte &ubAlpha = 255,
        const eC_Value &vAngle = eC_FromFloat(0),
        const eC_Value &vRotCenterX = eC_FromFloat(0.5), const eC_Value &vRotCenterY = eC_FromFloat(0.5));

    virtual void BlitImgExtImpl(const CGUIImageData& kImageData, const eC_Value& vAbsDestX, const eC_Value& vAbsDestY);

    /** Blit the contents or part of content of a memory-image to the screen
    @param pkMemory memory containing image-data
    @param uiAbsSrcX x-position in source
    @param uiAbsSrcY y-position in source
    @param uiSrcWidth width of area in source
    @param uiSrcHeight height of area in source
    @param vAbsDestX x-position in destination
    @param vAbsDestY y-position in destination
    @param vDestWidth width of destination area
    @param vDestHeight height of destination area
    @param ubAlpha alpha
    @param vAngle angle of rotation
    @param vRotCenterX x-position of center of rotation
    @param vRotCenterY y-position of center of rotation
    */
    virtual void BlitMemoryImpl(
        void* pkMemory,
        const eC_UInt &uiAbsSrcX, const eC_UInt &uiAbsSrcY, const eC_UInt &uiSrcWidth, const eC_UInt &uiSrcHeight,
        const eC_Value &vAbsDestX, const eC_Value &vAbsDestY, const eC_Value &vDestWidth, const eC_Value &vDestHeight,
        const eC_UByte &ubAlpha = 255,
        const eC_Value &vAngle = eC_FromFloat(0),
        const eC_Value &vRotCenterX = eC_FromFloat(0.5), const eC_Value &vRotCenterY = eC_FromFloat(0.5));

    virtual eC_UInt GetImgWidthImpl(const ImageResource_t &eID) const;
    virtual eC_UInt GetImgHeightImpl(const ImageResource_t &eID) const;

    /** Calculates the size of an image in pixels. Called by the image cache.
    @param eID id of image
    @return size of image in pixels
    */
    virtual eC_UInt GetImageSize(const ImageResource_t& eID) const;

    /** Sets the default 2D view used by the gfx wrapper for 2D drawing operations, like FilledRect, Line, Blit, etc.
    */
    void SetDefault2DView();

    /** Initialize the mail-layer
    @param uiWidth width of layer
    @param uiHeight height of layer
    */
    virtual void InitMainLayer(
        const eC_UInt& uiWidth,
        const eC_UInt& uiHeight);

    CGfxWrapeGML(void);
    virtual ~CGfxWrapeGML(void);

protected:
    /** Screen-configuration to be used in GfxWrapper
    */
    typedef struct
    {
        eC_UInt uiID;           ///< id of screen
        eC_UInt uiScreenCount;  ///< number of screens
        eGML_Screen* pkScreen;  ///< pointer to screens
    } ScreenConfig_t;

    eGML* m_pEGML; ///< eGML instance pointer
    eGML_Bitmap* m_pkRenderBitmap;  ///< Target bitmap for render operations in eGML
    ScreenConfig_t* m_pkScreens; ///< Screens for eGML

    eC_UInt m_uiNumberOfScreens; ///< number of configured screens
    eC_Int m_iMainLayerIndex; ///< index of the main-screen

    /// while subpixel rendering is enabled, blitting will be done with subpixel-accuracy (Resulting in a time-consuming RotBlit)
    eC_Bool m_bSubpixelEnabled;

    /// Array of pointers to eImg bitmaps (includes alpha layer)
    eC_TArray <eImg_Bitmap*> m_pImage;

    eC_Flag m_uiBlitFlags;  ///<Flags for blit operation

private:
    virtual CGUIBitmapPlane* CreateGfxDependentBitmap(
        GUIBitmapPlaneID_t uiNewBitmapID,
        const CGUIObject& rkParentObject,
        const eC_Value &vWidth,
        const eC_Value &vHeight);

    eC_Bool Init(void);
    eC_Bool DeInit(void);

    /** Copy-constructor intentionally not implemented.
    */
    CGfxWrapeGML(const CGfxWrapeGML& kSource);

    /** Operator= method intentionally not implemented.
    */
    CGfxWrapeGML& operator=(const CGfxWrapeGML& kSource);
};

#endif
