/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

/** @brief Contains declarations for functions that are called during start-up
           for obtaining custom extensions.
  */
#if !defined(CUSTOM_EXTENSION_FUNCS_H)
#define CUSTOM_EXTENSION_FUNCS_H

#include "CustomExtensionFactory.h"

#include <vector>
#include "GUICommand.h"

/** Describes an enumeration value so that it can be mapped to readable text in
    the editor's GUI. */
struct EnumMapping
{
    /** Constructs an enumeration mapping descriptor.
        @param rkTypeTag The XML tag used when reading from or writing the
               enumeration value to a stream. In other words, this must be the
               same string that is used in the respective control's/command's/
               behaviour's/layouter's WriteToStream and ReadFromStream
               implementations.
        @param iEnumValue The enumeration value.
        @param rkValueName The string representation to be used for iEnumValue.
      */
    EnumMapping(
        const eC_String& rkTypeTag,
        eC_Int iEnumValue,
        const eC_String& rkValueName) :
        m_kTypeTag(rkTypeTag),
        m_iEnumValue(iEnumValue),
        m_kValueName(rkValueName)
    {}

    /// The XML tag used when reading from or writing the enumeration value to a stream.
    eC_String m_kTypeTag;

    /// The enumeration value.
    eC_Int m_iEnumValue;

    /// The enumeration value as a string.
    eC_String m_kValueName;
};

/// Describes a custom attribute repetition.
struct AttributeRepetitionDescriptor
{
    /// The possible types of attributes.
    enum AttributeType // Copy from class ObjectAttribute::AttributeType
    {
        AT_INT,     ///< Integer value.
        AT_HEX,     ///< Integer value, formatted in hex.
        AT_SHORT,   ///< Short integer value.
        AT_FLOAT,   ///< Float value.
        AT_BOOL,    ///< Boolean value.
        AT_STRING,  ///< String.
        AT_OBJECTID,///< Object handle (ID).
        AT_IMAGEID, ///< Image ID.
        AT_TEXTID,  ///< Text ID.
        AT_FONTID,  ///< Font ID.
        AT_SOUNDID, ///< Sound ID.
        AT_GENRESID, ///< General resource ID.
        AT_DATAPOOLID, ///< DataPool resource ID.
        AT_COMMENT, ///< Comment (for XML comment tags without values).
    };

    /** Constructs a repetition descriptor.
        @param rkRepetitionTitle The title of the repetition.
        @param rkXMLTag The XML tag of the repetition.
        @param rkAttributeValue The attribute value of the repetition as string.
        @param kAttributeType The attribute type of the repetition as enum.
      */
    AttributeRepetitionDescriptor(
        const eC_String &rkRepetitionTitle,
        const eC_String &rkXMLTag,
        const eC_String &rkAttributeValue,
        AttributeRepetitionDescriptor::AttributeType kAttributeType) :
        m_kRepetitionTitle(rkRepetitionTitle),
        m_kXMLTag(rkXMLTag),
        m_kAttributeValue(rkAttributeValue),
        m_kAttributeType(kAttributeType)
    {}

    /// The repetition title as a string (The attribute name of the tag containing the number of repetitions).
    /// Marks the end of a repetition lists (XMLTag and AttributeValue have to be empty strings).
    eC_String m_kRepetitionTitle;

    /// The repetition XML tag as a string.
    eC_String m_kXMLTag;

    /// The repetition attribute value as a string.
    eC_String m_kAttributeValue;

    /// The repetition attribute type enum.
    AttributeType m_kAttributeType;
};

typedef std::vector<EnumMapping> EnumMappingVector;
typedef std::vector<AttributeRepetitionDescriptor> RepetitionVector;

/** Called on start-up to allow adding of custom enum mappings.
    @param rkEnumMappings Receives the custom enum mappings.
  */
void GetCustomEnumMappings(EnumMappingVector& rkEnumMappings);

/** Called on start-up to allow adding of custom attribute repetition mappings.
    @param rkRepetitions Receives the custom repetition mappings.
  */
void GetCustomRepetitions(RepetitionVector& rkRepetitions);

#endif // CUSTOM_EXTENSION_FUNCS_H
