/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef GUIVIDEODECODER__H_
#define GUIVIDEODECODER__H_

#include "eC_Types.h"
#include "eC_File.h"
#include "eC_TList_doubleLinked.h"
#include "eC_String.h"

#include "GUICommonEnums.h"
#include "GUIVideoData.h"

/// Base class for the video-decoders.

/** All decoder classes must be derived from this class. */

class CGUIVideoDecoder
{
public:
    /** Virtual destructor. */
    virtual ~CGUIVideoDecoder()
    {
    }

    /** Must be implemented in each derived class to decode an image.
        @param kImageData The decoded image has to be stored here by each
               derived implementation.
        @param pTmpImgFile Pointer to image file to load.
        @return Implementations must return True if the image was decoded
                successfully, False if anything went wrong and the contents of
                kImageData are invalid.
    */
    virtual eC_Bool Decode(CGUIVideoData &kImageData, eC_File *pTmpImgFile) = 0;

    /** Decode a single frame
    @param pkData pointer to data
    @param uiLength length of frame
    @param pkDestination destination-address
    */
    virtual void DecodeFrame(const eC_UByte* const pkData, const eC_UInt& uiLength, eC_UByte* pkDestination) = 0;

    /** Set bytes per pixel
    @param ubBPP
    */
    virtual void SetBPP(const eC_UByte& ubBPP) = 0;

    /** Set destination color-format. This will be used
    to convert the video data to the appropriate format.
    @param eDestinationFormat
    */
    virtual void SetDestinationFormat(const CGUICommonEnums::ImageType_t& eDestinationFormat) = 0;

protected:
    /** Constructor of CGUIImageDecoder, which sets all supported image types to False. */
    CGUIVideoDecoder()
    {
    }

    /** Swaps 16-bit endianess of the referenced value.
    @param[in,out] ubData Given value.
    */
    inline void SwapEndianess16(eC_UShort &ubData)
    {
        eC_UShort usTmp = ubData;
        ubData= ((usTmp&0x00FF)<<8)|((usTmp&0xFF00)>>8);
    }

    /** Swaps 32-bit endianess of the referenced value.
    @param[in,out] uiData Given value.
    */
    inline void SwapEndianess32(eC_UInt &uiData)
    {
        eC_UInt uiTmp = uiData;
        uiData = (((uiTmp&0x000000FF)<<24)|((uiTmp&0x0000FF00)<<8)|((uiTmp&0x00FF0000)>>8)|((uiTmp&0xFF000000)>>24));
    }
};

#endif
