/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef GUIVIDEODATA__H_
#define GUIVIDEODATA__H_

#include "eC_Types.h"

/// Contains information about a video loaded by CGUIVideoDecoder.
/** Typical use case for this class is that a graphics wrapper creates an
    object of this class, sets some attributes to make sure the image uses the
    correct format etc., then passes it to the image loader which hands it over
    to the individual image decoders that fill the data section with the
    decoded image data.
  */
class CGUIVideoData
{
public:
    /** CGUIImageData constructor */
    CGUIVideoData();

    /** CGUIImageData destructor */
    ~CGUIVideoData();

    /** Sets the width of the image. Should only be called by an image decoder.
        @param uiWidth Width of the image.
      */
    void SetWidth(const eC_UInt& uiWidth);

    /** Sets the height of the image. Should only be called by an image decoder.
        @param uiHeight Height of the image.
      */
    void SetHeight(const eC_UInt& uiHeight);

    /** Sets the bits per pixel of the image. Should only be called by an image decoder.
        @param ubBPP Height of the image.
      */
    void SetBytesPerPixel(const eC_UByte& ubBPP);

    /** Sets the data pointer of the image. Should only be called by an image
        decoder.
        @param *pubData Pointer to memory to be used for storing a decoded image.
      */
    void SetData(eC_UByte *pubData);

    /** Sets length of data
    @param uiLength
    */
    void SetDataLength(const eC_UInt& uiLength);

    /** Sets the number of frames
    @param uiNumberOfFrames
    */
    void SetNumberOfFrames(const eC_UInt& uiNumberOfFrames);

    /** Sets the number of microseconds per frame
    @param uiMicroSecondsPerFrame
    */
    void SetMicroSecondsPerFrame(const eC_UInt& uiMicroSecondsPerFrame);

    /** Sets the frame-offset
    @param uiIndex index of frame
    @param uiOffset offset from beginning
    @param uiSize size in bytes
    */
    void SetFrameOffset(const eC_UInt& uiIndex, const eC_UInt& uiOffset, const eC_UInt& uiSize);

    /** Sets the index of the current frame
    @param uiFrame
    */
    void SetCurrentFrame(const eC_UInt& uiFrame);

    /** Sets the destructor's behaviour.
        @param bDeleteDataOnDestruction Pass true to have the destructor delete
               the decoded image data.
      */
    void SetDeleteDataOnDestruction(const eC_Bool bDeleteDataOnDestruction);

    /// @return The width of the image.
    eC_UInt GetWidth() const;

    /// @return The height of the image.
    eC_UInt GetHeight() const;

    /// @return Pointer to the decoded image data.
    eC_UByte* GetData() const;

    /// @return The bit depth of the image.
    eC_UByte GetBytePerPixel() const;

    /// @return True if this image data deletes its contents upon destruction.
    eC_Bool GetDeleteDataOnDestruction() const;

    /** Returns the size (in bytes) of one frame.
        Width, height and bits per pixel must have been set correctly before.
        @return Frame size in bytes.
    */
    eC_UInt GetFrameSize() const;

    /** Returns the length of data
    @return length of data
    */
    eC_UInt GetDataLength() const;

    /** Return number of frames
    @return number of frames
    */
    eC_UInt GetNumberOfFrames() const;

    /** Return the number of microseconds per frame
    @return number of microseconds
    */
    eC_UInt GetMicroSecondsPerFrame() const;

    /** Return the index of the current frame
    @return index of current frame
    */
    eC_UInt GetCurrentFrame() const;

    /** Returns offset and size of current frame
    @param Offset
    @param Size
    */
    void GetFrameData(eC_UInt& Offset, eC_UInt& Size) const;

    /** Deletes the image data array. */
    void DeleteData();

    /** Operator= method. This does a flat copy of the given source CGUIImageData object, which means that both
        GUIImageData objects will then reference the SAME DATA in memory.
        @param kSource Source object to be copied.
        @return This instance.
    */
    CGUIVideoData& operator=(const CGUIVideoData& kSource);

private:
    struct Frames_t
    {
        eC_UInt Offset;
        eC_UInt Size;
    };

private:
    /** Copy-constructor. Do not use.
        Dummy declaration without implementation, just to hide it.
        @param kSource Source object to be copied.
    */
    CGUIVideoData(const CGUIVideoData& kSource);

    eC_UInt m_uiWidth;
    eC_UInt m_uiHeight;
    eC_UByte m_ubBPP;

    eC_UInt m_uiDataLength;
    eC_UInt m_uiNumberOfFrames;
    eC_UInt m_uiMicroSecondsPerFrame;

    eC_UInt m_uiCurrentFrame;
    Frames_t* m_pkFrameOffsets;

    eC_UByte* m_pubData;
    eC_Bool m_bDeleteData;
};

#endif
