// File: Bidi.h
//
/*
Adapted reference implementation of the Bidi Algorithm from ftp://ftp.unicode.org/Public/PROGRAMS/BidiReferenceCpp/v26/
The reference implementation is distributed under the Terms of Use in http://www.unicode.org/copyright.html.

*/

#ifndef _BIDI_H_
#define _BIDI_H_

#include <eC_Types.h>
#include <eC_TList.h>
#include <eC_String.h>

#define BIDI_MAX_CCH 1024

// For convenience of external callers, the following constitute the interface to the actual algorithm. 
// For usage notes and paramter descriptions see the file bidi.cpp

void BidiLines(
    int baselevel, eC_UTF16* pszLine, int * pclsLine,
    int * plevelLine, int cchPara, int fMirror, bool * pbrk = 0);

void GetEmbeddingLevels(const eC_String& kSource, int *levels);

eC_Int BidiResolve(eC_UTF16* paPszInput, eC_Int iCch);

enum bidi_class
{
    ON = 0,
    L,
    R,
    AN,
    EN,
    AL,
    // Non-spacing Mark
    NSM,
    CS,
    ES,
    ET,

    BN,
    S,
    WS,
    B,
    RLO,
    RLE,
    LRO,
    LRE,
    PDF,
    N = ON,
};

int EmbeddingDirection(int level);

int GetEmbeddingDirection(int level);

/** right-embedding
*/
struct RightEmbedding
{
    eC_UInt m_uiStartRtL; ///< start
    eC_UInt m_uiEndRtL; ///< end
    RightEmbedding() {}; ///< default-constructor

    /** constructor
    @param uiStartRtL
    @param uiEndRtL
    */
    RightEmbedding(eC_UInt uiStartRtL, eC_UInt uiEndRtL)
    {
        m_uiStartRtL = uiStartRtL;
        m_uiEndRtL = uiEndRtL;
    }

    /** compare
    @param rtl
    @return equal
    */
    eC_Bool operator==(const RightEmbedding rtl) const
    {
        return ((this->m_uiStartRtL == rtl.m_uiStartRtL) &&
            (this->m_uiEndRtL == rtl.m_uiEndRtL));
    }
};

/** left-embedding
*/
struct LeftEmbedding
{
    eC_UInt m_uiStartLtR; ///< start
    eC_UInt m_uiEndLtR; ///< end
    LeftEmbedding() {}; ///< default-constructor

    /** constructor
    @param uiStartLtR
    @param uiEndLtR
    */
    LeftEmbedding(eC_UInt uiStartLtR, eC_UInt uiEndLtR)
    {
        m_uiStartLtR = uiStartLtR;
        m_uiEndLtR = uiEndLtR;
    }

    /** compare
    @param ltr
    @return equal
    */
    eC_Bool operator==(const LeftEmbedding ltr) const
    {
        return ((this->m_uiStartLtR == ltr.m_uiStartLtR) &&
            (this->m_uiEndLtR == ltr.m_uiEndLtR));
    }

    /** assign
    @param ltrSource
    @return new object
    */
    LeftEmbedding& operator=(const LeftEmbedding& ltrSource)
    {
        this->m_uiStartLtR = ltrSource.m_uiStartLtR;
        this->m_uiEndLtR = ltrSource.m_uiEndLtR;
        return *this;
    }
};

#endif //#ifndef _BIDI_H_
