/* Adapted implementation of BidiJoin algithm from https://github.com/xbmc/xbmc/ */

/*Original copyright info:*/
/*
*  Copyright (C) 2005-2008 Team XBMC
*  http://www.xbmc.org
*
*  A port of Mohammed Yousif's C Arabic shaping code
*  Ported by Nibras Al-shaiba
*
*  This Program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2, or (at your option)
*  any later version.
*
*  This Program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with XBMC; see the file COPYING.  If not, write to
*  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*  http://www.gnu.org/copyleft/gpl.html
*
*/

#ifndef __ARABIC_UTF_CONVERTER_H__
#define __ARABIC_UTF_CONVERTER_H__

#include "bidi.h"
#include "eC_String.h"

#define MAP_LENGTH 37
#define COMB_MAP_LENGTH 4
#define TRANS_CHARS_LENGTH 39

/**
* @brief This class is a helper to convert strings containing bi-directional text-parts.
*/
class ArabicUTFConverter
{
public:
    /** structure for replacement of single characters to a new glyph
    */
    typedef struct
    {
        eC_UTF16 code; ///< character-code
        eC_UTF16 mIsolated; ///< character-code for isolated use
        eC_UTF16 mInitial; ///< character-code for use as first character
        eC_UTF16 mMedial; ///< character-code for use as middle-character
        eC_UTF16 mFinal; ///< character-code for use as terminating character
    } CharRep_t;

    /** structure for replacement of combined characters to a new glyph
    */
    typedef struct
    {
        eC_UTF16 code[2]; ///< character-codes
        eC_UTF16 mIsolated; ///< character-code for isolated use
        eC_UTF16 mInitial; ///< character-code for use as first character
        eC_UTF16 mMedial; ///< character-code for use as middle-character
        eC_UTF16 mFinal; ///< character-code for use as terminating character
    } CombCharRep_t;

public:
    /** This method is used to convert an input string containing arabic unicode-characters
    into a new string with all characters replaced with their correct replacement according to their position in the text.
    @param rkSourceString the input-string
    @param bConverted set to true if conversion has been done
    @return the converted string
    */
    static eC_String ConvertArabic(const eC_String& rkSourceString, eC_Bool& bConverted);

    /** Get the visual representation of character
    @param c character
    @return visual representation
    */
    static CharRep_t GetCharRep(eC_UTF16 c);

    /** Get the visual representation of character-combination
    @param c character-combination
    @return visual representation
    */
    static CombCharRep_t GetcombCharRep(eC_UTF16 c);

    /** Get code for unicode character
    @param c character
    @return code
    */
    static eC_UTF16 GetCode(eC_UTF16 c);

    /** Get the text in utf16-encoding
    @param kSource source-string
    @param ruiLength length of string in bytes
    */
    static void GetTextInUTF16(const eC_String& kSource, eC_UInt& ruiLength);

    /** Get internal data of text
    @param rkSourceString source-string
    @param bGlobalCodePresentation if global is active
    @return internal text-data
    */
    static eC_String GetInternTextData(const eC_String& rkSourceString, const eC_Bool& bGlobalCodePresentation = true);

private:
    static eC_Int GetTextContextualForm(eC_UTF16* pcInputString, const eC_UInt& uiLength, eC_UTF16* pcOutputString, const eC_Bool& bGlobalCodePresentation = true);
    static eC_UTF16* GetCombCode(eC_UTF16 c);

    static eC_Int ShapeArabic(eC_UTF16* paInputString, const eC_UInt& uiLength, eC_UTF16* paOutputString, eC_Bool& bConverted);

    static eC_Int CombineArabic(eC_UTF16* paInputString, const eC_UInt& uiLength, eC_UInt* cursor, eC_UTF16* paOutputString);

    static eC_Bool IsArabicChar(eC_UTF16 cCharacter);

    static eC_Bool CharacterMapContains(eC_UTF16 c);
    static eC_Bool CombiCharacterMapContains(eC_UTF16 c);

    static CombCharRep_t GetCombCharRep(eC_UTF16 c1, eC_UTF16 c2);

    static eC_Bool IsTransparent(eC_UTF16 c);
};

#endif
