/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifdef GUILIANI_WRITE_GUI
#ifndef GUI_STREAMWRITER_JSON__H
#define GUI_STREAMWRITER_JSON__H

#include "GUIStreamWriter.h"
#include "eC_TList_doubleLinked.h"

/// Writes json-formatted streaming files.

/** This implementation produces human-readable output which can be passed to a json-viewer.

@see GUILIANI_STREAMING
@ingroup GUILIANI_STREAMING
*/
class CGUIStreamWriterJSON : public CGUIStreamWriter
{
public:
    CGUIStreamWriterJSON();
    ~CGUIStreamWriterJSON();

    virtual void WriteUByte(const eC_UByte ubByte, const eC_Char* const pcTag);
    virtual void WriteByte(const eC_Byte bByte, const eC_Char* const pcTag);
    virtual void WriteUInt(const eC_UInt uiInt, const eC_Char* const pcTag);
    virtual void WriteUShort(const eC_UShort usShort, const eC_Char* const pcTag);
    virtual void WriteInt(const eC_Int iInt, const eC_Char* const pcTag);
    virtual void WriteHex(const eC_UInt uiInt, const eC_Char* const pcTag);
    virtual void WriteColor(const ColorValue_t eColor, const eC_Char* const pcTag);
    virtual void WriteFloat(const eC_Float fFloat, const eC_Char* const pcTag);
    virtual void WriteString(const eC_String &pkString, const eC_Char* const pcTag);
    virtual void WriteShort(const eC_Short iShort, const eC_Char* const pcTag);
    virtual void WriteBool(const eC_Bool bBool, const eC_Char* const pcTag);
    virtual void WriteBinary(const eC_TArray<eC_UByte>& aData, const eC_Char* const pcTag);

    virtual void WriteCommentTag(const eC_Char* const pcTag);
    virtual void WriteCommentTag(const eC_String& kTag);

    virtual void WriteStreamingFileHeader(const eC_Int iFileVersion);
    virtual eC_Bool SetFileEndianess(FileEndianess_t eFileEndianess);
    virtual eC_Bool SetFileAlignment(FileAlignment_t eFileAlignment);

private:
    /** write node to stream
    @param kName name of node
    @param kValue value of node
    @param bAddDoubleQuotes if true double-quotes will be added
    */
    void WriteNode(const eC_String& kName, const eC_String& kValue, const eC_Bool& bAddDoubleQuotes);

    /** write a node-name to stream
    @param kName name of node
    */
    void WriteNodeName(const eC_String& kName);

    /** write a node-value to stream
    @param kValue value of node
    @param bAddDoubleQuotes if true double-quotes will be added
    */
    void WriteNodeValue(const eC_String& kValue, const eC_Bool& bAddDoubleQuotes);

    /** begin a new object
    @param kName name of object
    */
    void BeginObject(const eC_String& kName);

    /// end object
    void EndObject();

    /** begin a new array
    @param kName name of array
    */
    void BeginArray(const eC_String& kName);

    /// end array
    void EndArray();

    void WriteIndentation();
    void WriteLineEnding();

private:
    enum TokenType_t
    {
        TT_OBJECT,
        TT_ARRAY,
        TT_NODE
    };

    typedef struct
    {
        TokenType_t eType;
        eC_UInt uiElementCount;
    } TokenEntry_t;

private:
    CGUIStreamWriterJSON(const CGUIStreamWriterJSON& kSource);
    CGUIStreamWriterJSON& operator=(const CGUIStreamWriterJSON& kSource);

    void WriteUnterminatedString(const eC_String& kString);

private:
    eC_TListDoubleLinked<TokenEntry_t> m_kTokenStack;
};

#endif
#endif
