/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifdef GUILIANI_STREAM_GUI

#ifndef GUI_STREAMREADER_XML__H
#define GUI_STREAMREADER_XML__H

#include "GUIStreamReader.h"

/// Reads streaming files written by CGUIStreamWriterXML.

/** This implementation has the ability to compare each tag that was read with
    the expected tag. Each of the read methods performs a check for the correct
    tag and throws a CGUIStreamingException if the tags do not match.

    To prevent checking of tags, e.g. when the tag is not necessary to be known,
    just provide an empty string. This will de-activate this check.

    If performance in production scenarios is needed the StreamReaderXML might not
    be the best choice, since it generates lot of overhead for the human-readable
    file-content and integrity checks. If performance is of the essence the
    StreamReaderBinary should be used instead.

    The ReadString() implementation unescapes XML predefined entities. See
    StringOperations::UnescapeXML() for details.

    @ingroup GUILIANI_STREAMING
*/
class CGUIStreamReaderXML : public CGUIStreamReader
{
public:
    CGUIStreamReaderXML(void);

    virtual const eC_String& GetFileSuffix() const
    {
        static const eC_String ckSuffix(".xml");
        return ckSuffix;
    }

    virtual eC_UByte ReadUByte(const eC_Char* pcTag = NULL);
    virtual eC_Byte ReadByte(const eC_Char* pcTag = NULL);
    virtual eC_UInt ReadUInt(const eC_Char* pcTag = NULL);
    virtual eC_Int ReadInt(const eC_Char* pcTag = NULL);
    virtual eC_UInt ReadHex(const eC_Char* pcTag = NULL);
    virtual eC_Float ReadFloat(const eC_Char* pcTag = NULL);
    virtual eC_String ReadString(const eC_Char* pcTag = NULL);
    virtual eC_UShort ReadUShort(const eC_Char* pcTag = NULL);
    virtual eC_Short ReadShort(const eC_Char* pcTag = NULL);
    virtual eC_Bool ReadBool(const eC_Char* pcTag = NULL);
    virtual void ReadBinary(eC_TArray<eC_UByte>& aData, const eC_Char* pcTag = NULL);
    virtual ColorValue_t ReadColor(const eC_Char* pcTag = NULL);

    virtual void DeleteCommentTag(const eC_Char* pcTag = NULL);
    virtual void DeleteCommentTag(const eC_String &kString);

    /// @return clone this stream-reader
    virtual CGUIStreamReader* Clone() const;

    virtual eC_Int ReadStreamingFileHeader()
    {
        return ReadInt("FileVersion");
    }

private:
    /** This prototype is declared only to prevent auto-generation by the compiler.
        There is no implementation for this method. */
    CGUIStreamReaderXML(const CGUIStreamReaderXML& kSource);

    /** This prototype is declared only to prevent auto-generation by the compiler.
        There is no implementation for this method. */
    CGUIStreamReaderXML& operator=(const CGUIStreamReaderXML& kSource);

    /** Parses for leading and trailing XML tags and writes the enclosed
        value string into m_aReadBuffer.
        @param pcTag The expected tag. Is checked if GUILIANI_STREAM_CMP_TAGS
               is defined.
        @return True if node was successfully parsed, otherwise False.
        @throw CGUIStreamingException if a wrong tag was found.
      */
    eC_Bool ParseNode(const eC_Char* pcTag);

    /** Helper that reads the next character in the streaming file and
        increments the line counter if it is a newline character.
        @return The character read from the streaming file.
      */
    inline eC_Char ReadChar();
};

#endif
#endif
