/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef GUILAYERCONFIG__H_
#define GUILAYERCONFIG__H_

#include "eC_Types.h"

#include "GUIObject.h"

static const eC_UInt cuiMaxNumberOfBuffers = 2;

/** types of pixel-formats
*/
enum LayerPixFormat_t
{
    LPF_UNUSED = 0, ///< unused enum
    LPF_RGB8888, ///< RGB with 32 bits (alpha is ommited)
    LPF_RGBA8888, ///< RGBA with 32 bits
    LPF_RGB565, ///< RGB with 5,6,5
    LPF_NUM ///< last enum
};

/** enum used for call to UpdateLayer
*/
enum LayerUpdate_t
{
    LU_UNUSED = 0, ///< unused enum
    LU_VISIBILITY, ///< visibility was changed
    LU_POSITION, ///< position was changed
    LU_SIZE, ///< size was changed
    LU_ALPHA, ///< alpha-value was changed
    LU_CHROMAKEY, ///< chroma-key was changed
    LU_CLIPRECT, ///< clipping rect was changed
    LU_NUM ///< last enum
};

/** configuration for layer
*/
struct LayerConfiguration_t
{
    LayerConfiguration_t()
    {
        m_uiLayerID = 0;
        m_bMainLayer = false;
        m_ePixFormat = LPF_UNUSED;
        m_uiNumBuffers = 0;
        for (eC_UInt uiBufferIndex = 0; uiBufferIndex < cuiMaxNumberOfBuffers; ++uiBufferIndex)
            m_pkBuffers[uiBufferIndex] = NULL;
        m_pkRenderBuffer = NULL;
        m_pkDisplayBuffer = NULL;
        m_bCached = false;
        m_iXPos = 0;
        m_iYPos = 0;
        m_uiWidth = 0;
        m_uiHeight = 0;
        m_uiPaddedWidth = 0;
        m_ubAlpha = 255;
        m_bVisible = false;
        m_iVisibleXPos = 0;
        m_iVisibleYPos = 0;
        m_uiVisibleWidth = 0;
        m_uiVisibleHeight = 0;
        m_bChromaKeyActive = false;
        m_uiChromaKey = 0;
        m_ubChromaKeyAlpha = 0;
        m_pkConnectedObject = NULL;
        m_bModified = false;
    }

    /** Get bits per pixel for current layer
    @return bits per pixel
    */
    eC_UInt GetBPP()
    {
        switch (m_ePixFormat)
        {
            case LPF_RGB565:
                return 16;

            case LPF_RGB8888:
            case LPF_RGBA8888:
                return 32;

            default:
                return 0;
        }
    }

    eC_UInt m_uiLayerID; ///< ID to access layer
    eC_Bool m_bMainLayer; ///< if true the layer is used for the screen
    LayerPixFormat_t m_ePixFormat; ///< pixelformat
    eC_UInt m_uiNumBuffers; ///< number of buffers
    volatile void* m_pkBuffers[cuiMaxNumberOfBuffers]; ///< buffer
    void* m_pkRenderBuffer; ///< current buffer for rendering
    void* m_pkDisplayBuffer; ///< current buffer for display
    eC_Bool m_bCached; ///< buffers are cached, if platform needs this
    eC_Int m_iXPos; ///< x-position
    eC_Int m_iYPos; ///< y-position
    eC_UInt m_uiWidth; ///< width
    eC_UInt m_uiHeight; ///< height
    eC_UInt m_uiPaddedWidth; ///< padded width if platform needs this
    eC_UByte m_ubAlpha; ///< alpha-value for layer
    eC_Bool m_bVisible; ///< layer is visible
    eC_Int m_iVisibleXPos; ///< first visible horizontal pixel
    eC_Int m_iVisibleYPos; ///< first visible vertical pixel
    eC_UInt m_uiVisibleWidth; ///< visible horizontal portion of the layer
    eC_UInt m_uiVisibleHeight; ///< visible vertical portion of the layer
    eC_Bool m_bChromaKeyActive; ///< chroma-key is active
    eC_UInt m_uiChromaKey; ///< chroma-key
    eC_UByte m_ubChromaKeyAlpha; ///< chroma-key replace alpha
    CGUIObject* m_pkConnectedObject; ///< connected object
    eC_Bool m_bModified; ///< is layer modified?
};

#endif
