/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef GUIIMAGEDECODER__H_
#define GUIIMAGEDECODER__H_

#include "eC_Types.h"
#include "eC_File.h"
#include "eC_TList_doubleLinked.h"
#include "eC_String.h"
#include "GUIImageData.h"


/// Base class for the image decoders.

/** All decoder classes must be derived from this class. */

class CGUIImageDecoder
{
public:
    /** Virtual destructor. */
    virtual ~CGUIImageDecoder();

    /** Must be implemented in each derived class to decode an image.
        @param rkImageData The decoded image has to be stored here by each
               derived implementation.
        @param pkImageFile Pointer to image file to load.
        @return Implementations must return True if the image was decoded
                successfully, False if anything went wrong and the contents of
                kImageData are invalid.
    */
    virtual eC_Bool LoadImg(CGUIImageData& rkImageData, eC_File* pkImageFile) = 0;

    /** Get list of supported file-extensions
    @return list of extensions
    */
    eC_TListDoubleLinked<eC_String>& GetSupportedFileExtensions();

protected:
    /// Struct which holds the information about the supported image formats of the decoder.
    /**
        You must explicit set the supported image format to true, otherwise it will be ignored by the
        loader.
    */
    struct SupportedImageTypes_t
    {
        CGUICommonEnums::ImageType_t eImgType;///< Image type
        eC_Bool bSupported; ///< True if supported
    };

protected:
    /** Constructor of CGUIImageDecoder, which sets all supported image types to False. */
    CGUIImageDecoder();

    void WriteConvertedColor(
        const eC_UInt& uiInputColor,
        const CGUICommonEnums::ImageType_t& eInputFormat,
        const CGUICommonEnums::ImageType_t& eOutputFormat,
        eC_UByte* pubData,
        eC_UInt& uiDataIndex,
        const eC_UByte& ubAdvance,
        const eC_UInt& uiCount = 1);

protected:
    SupportedImageTypes_t m_kSupportedImageTypes[cuiNumOfImageTypes]; ///< Array of supported image types
    eC_TListDoubleLinked<eC_String> m_kSupportedFileExtensions; ///< list of supported file-extensions
};

#endif
