/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef GUIIMAGEDATA__H_
#define GUIIMAGEDATA__H_

#include "eC_Types.h"
#include "GUICommonEnums.h"

// TODO: concept for supporting vector and bitmap-images
//new for SVGs
#include <vector>
#include "SVGShape.h"

/// Contains information about an image loaded by CGUIImageLoader.
/** Typical use case for this class is that a graphics wrapper creates an
    object of this class, sets some attributes to make sure the image uses the
    correct format etc., then passes it to the image loader which hands it over
    to the individual image decoders that fill the data section with the
    decoded image data.
  */
class CGUIImageData
{
public:
    /** CGUIImageData constructor */
    CGUIImageData();

    /** CGUIImageData destructor */
    ~CGUIImageData();

    /** Sets the width of the image. Should only be called by an image decoder.
        @param uiWidth Width of the image.
      */
    void SetWidth(const eC_UInt &uiWidth);

    /** Sets the height of the image. Should only be called by an image decoder.
        @param uiHeight Height of the image.
      */
    void SetHeight(const eC_UInt &uiHeight);

    /** Sets the data pointer of the image. Should only be called by an image
        decoder.
        @param *pubData Pointer to memory to be used for storing a decoded image.
      */
    void SetData(eC_UByte *pubData);

    /** Sets the data-pointer of an SVG-image. Should only be called by an image.
    @param pkSVGShapes Pointer to std::vector<CSVGShape>.
    @todo calculate size and m_pubData has to be set but won't work with this approach.
    */
    void SetDataSVG(CSVGShape* pkSVGShapes);

    /** Gets the data pointer of an SVG-image.
    @return Pointer to CSVGShapes.
    @todo calculate size!
    */
    CSVGShape* GetDataSVG();

    /** Sets the 'filler' value. It is typically used by decoders that load
        non-alpha images but store it in an alpha-enabled format. This
        'filler' is then used as the alpha value.
        This method should be called before CGUIImageDecoder::LoadImg().
        @param ubFillerByte Value used for the alpha filler.
      */
    void SetFiller(const eC_UByte &ubFillerByte);

    /** Sets the image format type.
        @param eImageType The format type.
      */
    void SetImageType(const CGUICommonEnums::ImageType_t &eImageType);

    /** Sets the destructor's behaviour.
        @param bDeleteDataOnDestruction Pass true to have the destructor delete
               the decoded image data.
      */
    void SetDeleteDataOnDestruction(const eC_Bool bDeleteDataOnDestruction);

    /// @return The width of the image.
    eC_UInt GetWidth() const;

    /// @return The height of the image.
    eC_UInt GetHeight() const;

    /// @return The format type of the image.
    CGUICommonEnums::ImageType_t GetImageType() const;

    /** @return Value of the 'filler' byte.
        For an explanation of this value, see SetFiller().
      */
    eC_UByte GetFiller() const;

    /// @return Pointer to the decoded image data.
    eC_UByte* GetData() const;

    /// @return The bit depth of the image.
    eC_UByte GetBytePerPixel() const;

    /// @return True if this image data deletes its contents upon destruction.
    eC_Bool GetDeleteDataOnDestruction() const;

    /** Returns the size (in bytes) of this image.
        Width, height and bits per pixel must have been set correctly before.
        @return Image size in bytes.
    */
    eC_UInt GetImageSize() const;

    /** Returns whether the image has alpha channel or not.
        @return True if the image has alpha data, otherwise False.
    */
    eC_Bool HasAlpha() const;

    /** Deletes the image data array. */
    void DeleteData();

    /** Operator= method. This does a flat copy of the given source CGUIImageData object, which means that both
        GUIImageData objects will then reference the SAME DATA in memory.
        @param kSource Source object to be copied.
        @return This instance.
    */
    CGUIImageData& operator=(const CGUIImageData& kSource);

private:
    /** Copy-constructor. Do not use.
        Dummy declaration without implementation, just to hide it.
        @param kSource Source object to be copied.
    */
    CGUIImageData(const CGUIImageData& kSource);

    eC_UInt m_uiWidth; ///< Width.
    eC_UInt m_uiHeight; ///< Height.
    CGUICommonEnums::ImageType_t m_eImageType; ///< Format.

    /// Filler byte replaces the alpha value, if you load an image with alpha, but you don't want to use it.
    eC_UByte m_ubFillerByte;

    eC_UByte* m_pubData; ///< Image data.
    eC_Bool m_bDeleteData; ///< should the data be deleted on destruction?

    CSVGShape* m_pkSVGData;
    eC_UInt m_uiShapeCount;
};

#endif
