/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef GUIFACTORY__H_
#define GUIFACTORY__H_

#include "GUIBehaviourResource.h"
#include "GUICommandResource.h"
#include "GUIControlResource.h"
#include "GUILayouterResource.h"
#include "GUITextTypeResource.h"
#include "GUIAnimationTypeResource.h"
#include "RefCounted.h"
#include "SharedPtr.h"

#include <map>
#include <vector>


class CGUIObject;
typedef CGUIObject* (*ControlCreateCallback)();

class CGUILayouter;
typedef CGUILayouter* (*LayouterCreateCallback)();

class CGUIBehaviourDecorator;
typedef CGUIBehaviourDecorator* (*BehaviourCreateCallback)();

class CGUICommand;
typedef CGUICommand* (*CommandCreateCallback)();

class CGUIText;
typedef CGUIText* (*TextCreateCallback)();

#ifdef GUILIANI_ANIMATION_ACTIVE
class CGUIAnimation;
typedef CGUIAnimation* (*AnimationCreateCallback)();
#endif

class CGUIFactory;
/// Abstract factory pointer type.
typedef NSmartPtr::SharedPtr<CGUIFactory> CGUIFactoryPtr;

/// Describes a custom control.
struct ControlDescriptor
{
    /** Constructs a control descriptor.
    @param iID The unique class ID of the control.
    @param rkControlName Human-readable name of the control. For instance,
    for the control class CGUIButton, this could simply be 'button'.
    @param pkCreate Pointer to an instance of the custom control.
    */
    ControlDescriptor(
        eC_Int iID,
        const eC_String& rkControlName,
        ControlCreateCallback pkCreate) :
        m_iID(iID),
        m_kControlName(rkControlName),
        m_pkCreate(pkCreate)
    {}

    ControlDescriptor() :
        m_iID(DUMMY_CONTROL),
        m_kControlName(""),
        m_pkCreate(NULL)
    {}

    /// The control class ID.
    eC_Int m_iID;

    /// Human-readable name of the control.
    eC_String m_kControlName;

    /// Pointer to an instance of the control.
    ControlCreateCallback m_pkCreate;
};
typedef std::vector<ControlDescriptor> ControlVector;

/// Describes a custom behaviour decorator.
struct BehaviourDescriptor
{
    /** Constructs a behaviour descriptor.
    @param iID The unique class ID of the behaviour.
    @param rkClassIDName The behaviour class ID as a string.
    @param pkCreate Pointer to an instance of the custom behaviour.
    */
    BehaviourDescriptor(
        eC_Int iID,
        const eC_String& rkClassIDName,
        BehaviourCreateCallback pkCreate) :
        m_iID(iID),
        m_kClassIDName(rkClassIDName),
        m_pkCreate(pkCreate)
    {}

    BehaviourDescriptor() :
        m_iID(DUMMY_BEHAVIOUR),
        m_kClassIDName(""),
        m_pkCreate(NULL)
    {}

    /// The behaviour class ID.
    eC_Int m_iID;

    /// The behaviour class ID as a string.
    eC_String m_kClassIDName;

    /// Pointer to an instance of the behaviour.
    BehaviourCreateCallback m_pkCreate;
};
typedef std::vector<BehaviourDescriptor> BehaviourVector;

/// Describes a custom command.
struct CommandDescriptor
{
    /** Constructs a command descriptor.
    @param iID The unique class ID of the command.
    @param rkClassIDName The command class ID as a string.
    @param pkCreate Pointer to an instance of the custom command.
    */
    CommandDescriptor(
        eC_Int iID,
        const eC_String& rkClassIDName,
        CommandCreateCallback pkCreate) :
        m_iID(iID),
        m_kClassIDName(rkClassIDName),
        m_pkCreate(pkCreate)
    {}

    CommandDescriptor() :
        m_iID(DUMMY_COMMAND),
        m_kClassIDName(""),
        m_pkCreate(NULL)
    {}

    /// The command class ID.
    eC_Int m_iID;

    /// The command class ID as a string.
    eC_String m_kClassIDName;

    /// Pointer to an instance of the command.
    CommandCreateCallback m_pkCreate;
};
typedef std::vector<CommandDescriptor> CommandVector;

/// Describes a custom layouter.
struct LayouterDescriptor
{
    /** Constructs a layouter descriptor.
    @param iID The unique class ID of the layouter.
    @param rkClassIDName The layouter class ID as a string.
    @param pkCreate Pointer to an instance of the custom layouter.
    */
    LayouterDescriptor(
        eC_Int iID,
        const eC_String& rkClassIDName,
        LayouterCreateCallback pkCreate) :
        m_iID(iID),
        m_kClassIDName(rkClassIDName),
        m_pkCreate(pkCreate)
    {}

    LayouterDescriptor() :
        m_iID(DUMMY_LAYOUTER),
        m_kClassIDName(""),
        m_pkCreate(NULL)
    {}

    /// The layouter class ID.
    eC_Int m_iID;

    /// The layouter class ID as a string.
    eC_String m_kClassIDName;

    /// Pointer to an instance of the layouter.
    LayouterCreateCallback m_pkCreate;
};
typedef std::vector<LayouterDescriptor> LayouterVector;

/// Describes a custom texttype.
struct TextTypeDescriptor
{
    /** Constructs a text-type descriptor.
    @param iID The unique class ID of the text-type.
    @param rkClassIDName The text-type class ID as a string.
    @param pkCreate Pointer to an instance of the custom text-type.
    */
    TextTypeDescriptor(
        eC_Int iID,
        const eC_String& rkClassIDName,
        TextCreateCallback pkCreate) :
        m_iID(iID),
        m_kClassIDName(rkClassIDName),
        m_pkCreate(pkCreate)
    {}

    TextTypeDescriptor() :
        m_iID(DUMMY_LAYOUTER),
        m_kClassIDName(""),
        m_pkCreate(NULL)
    {}

    /// The text-type class ID.
    eC_Int m_iID;

    /// The text-type class ID as a string.
    eC_String m_kClassIDName;

    /// Pointer to an instance of the text-type.
    TextCreateCallback m_pkCreate;
};
typedef std::vector<TextTypeDescriptor> TextTypeVector;

#ifdef GUILIANI_ANIMATION_ACTIVE
/// Describes a custom animation.
struct AnimationDescriptor
{
    /** Constructs an animation descriptor.
    @param iID The unique class ID of the animation.
    @param rkClassIDName The animation class ID as a string.
    @param pkCreate Pointer to an instance of the custom animation.
    */
    AnimationDescriptor(
        eC_Int iID,
        const eC_String& rkClassIDName,
        AnimationCreateCallback pkCreate) :
        m_iID(iID),
        m_kClassIDName(rkClassIDName),
        m_pkCreate(pkCreate)
    {}

    AnimationDescriptor() :
        m_iID(DUMMY_LAYOUTER),
        m_kClassIDName(""),
        m_pkCreate(NULL)
    {}

    /// The layouter class ID.
    eC_Int m_iID;

    /// The layouter class ID as a string.
    eC_String m_kClassIDName;

    /// Pointer to an instance of the animation.
    AnimationCreateCallback m_pkCreate;
};
typedef std::vector<AnimationDescriptor> AnimationVector;
#endif

/// Interface for streaming factories.

/** Factories are used when loading streaming files.
    Each factory has to be derived from this base class.

@section sec_guifactory_overview Overview

    A factory creates GUI objects (controls), commands, layouters, behaviours
    and texts based on their IDs. The CGUIFactoryManager calls a method of
    this class, then calls ReadFromStream on the returned object. <p>
    For creating your own customized factory, you must derive it from CGUIFactory,
    implement the corresponding CreateXXX-interfaces and register it with the
    CGUIFactoryManager.

@section sec_guifactory_example Example usage

    Guiliani's standard objects are instantiated within the CGUIStandardFactory which comes with
    the framework. If you wish to add customized objects, so that they can be read from a streaming-file,
    you must implement a corresponding factory and register it with the framework. The factory's sole
    purpose is to instantiate your customized objects. This is done by mapping an ID read from a stream onto
    the matching <i>new CSomeCustomizedClass()</i> call.

    The following example code snippets demonstrate how this is done in practice.
    You must derive your customized factory from the CGUIFactory base class and re-implement those
    interfaces, for which you have created customized objects (In this example Controls and Commands).

@code
class ExampleFactory : public CGUIFactory
{
protected:
    virtual CGUIObject* CreateControl(ControlClassID_t eControlID);
    virtual CGUICommand* CreateCommand(CommandClassID_t eCommandID);
};
@endcode

    There exist dedicated methods for each of the different object types (controls, commands, layouters, behaviours and texts.)
    The factory method gets called by the framework and receives an ID, which it then uses internally within
    a switch/case statement to construct the matching object.<p>

    The following example implements the <i>CreateControl</i> interface of a customized factory named <i>ExampleFactory</i>.
    If Guiliani itself reads an ID from a stream which it did not recognize, it calls the registered UserFactory's methods,
    thus requesting them to instantiate the object identified by <i>eControlID</i>. These IDs are typically declared as
    enumerations within the UserControlResource.h / UserCommandResource.h / UserLayouterResource.h / UserBehaviourResource.h
    and UserTextTypeResource.h headers. All the factory needs to do, is to check for the ID in a switch/case, create the
    corresponding object if possible, or return NULL otherwise.<p>

@code
CGUIObject* ExampleFactory::CreateControl(ControlClassID_t eControlID)
{
    switch (eControlID)
    {
    case CTL_EXAMPLE:
        return new ExampleControl();
        break;
    default:
        break;
    }
    return NULL;
}
@endcode

    The same applies to all other types of objects, such as e.g. commands. As you can see from the following example:
@code
CGUICommand* ExampleFactory::CreateCommand(CommandClassID_t eCommandID)
{
    switch (eCommandID)
    {
    case CMD_EXAMPLE:
        return new ExampleCommand();
        break;
    default:
        break;
    }

    return NULL;
}
@endcode

    Once you have finished implementing your customized factory, use the CGUIFactoryManagers AddUserFactory interface
    to make your factory known within the framework, and you are done. The framework will now be able to use your factory to
    load your custom objects from a stream.

@code
    GETFACTORY.AddUserFactory(new ExampleFactory());
@endcode

*/
class CGUIFactory : public NSmartPtr::RefCounted
{
public:
    /** Register new control
    @param desc description of the control
    @return True if successfully registered
    */
    eC_Bool RegisterControl(ControlDescriptor desc);

    /** Unregister the control with the given ID
    @param iID ID of the control
    */
    void UnregisterControl(const eC_Int& iID);

    /** Get all registered controls
    @param rkControls fill this
    */
    void GetControls(ControlVector& rkControls);

    /** Register new behaviour
    @param desc description of the behaviour
    @return True if successfully registered
    */
    eC_Bool RegisterBehaviour(BehaviourDescriptor desc);

    /** Unregister the behaviour with the given ID
    @param iID ID of the behaviour
    */
    void UnregisterBehaviour(const eC_Int& iID);

    /** Get all registered behaviours
    @param rkBehaviours fill this
    */
    void GetBehaviours(BehaviourVector& rkBehaviours);

    /** Register new command
    @param desc description of the command
    @return True if successfully registered
    */
    eC_Bool RegisterCommand(CommandDescriptor desc);

    /** Unregister the command with the given ID
    @param iID ID of the command
    */
    void UnregisterCommand(const eC_Int& iID);

    /** Get all registered commands
    @param rkCommands fill this
    */
    void GetCommands(CommandVector& rkCommands);

    /** Register new layouter
    @param desc description of the layouter
    @return True if successfully registered
    */
    eC_Bool RegisterLayouter(LayouterDescriptor desc);

    /** Unregister the layouter with the given ID
    @param iID ID of the layouter
    */
    void UnregisterLayouter(const eC_Int& iID);

    /** Get all registered layouters
    @param rkLayouters fill this
    */
    void GetLayouters(LayouterVector& rkLayouters);

    /** Register new text-type
    @param desc description of the text-type
    @return True if successfully registered
    */
    eC_Bool RegisterText(TextTypeDescriptor desc);

    /** Unregister the text-type with the given ID
    @param iID ID of the text-type
    */
    void UnregisterText(const eC_Int& iID);

    /** Get all registered text-types
    @param rkTextTypes fill this
    */
    void GetTextTypes(TextTypeVector& rkTextTypes);

#ifdef GUILIANI_ANIMATION_ACTIVE
    /** Register new animation
    @param desc description of the animation
    @return True if successfully registered
    */
    eC_Bool RegisterAnimation(AnimationDescriptor desc);

    /** Unregister the animation with the given ID
    @param iID ID of the animation
    */
    void UnregisterAnimation(const eC_Int& iID);

    /** Get all registered animations
    @param rkAnimations fill this
    */
    void GetAnimations(AnimationVector& rkAnimations);
#endif

protected:
    /// The factory manager needs to call the load methods.
    friend class CGUIFactoryManager; ///< @todo to access private functions

    /// Destructor.
    virtual ~CGUIFactory();

    /** Creates a new GUI object.
        @param eControlID ID of the control.
        @return The object that has been created or NULL if the ID was unknown.
    */
    virtual CGUIObject* CreateControl(ControlClassID_t eControlID) { return NULL; }

    /** Creates a new layouter.
        @param eLayouterID ID of the layouter.
        @return The layouter that has been created or NULL if the ID was unknown.
    */
    virtual CGUILayouter* CreateLayouter(LayouterClassID_t eLayouterID) { return NULL; }

    /** Creates a new behaviour decorator.
        @param eBehaviourID ID of the behaviour.
        @return The behaviour that has been created or NULL if the ID was unknown.
    */
    virtual CGUIBehaviourDecorator* CreateBehaviour(BehaviourClassID_t eBehaviourID) { return NULL; }

    /** Creates a new command.
        @param eCommandID ID of the command.
        @return The command that has been created or NULL if the ID was unknown.
    */
    virtual CGUICommand* CreateCommand(CommandClassID_t eCommandID) { return NULL; }

    /** Creates a new text object.
        @param eTextType The text type ID.
        @return The text that has been created or NULL if the ID was unknown.
      */
    virtual CGUIText* CreateText(TextTypeID_t eTextType) { return NULL; }

#ifdef GUILIANI_ANIMATION_ACTIVE
    /** Creates a new animation object.
        @param eAnimation The animation ID.
        @return The animation that has been created or NULL if the ID was unknown.
      */
    virtual CGUIAnimation* CreateAnimation(AnimationType_t eAnimation) { return NULL; }
#endif

protected:
    // controls
    typedef std::map<eC_Int, ControlDescriptor> ControlCallbackMap; ///< map for controls
    ControlCallbackMap m_kControlMap; ///< map for controls

    // behaviours
    typedef std::map<eC_Int, BehaviourDescriptor> BehaviourCallbackMap; ///< map for behaviours
    BehaviourCallbackMap m_kBehaviourMap; ///< map for behaviours

    // commands
    typedef std::map<eC_Int, CommandDescriptor> CommandCallbackMap; ///< map for commands
    CommandCallbackMap m_kCommandMap; ///< map for commands

    // layouters
    typedef std::map<eC_Int, LayouterDescriptor> LayouterCallbackMap; ///< map for layouters
    LayouterCallbackMap m_kLayouterMap; ///< map for layouters

    // texts
    typedef std::map<eC_Int, TextTypeDescriptor> TextCallbackMap; ///< map for text-types
    TextCallbackMap m_kTextMap; ///< map for text-types

#ifdef GUILIANI_ANIMATION_ACTIVE
    // animations
    typedef std::map<eC_Int, AnimationDescriptor> AnimationCallbackMap; ///< map for animations
    AnimationCallbackMap m_kAnimationMap; ///< map for animations
#endif
};

#endif
