/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef GUIEVENTHANDLER__H_
#define GUIEVENTHANDLER__H_

#include "eC_TList_doubleLinked.h"
#include "eC_Types.h"
#include "GUIObjectHandleResource.h"

/// Retrieve pointer to event handler instance
#define GETEVENTHDL            CGUIEventHandler::GetInstance()

/// Default time (in milliseconds) for long click events
const eC_UInt cuiDefaultLongClickTime = 1000;
/// Default time (in milliseconds) for double click events
const eC_UInt cuiDefaultDoubleClickTime = 300;


class CGUIObject;
class CGUIEvent;
class CGUIMouseEvent;
class CGUICompositeObject;

/// This is the event handler.

/** It handles keyboard and mouse events based on CGUIEvent in HandleEvent by
    calling the appropriate methods on GUI objects. It also manages the focus,
    keeps track of the currently highlighted and pressed objects and the current
    modal dialog.

    To learn more about event handling, see @ref page_event_handling.
*/
class CGUIEventHandler
{
    friend class CGUIObject; ///< @todo to access private functions
    friend class CGUI; ///< @todo to access private functions
public:
    /** return reference to event handler instance
    @return instance
    */
    static  CGUIEventHandler& GetInstance();

    /** Activate or De-Activate legacy behaviour to be backwards-compatible to older Guiliani-versions
    @param bLegacyBehaviour true if mode should be activated
    */
    void SetLegacyBehaviour(const eC_Bool bLegacyBehaviour);

    /** Get object that is currently focused.
    @return Pointer to the currently focused object.
    */
     CGUIObject* GetFocussedObject() const;

    /** Return the object which was formerly focused.
    @return object which lost the focus
    @note This is only valid within GetFocus())
    */
     CGUIObject* GetFormerlyFocussedObject() const;

    /** Informs the event handler that the focus is lost.
        This happens for instance automatically if the currently focused object changes to a state
        (such as grayed out, disabled or invisible) in which it must not remain focused.
    */
    void FocusLost();

    /** Return the currently highlighted object (the one beneath the cursor)
    @return currently highlighted object, if any
    */
     CGUIObject* GetHighlightedObject() const;

    /** Return the object which is currently being pressed
    @return currently pressed object, if any
    */
     CGUIObject* GetPressedObject() const;

    /** Return the object which is currently being dragged
    @return currently dragged object, if any
    */
     CGUIObject* GetDraggedObject() const;

    /** Return the object which is currently under the mouse pointer.
    @return object currently under mouse-pointer
    */
     CGUIObject* GetObjectUnderMousePointer() const;

    /** Return the object which formerly was under the mouse pointer.
    @return object formerly under mouse-pointer
    @note This is only valid within DoMouseEnter()
    */
     CGUIObject* GetFormerlyUnderMouseObject() const;

    /** Get the current modal dialog that is active.
    @return current active modal dialog
    */
    CGUICompositeObject* GetCurrentModalDialog() const;

    /** Make a specific dialog the active modal dialog.
        This also sets the focus into the dialog if there is at least one focusable object in this dialog.
        Modal dialogs can be stacked, and if the modal dialog is already in the list of modal dialogs,
        it will become the active modal dialog.
        @param pModalDialog Pointer to the CompositeObject which will become the model dialog
    */
    void SetModalDialog(CGUICompositeObject* pModalDialog);

    /** Release the given modal dialog (and thus remove it from the stack of modal dialogs)
        If there were other previous dialogs on the modal dialog stack, the most recent one will now
        become the active modal dialog again. In this case the framework will attemp to focus a child object
        of this newly actived modal dialog.
        @param pModalDialog Pointer to the modal dialog object
    */
    void ReleaseModalDialog(CGUICompositeObject* pModalDialog);

    /**
       Check whether the object is a modal dialog or not.
       @param pObj Object to query
       @return True if modal, False otherwise
    */
    eC_Bool IsObjModalDialog(CGUICompositeObject* const pObj) const;

    /** Add a new exception for modal dialogs.
        this object can be used even when a modal dialog is shown
        use with caution!!
        @param pkException object for exception
    */
    void AddModalDialogException(CGUICompositeObject* pkException);

    /** Remove a previously added exception for modal dialogs
        @param pkException object to remove
    */
    void RemoveModalDialogException(CGUICompositeObject* pkException);

    /** Indicates whether the last received Event was generated by a pointing device
    @return true if mouse was used
    */
    eC_Bool MouseWasUsed() const;

    /** Sets the MouseWasUsed flag
    @param bMouseUsed
    */
    void SetMouseWasUsed(eC_Bool bMouseUsed);

    /** Returns the last known mouse position
    @param uiMouseX x-position
    @param uiMouseY y-position
    */
    void GetLastMousePos(eC_UInt &uiMouseX, eC_UInt &uiMouseY) const;

    /** Sets an attribute that controls whether user input ought to be ignored or not.
        @param  bDisableEventHandling true: Disable user input.
                                      else: Enable it.
    */
    void SetDisableEventHandling(eC_Bool bDisableEventHandling);

    /** Returns whether user input currently is ignored.
        @return True if user input currently is ignored, otherwise False.
    */
    eC_Bool GetDisableEventHandling() const;

    /** Returns a pointer to the event that is currently being handled.

        Use this method in an event handling slot to examine the current event
        in greater detail, for instance for checking keyboard modifier keys.

        This method returns NULL if the event handler is not currently handling
        an event. This means that user code can only expect to receive a valid
        pointer from this method in any of the event handling slots declared in
        CGUIBehaviour.
        @return pointer to the event that is currently being handled.
      */
    const CGUIEvent* GetCurrentEvent() const;

    /** Set the modifiers for mousewheel scrolling.
        When scroll events occur and these modifiers are pressed the DoScrollLeft()
        and DoScrollRight() slots are called instead of DoScrollUp() and DoScrollDown().

        The default modifier is the SHIFT key.
        @param uiScrollModifiers The modifiers bitfield.
        @see CGUIEvent::Modifier_t
    */
    void SetScrollModifiers(eC_UInt uiScrollModifiers);

    /** Set the threshold (in pixels) after which a drag event shall not be recognized as a "click" anymore.
    Especially for touch-screen devices, it is usually not desired that <i>dragging</i> an object also results in <i>clicking</i> it.
    This threshold defines the dragging distance after which releasing the finger will not call DoClick() anymore, but instead always
    result in a call to DoButtonUp().

    @param iDragThreshold Threshold (in pixels) after which a drag event shall not be recognized as a "click" anymore.
    @deprecated Function call will not work anymore, use SetLongClickThresholdX, SetLongClickThresholdY, 
    SetClickToleranceX, SetClickToleranceY instead.
    */
    void SetDragThreshold(const eC_Int iDragThreshold);

    /** Returns the current drag threshold.
    @return Threshold (in pixels) after which a drag event shall not be recognized as a "click" anymore.
    */
    eC_Int GetDragThreshold() const;

    /** Disables highlighting of objects in reaction to "mouse-over" events.
        By default highlighting is enabled.
        It is advisable to disable highlighting on devices which do not require it.
        @param bDisabled True disables highlighting, False enables it.
    */
    void SetHighlightingDisabled(const eC_Bool bDisabled);

    /** Sets the time in milliseconds that a mouse button needs be held down until a long click event is generated.
        @param uiLongClickTime Time in milliseconds until a long click event is generated. */
    void SetLongClickTime(const eC_UInt uiLongClickTime);

    /** Returns the time in milliseconds that a mouse button needs be held down until a long click event is generated.
        @return Time in milliseconds during which a long click event is generated. */
    eC_UInt GetLongClickTime() const;

    /** Sets the time in milliseconds during which a button has to be clicked twice to generate a double click event.
        @param uiDoubleClickTime Time in milliseconds in which a double click event is generated. */
    void SetDoubleClickTime(const eC_UInt uiDoubleClickTime);

    /** Returns the time in milliseconds during which a button has to be clicked twice to generate a double click event.
        @return Time in milliseconds during which a double click event is generated. */
    eC_UInt GetDoubleClickTime() const;

    /** Returns the maximum allowed idle time in milliseconds.
        @return maximum allowed idle time in milliseconds */
    eC_UInt GetIdleTime();

    /** Returns the start coordinates of an ongoing drag operation.
        The returned values are valid only if there currently is a drag operation in progress.
        Do not use them after the concluding DoDragEnd() event.
        @param iDragStartX X Position in screen coordinates where the drag sequence started
        @param iDragStartY Y Position in screen coordinates where the drag sequence started */
    void GetDragStart(eC_Int& iDragStartX, eC_Int& iDragStartY);

    /** Returns the timestamp of the last ButtonDown-Event.
        @return Timestamp of the last ButtonDown-Event */
    eC_UInt GetLastButtonDownTime();

    /** Executes a click on a given object, iterates through whole event hierarchy
        @param pObject Object to be clicked
        @return true if executed
    */
    eC_Bool ExecuteClickHandler(CGUIObject* pObject);

    /** Checks if the given object lies within the currently active modal dialog (if there is one).
    @param pObj Pointer to object which shall be checked
    @return True if there is no modal dialog or if the object lies within the modal dialog, False otherwise */
    eC_Bool IsWithinModalDialogIfAny(const CGUIObject* pObj) const;

private:
    ~CGUIEventHandler();

    /** Takes care of long clicks (chain of responsibility).
    @param bHandled Will be set to true if event was handled.
    */
    void HandleLongClick(eC_Bool& bHandled);

    /** Takes care of focus (chain of responsibility).
    */
    void HandleFocus();

    /** Takes care of character inputs (chain of responsibility).
    @param pEvent Event to be handled.
    @param bHandled Will be set to true if event was handled.
    */
    void HandleChar(CGUIEvent* pEvent, eC_Bool& bHandled);

    /** Takes care of key ups (chain of responsibility).
    @param pEvent Event to be handled.
    @param bHandled Will be set to true if event was handled.
    */
    void HandleKeyUp(CGUIEvent* pEvent, eC_Bool& bHandled);

    /** Takes care of key down presses (chain of responsibility).
    @param pEvent Event to be handled.
    @param bHandled Will be set to true if event was handled.
    */
    void HandleKeyDown(CGUIEvent* pEvent, eC_Bool& bHandled);

    /** Takes care of mouse wheel (chain of responsibility).
    @param pEvent Event to be handled.
    @param bHandled Will be set to true if event was handled.
    */
    void HandleMouseWheel(CGUIEvent* pEvent, eC_Bool& bHandled);

    /** Takes care of mouse movements (chain of responsibility).
    @param pEvent Event to be handled.
    @param bHandled Will be set to true if event was handled.
    */
    void HandleMouseMove(CGUIEvent* pEvent, eC_Bool& bHandled);

    /** Takes care of right button down (chain of responsibility).
    @param pEvent Event to be handled.
    @param bHandled Will be set to true if event was handled.
    */
    void HandleRButtonDown(CGUIEvent* pEvent, eC_Bool& bHandled);

    /** Takes care of left button down (chain of responsibility).
    @param pEvent Event to be handled.
    @param bHandled Will be set to true if event was handled.
    */
    void HandleLButtonDown(CGUIEvent* pEvent, eC_Bool& bHandled);

    /** Takes care of right button up (chain of responsibility).
    @param pEvent Event to be handled.
    @param bHandled Will be set to true if event was handled.
    */
    void HandleRButtonUp(CGUIEvent* pEvent, eC_Bool& bHandled);

    /** Takes care of left button up (chain of responsibility).
    @param pEvent Event to be handled.
    @param bHandled Will be set to true if event was handled.
    */
    void HandleLButtonUp(CGUIEvent* pEvent, eC_Bool& bHandled);

    /** The supplied event will be forwarded by the event handler to the designated receiver.
        If HandleEvent() did not find a behaviour's DoUserEvent() that handles the event it calls
        standard event handling functions like DoDrag(), DoClick() and so on starting the chain
        of responsibility at the focused object.
        @param pEvent Event to be handled.
    */
    void HandleEvent(CGUIEvent *pEvent);

    /** The supplied event will be forwarded to HandleEvent. This method checks if a button with
    one of the given object ID's was pressed. If not NO_HANDLE will be returned. Alternatively
    it is possible to use GK_ACTION, GK_SPACE key to press the button with the current focus or
    GK_ESCAPE key to return with eOBJCancel to simulate a press of the cancel button.
    @param pEvent Event to be handled.
    @param eObjOK The ID of a button representing the OK button.
    @param eObjCancel The ID of a button representing a CANCEL button.
    @param eObjAdditional The ID of a third button which can be used to exit the modal dialog.
    @return The pressed object handle if equal to one of the three given parameters else NO_HANDLE.
    */
    ObjectHandle_t HandleModalEvents(
        CGUIEvent* pkEvent,
        ObjectHandle_t eObjOK = NO_HANDLE,
        ObjectHandle_t eObjCancel = NO_HANDLE,
        ObjectHandle_t eObjAdditional = NO_HANDLE);

    /** Notifies event handler that an object has been destroyed and allows it
        to check validity of internal pointers. */
    void NotifyOfDestruction(CGUIObject* pObj);

    /** Sets the object which is currently being pressed.
        If this object is a new valid object then its area will be invalidated so that it is
        redrawn later on. */
    void SetPressedObject(CGUIObject* pObj, eC_Bool bOverwriteOriginal = false);

    /** Set the highlighted object (the one beneath the cursor)
        If this object is a new valid object then its area will be invalidated so that it is
        redrawn later on. */
    void SetHighlightedObject(CGUIObject* pHighlightedObject);

    /** Set currently focused object.
        @param pObj Object which will be the focused object. */
    void SetFocussedObject(CGUIObject* pObj);

    /** Set the object which is currently being dragged
        @param pObj Object which is being dragged */
    void SetDraggedObject(CGUIObject* pObj);

    /** Copy-constructor.
        Dummy implementation to avoid unintended use of compiler-generated default    */
    CGUIEventHandler(const CGUIEventHandler& kSource);

    /** Operator= method.
        Dummy implementation to avoid unintended use of compiler-generated default    */
    CGUIEventHandler& operator=(const CGUIEventHandler& kSource);

    CGUIEventHandler();

    /// static protected member, event handler instance
    static CGUIEventHandler ms_kEventHandler;

    /** Pointer to the object which lay beneath the mouse-cursor during the last pass.
        (used for MouseOver-effects) */
    CGUIObject* m_pkFormerlyUnderMouseObject;

    /// Pointer to object which is currently being focused.
    /// May be NULL if the focused object is not set for any reason,
    /// e.g. no object is focusable (see IsFocussable()).
    CGUIObject* m_pkFocussedObject;

    /// Pointer to object which currently lies beneath the cursor.
    /// May be NULL if the cursor (mouse pointer) is outside of a modal dialog
    /// or outside of the GUI root object (usually called CMyGUI) or if the
    /// object that is under the cursor is not highlightable (see
    /// IsHighlightable()).
    CGUIObject* m_pkHighlightedObject;

    /// Pointer to object which is currently being pressed.
    /// May be NULL if nothing is pressed or if the object that the user tries
    /// to press is not pressable (check IsPressable()).
    CGUIObject* m_pkPressedObject;

    /// Pointer to object which was originally pressed (object where the last mouse down
    /// or uequivalent key press event occured).
    /// May be NULL if nothing is pressed or if the object that the user tries
    /// to press is not pressable (check IsPressable()).
    CGUIObject* m_pkOriginallyPressedObject;

    /// Pointer to object which is currently being pressed.
    /// May be NO_HANDLE if nothing is pressed or if the object that the user tries
    /// to press is not pressable (check IsPressable()).
    /// Special case used for handling of modal dialogs.
    /// Contains the pressed object id until it is overwritten due to another object pressed.
    ObjectHandle_t m_ePressedObjectID;

    /// Pointer to object which is currently being dragged.
    CGUIObject* m_pDraggedObject;

    /// Pointer to object which is currently beneath the cursor
    /// May be NULL if the cursor (mouse pointer) is outside of a modal dialog
    /// or outside of the GUI root object (usually called CMyGUI).
    CGUIObject* m_pkObjectUnderMousePointer;

    /// List of pointers to the modal dialogs. It remembers all the modal dialogs that are set.
    eC_TListDoubleLinked<CGUICompositeObject*> m_kModalDialogList;

    /// list of object which can also be used when a modal dialog is shown
    eC_TListDoubleLinked<const CGUICompositeObject*> m_kModalDialogExceptionList;

    /// Current status of the left mouse button.
    eC_Bool m_bLeftButtonPressed;
    /// Current status of the right mouse button.
    eC_Bool m_bRightButtonPressed;

    /// Last known mouse position, x coordinate.
    eC_Int m_iOldMouseX;
    /// Last known mouse position, y coordinate.
    eC_Int m_iOldMouseY;

    /// Indicates whether the last received Event was generated by a pointing device
    eC_Bool m_bMouseUsed;

    /// If set to true, the user can not trigger any events (e.g. button clicks) anymore.
    /// Like this one can ensure that an event has to be handled completely first until
    /// the user is allowed to trigger further events (e.g. is does not make sense to issue
    /// 2 switch dialog commands by 2 button clicks).
    eC_Bool m_bDisableEventHandling;

    /// This Object is used only in the event handler as temp object.
    CGUIObject* m_pkTempObject;

    /// Copy of the pointer of the event that is currently handled by HandleEvent.
    CGUIEvent* m_pkEvent;

    /// When scroll events occur and these modifiers are pressed the DoScrollLeft() and DoScrollRight() slots are called
    /// instead of DoScrollUp() and DoScrollDown().
    eC_UInt m_uiScrollModifiers;

    /// List of objects requesting the focus (Can be more than just one object in case SetFocussedObject is called recursively)
    eC_TListDoubleLinked<CGUIObject*> m_kListOfObjectsToFocus;
    /// Object pointer used during chain-of-responsibility of SetFocussedObject
    CGUIObject* m_pkTempFocussedObject;
    /// Object which was focused before moving the focus to another object (only valid during GetFocus())
    CGUIObject* m_pkFormerlyFocussedObject;

    /// Start X-position of the current drag-event in screen coordinates
    eC_Int m_iDragStartX;
    /// Start Y-position of the current drag-event in screen coordinates
    eC_Int m_iDragStartY;

    /// Threshold (in pixels) after which a drag event shall not be recognized as a "click" anymore
    eC_Int m_iDragThreshold;

    /// Current mouse x pos in case of a mouse event as integer
    eC_Int m_iCurXPos;

    /// Current mouse y pos in case of a mouse event as integer
    eC_Int m_iCurYPos;

    /// Current mouse x pos in case of a mouse event as eC_Value
    eC_Value m_vCurXPos;

    /// Current mouse y pos in case of a mouse event as eC_Value
    eC_Value m_vCurYPos;

    /// Indicates if highlighting is disabled or enabled
    eC_Bool m_bHighlightingDisabled;

    /// Indicates if a dragging operation is currently ongoing
    eC_Bool m_bIsDragging;

    /// Timestamp of the last generated Click event
    eC_UInt m_uiLastClickTime;

    /// Timespan in milliseconds in which a second click must be detected to generate a Double-Click Event
    eC_UInt m_uiDoubleClickTime;

    /// Timestamp of the last generated ButtonDown event
    eC_UInt m_uiLastButtonDownTime;

    /** Timespan in milliseconds after which a Longclick gets generated if the pressed object remains the same
        and the mouse movement does not exceed the allowed drag-threshold */
    eC_UInt m_uiLongClickTime;

    /// Flag indicating if a long click has already been generated for the preceeding ButtonDown event
    eC_Bool m_bLongClickGenerated;

    /// Object which received the last DoClick-Event. This is used for detecting double-clicks.
    CGUIObject* m_pkLastClickedObject;

    /** WARNING: the following part has been added for backwards-compatibility supporting older releases of Guiliani
        force DoButtonUp to be sent to PressedObject, even if ObjectUnderMousePointer has not received DoButtonDown
    */
    eC_Bool m_bForceDoButtonUpToPressedObject;

    /** ignore the dimensions of the rect of PressedObject when sending a click.
        this means that a DoClick gets emitted even if the mouse is not inside the PressedObject
        additionally the PressedObject is NOT set to NULL on leaving the corresponding object
    */
    eC_Bool m_bIgnorePressedObjectRect;

    /** ignore current dragging-status when emitting a DoClick
        this means that even if a drag-action has been started a click gets emitted on release
    */
    eC_Bool m_bDragAndClick;
};

#endif
