/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef GUICOMMONENUMS_H
#define GUICOMMONENUMS_H

/// The number of entries in the ImageType_t enum. It's very important to set this always to the right number of entries!
const eC_UInt cuiNumOfImageTypes = 14;

/** This class collects various enumerations which are common to multiple controls.
*/
class CGUICommonEnums
{
public:
    /// Axis-alignment for freedom of control
    enum AxisAlignment_t
    {
        AA_NONE, ///< not aligned
        AA_HORIZONTAL, ///< horizontal
        AA_VERTICAL, ///< vertical
        AA_BOTH ///< horizontal + vertical
    };

    /// Axis-types for displaying values
    enum AxisStyle_t
    {
        AS_LINEAR,  ///< linear
        AS_LOGARITHMIC, ///< logarithmic
    };

    /** Check if direction has horizonal part
    @param eAlignment alignment to check
    @return true if horizontal part is present
    */
    static eC_Bool HasHorizontal(const AxisAlignment_t& eAlignment)
    {
        return ((eAlignment == AA_HORIZONTAL) || (eAlignment == AA_BOTH));
    }

    /** Check if direction has vertical part
    @param eAlignment alignment to check
    @return true if vertical part is present
    */
    static eC_Bool HasVertical(const AxisAlignment_t& eAlignment)
    {
        return ((eAlignment == AA_VERTICAL) || (eAlignment == AA_BOTH));
    }

    /// orientation of controls
    enum Orientation_t
    {
        OR_HORIZONTAL, ///< horizontal orientation
        OR_VERTICAL ///< vertical orientation
    };

    /// Position of control
    enum Alignment_t
    {
        ALIGN_TOP, ///< aligned to top
        ALIGN_BOTTOM, ///< aligned to bottom
        ALIGN_LEFT, ///< aligned left
        ALIGN_RIGHT ///< aligned right
    };

    /** Enumeration ImageType_t for the loaded format.<br/>
    A = alpha, R = red, G = green, B = blue. The order of the letters in the abbreviation defines the order of a pixel.
    The number means the number of bits of each color channel.<br/>
    For example: RGBA_8888 means Red, Green, Blue, Alpha and every channel has 8 bits. So
    the whole pixel has 32 bits. Red has the most significant bit and alpha the least significant bits.
    The order is considered big endian so bytewise read, it forms 0xAA 0xRR 0xGG 0xBB.
    */
    enum ImageType_t
    {
        // Original format of the image.
        IMAGE_FORMAT = 0,

        // Trick: We use the upper byte to indicate how many bytes are used per pixel.
        // 1 byte per pixel.
        ALPHA_8 = 1 << 24,
        ALPHA_4,

        // 2 bytes per pixel.
        RGB_565 = 2 << 24,
        BGR_565,
        ABGR_4444,
        BGRA_4444,
        RGB_565_A8,

        // 3 bytes per pixel.
        RGB_888 = 3 << 24,
        BGR_888,

        // 4 bytes per pixel.
        ARGB_8888 = 4 << 24,
        ABGR_8888,
        RGBA_8888,
        BGRA_8888,
        RGB_8888
    };
};

#endif // GUICOMMONENUMS_H
