/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef GUICOLORPROPERTYOBSERVER__H_
#define GUICOLORPROPERTYOBSERVER__H_

#include "GUIObserver.h"

#include "GUICommonTypes.h"
#include "GUIPropertyResource.h"
#include "eC_TArray.h"

/**
@brief This observer provides the base-interface for controls when using color-properties
*/
class CGUIColorPropertyObserver : public CGUIObserver
{
public:
    /** manage color-property
    */
    struct ColorProperty_t
    {
        /// create a color-property
        ColorProperty_t() :
            bColorIsProperty(false),
            uiColorValue(0),
            eColorProperty(DUMMY_PROPERTY)
        {}

        /** create a color-property
        @param ColorIsProperty true if property
        @param ColorValue argb-value
        @param ColorProperty property
        */
        ColorProperty_t(const eC_Bool& ColorIsProperty, const eC_UInt& ColorValue, const GlobalProperty_t& ColorProperty) :
            bColorIsProperty(ColorIsProperty),
            uiColorValue(ColorValue),
            eColorProperty(ColorProperty)
        {}

        eC_Bool bColorIsProperty; ///< color is a property
        eC_UInt uiColorValue; ///< color-value in ARGB
        GlobalProperty_t eColorProperty; ///< color-property
    };

    /// default-constructor
    CGUIColorPropertyObserver() {}

    /// destructor
    ~CGUIColorPropertyObserver() {}

    virtual void OnNotification();

    /** this slot is called when color-properties have changed
    */
    virtual void OnColorPropertyChange() {}

    /** Add a new color-property
    @param bColorIsProperty If true the color will reference a property
    @param uiColorValue the color as ARGB
    @param eColorProperty the referenced property
    @return the index of the new color-property
    */
    eC_UInt AddColorProperty(
        const eC_Bool& bColorIsProperty,
        const eC_UInt& uiColorValue,
        const GlobalProperty_t& eColorProperty);

    /** Add a new color-property
    @param uiIndex index of the new color-property
    @param bColorIsProperty If true the color will reference a property
    @param uiColorValue the color as ARGB
    @param eColorProperty the referenced property
    */
    void AddColorProperty(
        const eC_UInt& uiIndex,
        const eC_Bool& bColorIsProperty,
        const eC_UInt& uiColorValue,
        const GlobalProperty_t& eColorProperty);

    /** Remove a color-property. This will also remove registration in property-handler
    @param uiIndex index of the color-property to remove
    */
    void RemoveColorProperty(const eC_UInt& uiIndex);

    /** Set a color-property
    @param uiIndex index of the color-property
    @param uiColorValue the color as ARGB
    */
    void SetColorProperty(
        const eC_UInt& uiIndex,
        const eC_UInt& uiColorValue);

    /** Set a color-property
    @param uiIndex index of the color-property
    @param eColorProperty the color as property
    */
    void SetColorProperty(
        const eC_UInt& uiIndex,
        const GlobalProperty_t& eColorProperty);

    /** Set a color-property
    @param uiIndex index of the color-property
    @param kColorValue the color as read from stream
    */
    void SetColorProperty(
        const eC_UInt& uiIndex,
        const ColorValue_t& kColorValue);

    /** Get the color-property for writing to stream
    @param uiIndex index of the color-property
    @return color-property for streaming
    */
    ColorValue_t GetColorProperty(const eC_UInt& uiIndex) const;

    /** Get the color-property as ARGB-value
    @param uiIndex index of the color-property
    @return color as ARGB-value
    */
    eC_UInt GetColorValue(const eC_UInt& uiIndex) const;

private:
    eC_Bool PropertyNotUsed(const eC_UInt& uiCheckIndex) const;

    void UpdateColorValue(const eC_UInt& uiIndex) const;

private:
    eC_TArray<ColorProperty_t> m_kColorProperties;
};

#endif
