/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef GUIWHEELCONTAINER_H
#define GUIWHEELCONTAINER_H

#include "GUICompositeObject.h"
#include "GUIEasing.h"
#include "GUIImageResource.h"
#include "GUINinePatch.h"

#include "GUICommonEnums.h"

// @guiliani_doxygen toplevel_control Wheel-Container
/**
<table border="0">
<tr>
<td width="200">@image html wheel_container.png</td>
<td>
The Wheel-Container can be used to group elements and to cycle through them by draging.
It is suitable if you want to have a number of navigational elements, but don't have enough space.

A specific use-case would be the menu of a smartwatch.
</td>
</tr>
</table>
*/
// @endguiliani_doxygen

/**
The controls in the Wheel-Container can also be CGUICompositeObjects to allow a wider range of use-cases.
The Snapping-Position can be used to place the currently active element on a specific position relative
to the upper-left edge of the container. Also a cyclic behaviour is possible where the child-elements
are placed on a wheel which does not have any end and repeats itsels over and over again.

@ingroup GUILIANI_CONTROLS
*/

class CGUIWheelContainer : public CGUICompositeObject
{
public:
    /** Constructor
        @param pParent Pointer to the parent object of this image.
        @param vX The x coordinate of the upper left corner of this image.
        @param vY The y coordinate of the upper left corner of this image.
        @param vWidth The width of this image.
        @param vHeight The height of this image.
        @param eID This wheel's object ID.
    */
    CGUIWheelContainer(
        CGUICompositeObject *const pParent,
        const eC_Value &vX, const eC_Value &vY,
        const eC_Value &vWidth, const eC_Value &vHeight,
        const ObjectHandle_t &eID = NO_HANDLE);

    /// Default constructor
    CGUIWheelContainer();
 
    /** Copy constructor
    @param kToBeCopied
    */
    CGUIWheelContainer(const CGUIWheelContainer& kToBeCopied);
 
    /** Assignment constructor
    @param kSource
    @return copy
    */
    CGUIWheelContainer& operator=(const CGUIWheelContainer& kSource);

    virtual ~CGUIWheelContainer();
 
    /** Adapt the wheel to its current parameters. 
        Call this after updating the range of values. */
    virtual void AdaptWheel();

    virtual eC_Bool AddObject(CGUIObject* pkObject);

    virtual void RemoveObject(CGUIObject* pkObject);

    /** Scrolls to the given entry/node index.
        @param uiNodeIndex Index of entry/node.
    */
    virtual void ScrollTo(eC_UInt uiNodeIndex);

    /** Scrolls using animations to the given entry/node value.
        @param uiNodeIndex Index of entry/node.
    */
    virtual void ScrollToAnimated(eC_UInt uiNodeIndex);

    /** GetCurrentIndex returns the current focused entry value.
        @return eC_UInt containing the current focused index.
    */
    virtual eC_UInt GetCurrentIndex();

    // Methods derived from CGUIObject 
    virtual eC_Bool DoDraw();
    virtual void DoAnimate(const eC_Value& vTimes = eC_FromInt(1));
    //virtual eC_Bool DoClick(const eC_Value &vAbsX, const eC_Value &vAbsY);
    virtual eC_Bool DoButtonDown(const eC_Value &vAbsX, const eC_Value &vAbsY);
    virtual eC_Bool DoDrag(const eC_Value &vDeltaX, const eC_Value &vDeltaY, const eC_Value &vAbsX, const eC_Value &vAbsY);
    virtual eC_Bool DoDragEnd(const eC_Value &vAbsX, const eC_Value &vAbsY);
    virtual eC_Bool SetValue(const CGUIValue& rkValue);

    virtual void SetWidth(const eC_Value &vX);
    virtual void SetHeight(const eC_Value &vY);

    /** Specifies whether the wheel is in Cyclic mode (Nodes continue from the beginning, when scrolling past the end).
        @param bCyclic True if wheel shall be cyclic, False otherwise. */
    void SetCyclic(eC_Bool bCyclic);

    /** Returns whether the wheel is in Cyclic mode (Nodes continue from the beginning, when scrolling past the end).
        @return True if wheel is cyclic, False otherwise. */
    eC_Bool IsCyclic() const;

    /** Sets the direction for the wheel
    @param eDirection
    */
    void SetDirection(const CGUICommonEnums::Orientation_t& eDirection);

    /** get the currently set direction
    @return direction
    */
    CGUICommonEnums::Orientation_t GetDirection() const;

    /** Set the snapping position 
    @param vTargetSnappingPos
    */
    void SetTargetSnappingPosition(const eC_Value& vTargetSnappingPos);

    /** get the currently set snapping position
    @return snapping-position
    */
    eC_Value GetTargetSnappingPosition() const;

    /** Sets the distance in pixels after which
    a drag-action will be executed
    @param vMinimumDragDistance
    */
    void SetMinimumDragDistance(const eC_Value& vMinimumDragDistance);

    /** Returns the minimum distance after which a drag-action is executed
    @return drag-distance
    */
    eC_Value GetMinimumDragDistance() const;

    /** Sets the distance which needs to be exceeded
    that a swipe (switch to the next element) is performed
    @param vSwipeDistance
    */
    void SetSwipeDistance(const eC_Value& vSwipeDistance);

    /** Returns the distance needed for a swipe
    @return swipe-distance
    */
    eC_Value GetSwipeDistance() const;

    /** Sets the children to centered mode or not
    @param bCenterChildren if true the children are
    centered according to the current orientation
    */
    void SetCenterChildren(const eC_Bool& bCenterChildren);

    /** Checks if children are centered
    @return true if centered
    */
    eC_Bool GetCenterChildren() const;

    /** Set the background-image
    @param eBGImage
    */
    void SetBGImage(const ImageResource_t& eBGImage);

    /** Get the background-image
    @return background-image
    */
    ImageResource_t GetBGImage() const;

    /** Set easing-parameters for easing
    @param eEasing
    @param vEasingTime
    */
    void SetEasing(
        const CGUIEasing::EasingType_t& eEasing,
        const eC_Value& vEasingTime);

    /** Get easing-parameters for easing
    @param eEasing
    @param vEasingTime
    */
    void GetEasing(
        CGUIEasing::EasingType_t& eEasing,
        eC_Value& vEasingTime) const;

#if defined(GUILIANI_STREAM_GUI)
    /** Reads all attributes from streaming file.
        This method is called by CGUIFactoryManager after one of the registered
        factories has created an instance of this class.
    */
    virtual void ReadFromStream();
#endif

#if defined(GUILIANI_WRITE_GUI)
    /** Writes all attributes to the streaming file. A CGUIStreamWriter
        has to be initialized first.
        @param bWriteClassID This flag is used to select if writing of control
               class ID, leading and trailing tags is performed.
    */
    virtual void WriteToStream(const eC_Bool bWriteClassID = false);
#endif

    /** Get the nine-patch which is used to draw the background of the wheel
        @return nine-patch
    */
    CGUINinePatch& GetNinePatch()
    {
        return m_kNinePatch;
    }

protected:
    /** Adjusts all nodes GUIText to reflect the current position. Used for non-cyclic wheels.
    */
    virtual void AdjustNodesPosition();
 
    /** Adjusts all the current node and scrolling position dependent of the DragDelta.
    */
    virtual void AdjustCurrentScrollPosAndNode();

    /** Sets the current node index via index.
        @param uiNodeIndex The current node index to set.
    */
    virtual void SetCurrentNode(eC_UInt uiNodeIndex);
    
    /** Retrieves the current node index.
        @return The current node index.
    */
    virtual eC_UInt GetCurrentNode();
    
    /** Retrieves an iterator pointing to the current node.
        @return The iterator to the current position
    */
    virtual ObjectPtrList::SafeIterator GetCurrentNodeSafeIter();
    
    /** Retrieves the index pointing to the current node.
        @param kNodeIter Iterator pointing to the position to which the index should be returned.
        @return The iterator to the current position
    */
    virtual eC_UInt GetNodeIndexFromIter(const ObjectPtrList::Iterator &kNodeIter);
    
    /** Set the current wheel position via Iterator.
        @param kNodeSafeIter Iterator pointing to the new position.
    */
    virtual void SetCurrentNodeSafeIter(const ObjectPtrList::SafeIterator &kNodeSafeIter);
     
    /** Retrieves an iterator pointing to the next node.
        @param  kIter Iterator pointing to double linked list.
        @param bWrapped true if iterator was wrapped around
        @return An iterator to the next node.
    */
    virtual ObjectPtrList::Iterator GetNextNodeCyclic(ObjectPtrList::Iterator kIter, eC_Bool& bWrapped);
     
    /** Retrieves an iterator pointing to the previous node.
        @param  kIter Iterator pointing to double linked list.
        @return An iterator to the previous node.
    */
    virtual ObjectPtrList::Iterator GetPreviousNodeCyclic(ObjectPtrList::Iterator kIter);

    /** Gets the current scrolling position dependent of direction.
        @return Value of the scrolling position.
    */
    virtual eC_Value GetCurrentScrollPosForCurrentDirection();

    /** Sets the scrolling position dependent to the direction of the wheel.
        @param vCurrentScrollPos New scrolling position horizontal or vertical.
    */
    virtual void SetCurrentScrollPosForCurrentDirection(eC_Value vCurrentScrollPos);

    /** Start animation.
    */
    virtual void StartScrollingAnimation();

    /** Stop animation.
    */
    virtual void StopScrollingAnimation();

    /** Set the current scroll-position
    @param vScrollPosition scroll position
    */
    void SetScrollPosition(const eC_Value& vScrollPosition);

    /** Calculates the index, cycle and offset (scrolling position) out of a pixel position.
    @param vOffset pixel-offset
    @param iIndex calculated index
    @param iCycle calculated cycle
    @param vScrollPos pixel-position
    */
    void GetNodeIndexForPosition(const eC_Value& vOffset, eC_Int &iIndex, eC_Int &iCycle, eC_Value &vScrollPos);

    /** Calculates the position in pixels out of entry index, cycle and offset (scrolling position).
    @param iIndex index
    @param iCycle cycle
    @param vScrollPos pixel-position
    @return pixel-offset
    */
    eC_Value GetPositionForNodeIndex(const eC_Int &iIndex, const eC_Int &iCycle, const eC_Value &vScrollPos);

private:
    void Init();
    void DeInit();

    eC_Value GetTotalSize();

private:
    ObjectPtrList::SafeIterator m_kCurrentNodeSafeIter;///< Iterator pointing to the current node/entry.
    eC_UInt m_uiCurrentNodeIndex; ///< Index of current node.
    eC_Value m_vCurrentScrollPosX; ///< Scrolling position of current node (offset in pixels (0-m_vNodeDefaultWidth)).
    eC_Value m_vCurrentScrollPosY;  ///< Scrolling position of current node (offset in pixels (0-m_vNodeDefaultHeight)).

    eC_Value m_vTargetPos;///< Target position in pixels for animation

    // Used for animation
    // Animation start position
    eC_Int m_iAniStartNodeIndex;///< Animation start node index.
    eC_Int m_iAniStartNodeCycle;///< Animation start node cycle.
    eC_Value m_vAniStartScrollPos;///< Animation start scrolling position
    // Animation target position
    eC_Int m_iAniTargetNodeIndex;///< Animation target node.
    eC_Int m_iAniTargetNodeCycle;///< Animation target node cycle.
    eC_Value m_vAniTargetScrollPos;///< Animation target scrolling position.

    eC_Bool m_bCyclic;///< True if wheel is cyclic. False if non-cyclic. The wheel then stops at begin/end of entries.

    eC_UInt m_uiDragTimeStamp;///< Time at start of animation and Timestamp of last received DoDrag()-Event
    eC_UInt m_uiDragStarted; ///< Holds the time when the last drag was started. Used to calculate the drag speed.
    CGUIPoint m_kDragStart; ///< Holds the start point of a drag.
    eC_Value m_vDistance; ///< distance from initial position

    ImageResource_t  m_eBGImage; ///< Used background image

    CGUICommonEnums::Orientation_t m_eDirection; ///< Direction of the wheel. Either horizontal or vertical.

    /// EasingType used for kinetic animation
    CGUIEasing::EasingType_t m_eEasingType; ///< Easing used for all animations which will not go further then the wheel end.

    /// Duration of easing animation for kinetic scrolling
    eC_Value m_vEasingDuration; ///< Easing duration used for all animations which will not go further then the wheel end.

    CGUINinePatch m_kNinePatch; ///< nine-patch for the background-image

    eC_Value m_vTargetSnappingPos; ///< position for the selected child-element
    eC_Value m_vMinimumDragDistance; ///< minimum distance to be exceeded to start a drag
    eC_Value m_vSwipeDistance; ///< distance which will trigger a swipe if exceeded

    eC_Value m_vScrollPos;

    eC_Bool m_bCenterChildren;
};

#endif // GUIWHEEL_H
