/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef __GUIVIDEO_H__
#define __GUIVIDEO_H__

#include "GUIObject.h"

#include "GUIGeneralResource.h"
#include "GUIImageResource.h"
#include "GUIVideoDecoder.h"
#include "GUIVideoData.h"

/**
@brief This control displays video-files
*/
class CGUIVideo : public CGUIObject
{
public:
    /** constructor.
        @param pParent Pointer to the designated parent object.
        @param vX X-position relative to its parent object, i.e. the x-offset from the left border of the parent object
        @param vY Y-position relative to its parent object, i.e. the y-offset from the upper border of the parent object
        @param vWidth Width of the object
        @param vHeight Height of the object
        @param eID Object identifier of this object (choose NO_HANDLE if none is required).
    */
    CGUIVideo(
        CGUICompositeObject *const pParent,
        const eC_Value &vX, const eC_Value &vY,
        const eC_Value &vWidth, const eC_Value &vHeight,
        const ObjectHandle_t &eID = NO_HANDLE);

    ~CGUIVideo();

    /** Start movie
    */
    void Play();

    /** Pause playing
    */
    void Pause();

    /** Resume paused movie
    */
    void Resume();

    /** Stop playing
    */
    void Stop();

    /** Load movie file
        @param kFileName name of file
    */
    void SetMovie(const eC_String& kFileName);

    /** Load movie file
        @param eVideoID resource-id
    */
    void SetMovie(const GeneralResource_t& eVideoID);

    /** Set start-frame for video
    @param uiStartFrame
    */
    void SetStartFrame(const eC_UInt& uiStartFrame);

    /** Get start-frame
    @return start-frame
    */
    eC_UInt GetStartFrame() const;

    /** Set if video will be automatically started
    @param bAutoStart
    */
    void SetAutoStart(const eC_Bool& bAutoStart);

    /** Return if video will be automatically started
    @return auto-start
    */
    eC_Bool GetAutoStart() const;

    /** Set if video will be looped
    @param bLoop if true video will be looped
    */
    void SetLoop(const eC_Bool& bLoop);

    /** Return whether video will be looped
    @return true if video is looped
    */
    eC_Bool GetLoop() const;

    /** goto specified frame in video
        @param uiFrameNumber
    */
    void GotoFrame(const eC_UInt& uiFrameNumber);

    /** Get currently set frame
        @return current frame
    */
    eC_UInt GetCurrentFrame() const;

    /** Return the width in pixel for the original video-data
    @return video-width
    */
    eC_UInt GetVideoWidth() const;

    /** Return the height in pixel for the original video-data
    @return video-height
    */
    eC_UInt GetVideoHeight() const;

    /** Return the number of bytes per pixel for the original video-data
    @return video-bpp
    */
    eC_UInt GetVideoBytePerPixel() const;

    /** Return the number of milliseconds for the video
    @return length of the video-data in MS
    */
    eC_UInt GetLengthInMS() const;

    /** Return the number of frames for the video
    @return length of the video-data in frames
    */
    eC_UInt GetLengthInFrames() const;

    virtual void DoAnimate(const eC_Value &vTimes = eC_FromInt(1));

    virtual eC_Bool DoDraw();

    virtual eC_Bool IsHighlightable() const { return false; }

#ifdef GUILIANI_STREAM_GUI
    /** Standard constructor. Only to be called by factory.
    No user code should call this constructor, not even in
    streaming mode (that is, when GUILIANI_STREAM_GUI is defined)!
    @see CGUIVideo() */
    CGUIVideo();

    virtual void ReadFromStream();
#endif

#ifdef GUILIANI_WRITE_GUI
    virtual void WriteToStream(const eC_Bool bWriteClassID = false);
#endif

private:
    void Init();

    void DeInit();

private:
    GeneralResource_t m_eVideoID; ///< id of the video-resource
    eC_UInt m_uiStartFrame; ///< starting-frame for the video
    eC_Bool m_bAutoStart; ///< should be auto-started after loading?
    eC_Bool m_bLoop; ///< loop-mode

    CGUIVideoData* m_pkVideoData; ///< current video-data
    CGUIVideoDecoder* m_pkDecoder; ///< currently used video-decoder
    eC_UByte* m_pkPixelBuffer; ///< buffer for image-data
    ImageResource_t m_eImageID; ///< image-id which is used to display video-frame
    eC_Value m_vCurrentTime; ///< currently expired time of video
};

#endif
