/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#if !defined(GUITABLEVIEW_H)
#define GUITABLEVIEW_H

#include "GUIScrollView.h"
#include "GUIModelIndex.h"
#include "eC_TList_doubleLinked.h"
#include "GUIViewItemGenerator.h"
#include "SafeGUIObjectPtr.h"

class CGUIViewItem;
class CGUIHeaderItem;
class CGUITableModel;
class CGUIRepositionCompositeObject;
class CGUIViewItemGenerator;
class CGUIViewItemTreeElement;

///@brief A view for instances of CGUITableModel.
// @guiliani_doxygen toplevel_control Table View
/**
    <table border="0">
        <tr>
            <td width="200">@image html table_view.png</td>
            <td>
                The "table view" control is used for the presentation of charts. Dynamic count of columns and rows are supported (Class: CGUITableView).
            </td>
        </tr>
    </table>
*/
// @endguiliani_doxygen
/**
The table view provides a dynamic view upon a table model.
It manages the actual GUI objects that display the data of a table.

Since it is a specialization of CGUIScrollView, its contents are
automatically scrolled. Specifically, the vertical scroll bar only scrolls
the table's rows without the header.

Typical usage is as follows:
- Create an instance of this class.
- Set ViewItemGenerators to specify the visual representation of the data.
- Append view columns (AppendColumn or InsertColumn).
- Attach a model by calling SetModel. At this point, all data that is
already present in the model is shown in the view, and later changes to
the model are also reflected in the view.

The GUIObjects for each cell are provided by CGUIViewItems. A ViewItem provides a ViewObject
that is added to the table view. The type of the object that will be created is determined by the ViewItem.
The ViewItem is also responsible to inform the View when the Value of the ViewObject was edited and provides
an interface to set a new value in the ViewObject. When a cell is selected, the ViewItem is informed to be
able to set the ViewObject (GUIObject) to a selected representation (if needed).
The TableView itself does not draw a "Selection Frame" around a selected cell. The ViewObject itself is responsible
for its selected representation.

ViewItems are generated by CGUIViewItemGenerators. Depending of the ViewIndex the ViewItemGenerator
has to create the desired ViewItem. If the ViewItem itself is not editable (@see CGUIViewItem::IsEditable),
the generator might provide an EditorObject which is displayed by the TableView if the editing mode for a cell is active.

Example:
@code

    // Create a table view.
    CGUITableView* pkTableView = new CGUITableView(this,
                                        eC_FromInt(0), eC_FromInt(0),
                                        eC_FromInt(250), eC_FromInt(200),
                                        OBJ_TABLEVIEW);

    // Add columns to the view and map it to the model columns.
    pkTableView->AppendColumn(0, eC_FromInt(20));
    pkTableView->AppendColumn(1, eC_FromInt(150));
    pkTableView->AppendColumn(2, eC_FromInt(80));

    // In the first column check boxes shall be displayed, so set a ViewItemGeneratorCheckBox.
    pkTableView->SetDefaultColumnViewGenerator(0, new CGUIViewItemGeneratorCheckBox());

    // Create  a table model.
    CGUITableModel* pkModel = new CGUITableModel();
    pkModel->AddColumn("Flag");
    pkModel->AddColumn("Name");
    pkModel->AddColumn("Value");

    // Set the model to be displayed by the view.
    pkTableView->SetModel(pkModel);
    // Add a row to the model.
    eC_UInt uiRow = pkModel->AppendRow();
    pkModel->SetItem(uiRow, 0, new CGUIModelItem(true));
    pkModel->SetItem(uiRow, 1, new CGUIModelItem("first row"));
    pkModel->SetItem(uiRow, 2, new CGUIModelItem(CGUIValue(1.0)));
    uiRow = pkModel->AppendRow();
    pkModel->SetItem(uiRow, 0, new CGUIModelItem(false));
    pkModel->SetItem(uiRow, 1, new CGUIModelItem("second row"));
    pkModel->SetItem(uiRow, 2, new CGUIModelItem(CGUIValue(2.0)));

    // Set some view attributes.
    pkTableView->SetHeaderColor(0xFFFF0000);
    pkTableView->SetGridLineWidth(eC_FromInt(1));
    pkTableView->EnableGrid(true, true);
    pkTableView->SetSelectionMode(CGUITableView::MULTI_SELECTION);
@endcode

@ingroup GUILIANI_CONTROLS
*/
class CGUITableView : public CGUIScrollView
{
friend class CGUIViewItem; ///< @todo to access private functions
friend class CGUIHeaderItem; ///< @todo to access private functions

public:
    /** Constructs a new table view.
    @param pParent Pointer to the designated parent object.
    @param vX X position relative to parent object.
    @param vY Y position relative to parent object.
    @param vWidth Width of the table view.
    @param vHeight Height of the table view.
    @param eID Object handle of this table view.
    */
    CGUITableView(
        CGUICompositeObject *const pParent,
        const eC_Value &vX, const eC_Value &vY,
        const eC_Value &vWidth, const eC_Value &vHeight,
        const ObjectHandle_t &eID = NO_HANDLE);

    /** Constructs a new table view.
    @param pParent Pointer to the designated parent object.
    @param rkRect Bounding rectangle relative to parent.
    @param eID Object handle of this table view.
    */
    CGUITableView(
        CGUICompositeObject *const pParent,
        const CGUIRect& rkRect,
        const ObjectHandle_t &eID = NO_HANDLE);

    /** Destructor*/
    ~CGUITableView();

    /** Make sure that the header is a child of the table view
        and is not added to the container of the scroll view.
        So the header is not scrolled by the ScrollView. All other
        objects are added into the ViewObjectContainer.
        @param pObject The object to add
        @return True if successful, False otherwise
    */
    eC_Bool AddObject(CGUIObject* pObject);

    /** Sets a model whose data is to be displayed by this view.
        @param pkModel Pointer to the model to be displayed. Pass NULL to clear
        the view.
    */
    void SetModel(CGUITableModel* pkModel);

    /// Triggers that can be used to select or edit an item.
    enum TriggerEvent_t
    {
        TRIGGER_NONE = 0,
        TRIGGER_CLICK = 1,
        TRIGGER_DOUBLE_CLICK = 2,
        TRIGGER_LONG_CLICK = 4,
        //TRIGGER_MOUSE_MOVE = 8,
        TRIGGER_GK_ACTION = 16,
        //SELECTION_FOUR_WAY_NAVIGATION = 32,
    };

    /** Establishes a mapping from a model column to a display (view) column.
        @param uiColumnIndex The column Index in the view.
        @param uiModelColumnIndex Index of the column in the model whose items
               are displayed in the new column.
        @param vWidth Initial width of the new column.
        @param vMinWidth The minimum width of the column. If a column is resized,
               this width is used as a limit. The default value is 10 pixel. Note if vMinWidth
               is greater than vWidth the column resize mode is disabled.
        @param eSelectionTrigger Determines with which trigger a ViewItem shall
               be selected, which means it is added to the selection list.
        @param eEditTrigger Determines with which trigger the table view will
               start the edit mode. If the ViewItem itself is not editable,
               the table view needs to start the edit mode, which it will only
               do if eEditTrigger is triggered.

               Note that view items which are editable themselves can always
               update the view. For example if a CheckBox is clicked, the View
               will be informed about the new value even if the trigger for
               editing is a double click.
        @see EditItem.
        @return True if the column was successfully inserted, otherwise False.
    */
    virtual eC_Bool InsertColumn(eC_UInt uiColumnIndex, eC_UInt uiModelColumnIndex, eC_Value vWidth, eC_Value vMinWidth = 10, TriggerEvent_t eSelectionTrigger = TRIGGER_CLICK, TriggerEvent_t eEditTrigger = TRIGGER_DOUBLE_CLICK);

    /** Establishes a mapping from a model column to a display (view) column.
        A column is appended at the right end of the view table. It displays
        the data of the model column with the given index.
        @param uiModelColumnIndex Index of the column in the model whose items
               are displayed in the new column.
        @param vWidth Initial width of the new column.
        @param vMinWidth The minimum width of the column. If a column is resized,
               this width is used as a limit. The default value is 10 pixel. Note if vMinWidth
               is greater than vWidth the column resize mode is disabled.
        @param eSelectionTrigger Determines with which trigger a ViewItem shall
               be selected, which means it is added to the selection list.
        @param eEditTrigger Determines with which trigger the table view will
               start the edit mode. If the ViewItem itself is not editable,
               the table view needs to start the edit mode, which it will only
               do if eEditTrigger is triggered.

               Note that view items which are editable themselves can always
               update the view. For example if a CheckBox is clicked, the View
               will be informed about the new value even if the trigger for
               editing is a double click.
        @see EditItem.
        @return True if the column was successfully appended, otherwise False.
    */
    eC_Bool AppendColumn(eC_UInt uiModelColumnIndex, eC_Value vWidth, eC_Value vMinWidth = 10, TriggerEvent_t eSelectionTrigger = TRIGGER_CLICK, TriggerEvent_t eEditTrigger = TRIGGER_DOUBLE_CLICK);

    /** Removes the column with given index from the view.
    @param uiColumnIndex The index of the column to remove.
    @return True if the column was successfully removed, else False.
    */
    virtual eC_Bool RemoveColumn(eC_UInt uiColumnIndex);

    /** Gets the current column count of the view.
    @return const eC_UInt The column count of the view.
    */
    inline eC_UInt GetColumnCount()
    {
        return m_kColInfos.GetQuantity();
    }

    /**
        Check if a ModelColumn is contained in the view.
        @param uiModelColumnIndex The Model column to check.
        @return True if the column is contained, False else.
        @see InsertColumn.
        @see AppendColumn.
    */
    eC_Bool ModelColumnContainedInView(const eC_UInt uiModelColumnIndex) const;

    /** Sets the width of a view column.
    @param uiColumn Index of the column in the view.
    @param vWidth New width of the column.
    @param cbAdjust Enable or disable automatic alignment. The default value is true.
    @return True if the column width was successfully set, False else.
    */
    eC_Bool SetColumnWidth(eC_UInt uiColumn, eC_Value vWidth, const eC_Bool cbAdjust=true);

    /** Returns the width of a view column.
        @param uiColumn Index of the view column.
        @return Width of the column, or 0 if the index is out of range.
      */
    eC_Value GetColumnWidth(eC_UInt uiColumn) const;

 #ifdef GUILIANI_STREAM_GUI
    /** Standard constructor. Only to be called by factory.
        No user code should call this constructor, not even in
        streaming mode (that is, when GUILIANI_STREAM_GUI is defined)!
    */
    CGUITableView();

    virtual void ReadFromStream();
#endif

#ifdef GUILIANI_WRITE_GUI
    /** Since the content (children) depend on the model, all children are
        removed from the row and header containers before calling
        the base class implementation. Afterwards, the children are added
        again.
        @copydoc CGUIStreamableObject::WriteToStream
      */
    virtual void WriteToStream(const eC_Bool bWriteClassID = false);
#endif


    /// @return The height of the table rows. Currently the same for each row.
    inline eC_Value GetRowHeight() const { return m_vRowHeight; }

    /** Set the height of the table rows. Currently the same for each row.
        The view is automatically adjusted.
        @param crvRowHeight The row height to set.
        @see GetRowHeight()
        @return True if successful, False otherwise
    */
    eC_Bool SetRowHeight(const eC_Value &crvRowHeight);

    /** Changes the font of the header items.
    @param eHeaderTextFont The new font in ARGB format.
    */
    void SetHeaderTextFont(const FontResource_t& eHeaderTextFont);

    /** Changes the background color of the header items.
    @param uiColor The new color in ARGB format.
    */
    void SetHeaderTextColor(eC_UInt uiColor);

    /** Changes the background color of the header items.
        @param uiColor The new color in ARGB format.
      */
    void SetHeaderColor(eC_UInt uiColor);

    /** Changes the color of the grid lines.
        @param uiColor The new color in ARGB format.
      */
    void SetGridColor(eC_UInt uiColor);

    /// @return The current color of the grid lines.
    inline eC_UInt GetGridColor() const { return m_uiGridColor; }

    /** Changes the width of the grid lines.
        The view is automatically adjusted.
        @param vGridLineWidth The width of the grid lines in pixel.
      */
    void SetGridLineWidth(eC_Value vGridLineWidth);

    /// @return The current color of the grid lines.
    inline eC_Value GetGridLineWidth() const { return m_vGridLineWidth; }

    /// @return The current text-font of the header items.
    inline FontResource_t GetHeaderTextFont() const
    {
        return m_eHeaderTextFont;
    }

    /// @return The current text-color of the header items.
    inline eC_UInt GetHeaderTextColor() const
    {
        return m_uiHeaderTextColor;
    }

    /// @return The current background color of the header items.
    inline eC_UInt GetHeaderColor() const { return m_uiHeaderBkgColor; }

    /** Enables drawing of grid lines between table cells.
        @param bHorizontal Whether horizontal lines are drawn.
        @param bVertical Whether vertical lines are drawn.
      */
    void EnableGrid(eC_Bool bHorizontal, eC_Bool bVertical);

    /** Changes the height of the header area.
        The header is automatically adjusted.
        @param vHeaderHeight The new height of the header area.
      */
    void SetHeaderHeight(eC_Value vHeaderHeight);

    /** Get the height of the Header.
        @return The height of the header area.
      */
    eC_Value GetHeaderHeight() const {return m_vHeaderHeight;}

   /** Resizes the table and all columns within the auto resized list.
    @param crvHeight The new height of the table.
    @see SetAutoResizedColumn()
    @see AdjustAutoresizedColumn()
    */
    void SetHeight(const eC_Value &crvHeight);

    /** Resizes the table and all columns within the auto resized list.
    @param crvWidth The new width of the table.
    @see SetAutoResizedColumn()
    @see AdjustAutoresizedColumn()
    */
    void SetWidth(const eC_Value &crvWidth);

    /**
        Set a column index to auto resize column list. The columns within the list are automatically resized
        when the table is resized. Columns will never get smaller than their minimum size.
        @param cuiColumn The column index to add.
        @see AdjustAutoresizedColumn()
    */
    void SetAutoResizedColumn(const eC_UInt cuiColumn);

    /**
        Remove a column index from auto resize column list.
        @param cuiColumn The column index to remove.
        @see AdjustAutoresizedColumn()
    */
    void RemoveAutoResizedColumn(const eC_UInt cuiColumn);

    virtual eC_Bool DoDraw();

    /** set a default view item generator for the table.
    @param pkViewItemGenerator
    */
    void SetDefaultViewGenerator(CGUIViewItemGenerator* pkViewItemGenerator);

    /** set a default view item generator for a column
    @param uiColumn
    @param pkViewItemGenerator
    */
    void SetDefaultColumnViewGenerator(const eC_UInt uiColumn, CGUIViewItemGenerator* pkViewItemGenerator);
    
    /** set a view item generator for a cell.
    @param rkIndex
    @param pkViewItemGenerator
    */
    void SetViewGenerator(const CGUIViewIndex& rkIndex, CGUIViewItemGenerator* pkViewItemGenerator);

    virtual eC_Bool DoClick(const eC_Value &vAbsX = eC_FromInt(-1), const eC_Value &vAbsY = eC_FromInt(-1));

    virtual eC_Bool DoDoubleClick(const eC_Value &vAbsX = eC_FromInt(-1), const eC_Value &vAbsY = eC_FromInt(-1));

    virtual eC_Bool DoLongClick(const eC_Value &vAbsX = eC_FromInt(-1), const eC_Value &vAbsY = eC_FromInt(-1));

    virtual eC_Bool DoKeyDown(
        const GUIKeyIdentifier_t &eKeyIdentifier,
        const eC_UInt &uiModifiers);

    /// Enumeration used for the SelectionMode.
    enum SelectionMode_t
    {
        NO_SELECTION, ///< No items in the view can be selected.
        SINGLE_SELECTION, ///< Only one cell in the view can be selected.
        //CONTINUOUS_SELECTION, ///< [Not yet implemented]Only continuous blocks can be selected.
        MULTI_SELECTION,       ///< Free selection of cells is possible.
        SELECT_ROW            ///< If one item in a row is selected / de-selected, the tableview will select / de-select the complete row. Multi Row selection is possible.
    };

    /**
        Indicates if a cell is selected or not.
        @param rkIndex The Index of the cell / the ViewItem to check.
        @return True if selected, False else.
    */
    eC_Bool IsItemSelected(const CGUIViewIndex& rkIndex);

    /**
        Set the SelectionMode for the View.
        The selection mode is checked when a selection update based on user input is triggered.
        When using the public API to select items, the selection mode is not checked.
        @param eSelectionMode The allowed selection.
    */
    void SetSelectionMode(SelectionMode_t eSelectionMode) { m_eSelectionMode = eSelectionMode; }

    /**
    Regenerate a view item for a cell. This is needed when a new generator was set.
    The ViewItemGenerator will be triggered to generate a ViewItem.
    The old ViewItem and its ViewObject will be deleted.
    @param rkIndex The ViewIndex of the ViewItem to regenerate.
    */
    void RegenerateViewItem(const CGUIViewIndex& rkIndex);

    /** Regenerate all view items of the column with index of parameter cuiColumn.
    This is needed when a new generator was set.
    The ViewItemGenerator will be triggered to generate the ViewItems.
    The old ViewItem and its ViewObject will be deleted.
    @param cuiColumn The column index for re-generation.
    */
    void RegenerateViewItems(const eC_UInt cuiColumn);

    /**
    Regenerate all view items of each column its index is equal to and greater than the index of parameter cuiColumn.
    The ViewItemGenerator will be triggered to generate the ViewItems. All old ViewItems and its ViewObject will be deleted.
    @param cuiColumn The column index for starting re-generation.
    */
    void RegenerateViewItemsFromColumn(const eC_UInt cuiColumn);

    /**
     If the item is selectable (ItemIsSelectable()) this method
     selects an item.
     @note This function does not clear the old selection before selecting the item.
     @param rkIndex The Index of the Item to select.
     @return True if the item was successfully selected, otherwise False.
     */
    eC_Bool SelectItem(const CGUIViewIndex& rkIndex);

    /// The Selection list holds the indices of all selected objects.
    typedef eC_TListDoubleLinked<CGUIViewIndex> SelectionList;

    /// The auto resize column list holds the indices of column auto resize candidates
    typedef eC_TListDoubleLinked<eC_UInt> AutoResizeColumnList;

    /**
        Select a list of items.
        @note This function does not clear the old selection before selecting the items.
        @see SelectItem
        @param rkItemList The list of ViewItems to be selected.
        @return False if at least one of the items could not be selected, otherwise True.
    */
    eC_Bool SelectItems(const SelectionList& rkItemList);

    /**
        Select a row.
        @note This function does not clear the old selection before selecting the row.
        @param uiRow The row to select.
        @return False if at least one of the row's items could not be selected, otherwise True.
    */
    eC_Bool SelectRow(const eC_UInt uiRow);

    /**
        Deselects a row.
        @param uiRow The row to select.
        @return False if at least one of the row's items could not be deselected, otherwise True.
    */
    eC_Bool DeselectRow(const eC_UInt uiRow);

    /**
        Select a column.
        @note This function does not clear the old selection before selecting the column.
        @param uiColumn The column to select.
        @return False if at lease one of the column's items could not be selected, otherwise True.
    */
    eC_Bool SelectColumn(const eC_UInt uiColumn);

    /**
        Deselects all ViewItems that are currently selected.
        @param bNotifyObservers Specifies whether a notification for SelectionObservers is triggered. Defaults to TRUE.
        @see CGUIViewItem::SetSelected.
    */
    void ClearSelection(const eC_Bool bNotifyObservers = true);

    /** Retrieve the selection list.
        @return Reference to the selection list.
    */
    const SelectionList& GetSelection(){return m_kSelectionList;}

    /** Map ModelIndex to ViewIndex.
    @param rkIndex The ModelIndex.
    @return The ViewIndex.
    */
    CGUIViewIndex GetViewIndex(const CGUIModelIndex& rkIndex) const;

    /** Map ViewIndex to ModelIndex
    @param rkIndex The ViewIndex.
    @return The ModelIndex.
    */
    CGUIModelIndex GetModelIndex(const CGUIViewIndex& rkIndex) const;

    /**
        Get the pointer to the currently used table model.
        @return The Pointer to the table model or NULL if no Model was set in the view.
    */
    CGUITableModel* GetModel() {return m_pkModel;}

    /** Returns the bounding rect for a table cell based on the view index.
    @param rkViewIndex The ViewIndex.
    @return the bounding rect of the table cell, relative to the ViewObjectContainer.
    */
    CGUIRect GetCellRect(const CGUIViewIndex& rkViewIndex);

    /**
        Hide a row or make it visible again.
        The row is no longer displayed when it is hidden. The following rows are moved up.
        @param uiRowIndex The index of the row to hide / show again.
        @param bHide True to hide a row, false to show it again.
        @return True in case of success, False else.
    */
    eC_Bool HideRow(const eC_UInt uiRowIndex, const eC_Bool bHide = true);

    /**
        Hide a range of rows or make them visible again.
        @see HideRow().
        @param uiStartRowIndex The first row of the range.
        @param uiEndRowIndex The last row of the range (included).
        @param bHide Set to true to hide rows, false to make them visible again.
        @return True in case of success, False else.
    */
    eC_Bool HideRows(const eC_UInt uiStartRowIndex, const eC_UInt uiEndRowIndex, eC_Bool bHide = true);

    /**
        Check if a row is hidden or not.
        @param uiRowIndex The index of the row to check.
        @return True if the row is hidden, False if not.
    */
    eC_Bool IsRowHidden(const eC_UInt uiRowIndex) const;

    /**
       Update the scroll view to make the recently selected table item visible.
    */
    void EnsureRecentlySelectedIsVisible();

    /**
       Update the scroll view to make the given index visible.
       @param rkViewIndex The view index that should become visible.
    */
    void EnsureItemIsVisible(const CGUIViewIndex& rkViewIndex);

    /**
       Set the left and right gap between view items and grid.
       The view items are automatically repositioned.
       @param crvViewItemsGap The gap between view items.
    */
    void SetViewItemsGap(const eC_Value& crvViewItemsGap);

    /**
        Set the top and bottom gap between view items and grid.
        The view items are automatically repositioned.
        @param crvViewItemsGapVertical The gap between view items.
    */
    void SetViewItemsGapVertical(const eC_Value& crvViewItemsGapVertical);

protected:
    /// Contains information about one row.
    struct RowInfo
    {
    /// Standard constructor
        RowInfo():
            m_bVisible(true),
            m_bExpanded(true)
        {
        }
        eC_Bool m_bVisible; ///< Indicates if the row is visible or not.
        eC_Bool m_bExpanded; ///< Indicates in tree view, if the sub tree of the row is expanded
    };
    /// The model needs to call the protected Insert/Remove/Update methods.
    friend class CGUITableModel; ///< @todo to access private functions

    /** Internal helper for selection handling. Clears the selection.
        This method does not trigger SelectionUpdated();
    */
    void RemoveAllItemsFromSelection();

    /** Helper function to make sure that grid lines are redrawn. Invalidates the grid line between uiRow -1 and uiRow.
    @param uiRow
    */
    void InvalidateAreaOfHorizontalLine(const eC_UInt& uiRow);

    /** Helper function for click and keyboard events. Checks if the trigger is allowed and selects the item / starts editing.
    @param vAbsX x-position
    @param vAbsY y-position
    @param eTriggerEvent trigger
    @param uiModifiers modifiers
    @return true on success
    */
    eC_Bool HandleTriggerEvent(
        const eC_Value &vAbsX,
        const eC_Value &vAbsY,
        const TriggerEvent_t& eTriggerEvent,
        eC_UInt uiModifiers);

    /** Called by the model to update the view item with new data.
    @param crkModelIndex The ModelIndex.
    @param crkNewValue The new value.
    */
    void UpdateViewItem(const CGUIModelIndex& crkModelIndex, const CGUIValue& crkNewValue);

    /** Callback called whenever a view item was updated.
        ViewItems use this callback to inform the view about changed values.
        @param rkIndex The ViewIndex
        @param kNewValue The new value.
    */
    virtual void ViewItemUpdated(const CGUIViewIndex& rkIndex, const CGUIValue& kNewValue){}

    /** Calculates the ViewIndex for a position.
    This is done based on column width and row height.
    @param vAbsX The absolute X position.
    @param vAbsY The absolute Y position.
    @return The ViewIndex of the ViewItem.
    */
    CGUIViewIndex GetViewIndexByPosition(const eC_Value &vAbsX, const eC_Value &vAbsY);

    /** Returns the XPosition of a column. Used for item positioning.
        @param uiColumnIndex
        @return The X position for the column (relative to the table position). -1 if the column index was invalid.
    */
    eC_Value GetColumnXPos(const eC_UInt& uiColumnIndex);

    /**
        Returns the YPosition of a row relative to the ViewItemContainer. Used for item positioning.
        Calculated based on RowHeight.
        @param uiRowIndex The Index of the row.
        @return The Y position of the row (relative to the table position). -1 if the row index was invalid.
    */
    eC_Value GetRowYPos(const eC_UInt& uiRowIndex);

    /** Starts item editing. If the view item itself is not editable, an editor object is generated.
    The editor object is provided by the CGUIViewItemGenerator.
    @see CGUIViewITemGenerator GetEditorObject.
    @param rkIndexToEdit The index of the ViewItem that shall be edited.
    */
    void EditItem(const CGUIViewIndex& rkIndexToEdit);

    /** Called by the ViewItems if the value has changed.
        The model will be updated with the new value.
        @param rkIndex The ViewIndex of the updated Item.
        @param kNewValue The new Value.
    */
    virtual void ItemEdited(const CGUIViewIndex& rkIndex, const CGUIValue& kNewValue);

    /** Hides editor object (if currently visible).
    @param bMoveFocusToViewItem If true the table view will try to set the focus to the currently edited ViewObject.
                                Set this parameter to false, if the view item will be deleted in the next step.
    */
    void StopEditing(eC_Bool bMoveFocusToViewItem = true);

    /** Updates the position of a view object based on the ViewIndex and sets the TableView as parent.
        @param pViewItem The ViewItem that shall be re-positioned. */
    virtual void RepositionViewItem(CGUIViewItem* pViewItem);

    /** Updates the position of a view object based on the ViewIndex and sets the TableView as parent.
    The method has enhanced performance to \ref RepositionViewItem. However, it is needed to call the method
    iteratively with subsequent ViewItems and the previously calculated Number of visible rows.
    @code{.cpp}
    ColumnInfoList::Iterator itCol = m_kColInfos.GetAt(0);
    ColumnItemList::Iterator itRow;
    eC_UInt uiNofVisibleRows = 0;
    RowInfoList::Iterator kRowInfoIter = m_kRowInfos.GetBegin();
    FOR_ALL_FORWARD(itRow, (*itCol).m_kViewItems)
    {
      RepositionViewItemIterative(*itRow, uiNofVisibleRows);
    }
    @endcode
    @param pViewItem The ViewItem that shall be re-positioned. 
    @param pkRowInfo Info structure for row to process, might be NULL
    @param[in,out] uiNofVisibleRows Number of visible rows */
    void RepositionViewItemIterative(CGUIViewItem* pViewItem, const RowInfo* pkRowInfo, eC_UInt& uiNofVisibleRows);
    /**
        Helper function for SetWidth() and SetAutoResizedColumn()
    */
    void AdjustAutoresizedColumn();

    /**
        Resizes the ViewObjectContainer and calls CGUISrollView::AdjustScrollBar
    */
    virtual void AdjustViewObjectContainer();

    /** Updates the position of all view items of each column its index is equal to and greater than the parameter uiColumn.
    Used for resizing of columns.
    @param cuiColumn The column index to start with re-positioning.
    */
    virtual void RepositionViewItemsFromColumn(const eC_UInt cuiColumn);

    /** Updates the position of all view items of the column with index of parameter uiColumn.
    Used for resizing of columns.
    @param cuiColumn The column index for re-positioning.
    */
    virtual void RepositionViewItems(const eC_UInt cuiColumn);

    /** Updates the position of all view items of each row >= uiRow.
    Used for resizing of columns.
    @param uiRow The row index to start with re-positioning.
    */
    virtual void RepositionViewItemsFromRow(const eC_UInt uiRow);

    /** Returns the ViewItem for a ViewIndex.
    @param rkIndex The index of the ViewItem.
    @return Pointer to the ViewItem or NULL if no ViewItem was generated (yet).
    */
    CGUIViewItem* GetViewItem(const CGUIViewIndex& rkIndex);

    /** Implements Dependencies of View Item Rectangle from Model
    @param CellRect Input and Output parameter containing the original and then modified rectangle.
    @param pItem The corresponding ViewItem.
    */
    virtual void ModifyCellRect(CGUIRect & CellRect, CGUIViewItem * pItem){}

    /** Notifies this view that its model was changed. Do not call this method
    from user code.
    @param uiStartRow Index of the first inserted row.
    @param uiEndRow Index of the row after the last inserted one.
    */
    void InsertRows(eC_UInt uiStartRow, eC_UInt uiEndRow);

    /** Notifies this view that its model was changed. Do not call this method
    from user code.
    @param uiStartRow Index of the first removed row.
    @param uiEndRow Index of the row after the last removed one.
    */
    void RemoveRows(eC_UInt uiStartRow, eC_UInt uiEndRow);

    /** Helper for adding a header item.
    @param uiModelColumnIndex Index of the column in the model.
    @param uiIndex Index of the view column (position of the header item).
    */
    void AddHeaderItem(eC_UInt uiModelColumnIndex, eC_UInt uiIndex);

    /** Helper for getting a header item.
    @param uiIndex Index of the view column (position of the header item).
    @return the header-item
    */
    CGUIHeaderItem* GetHeaderItem(eC_UInt uiIndex);

    /** Helper for removing a header item.
    @param uiIndex The index of the view column (position of the header item).
    */
    void RemoveHeaderItem(eC_UInt uiIndex);

    /** Changes the texts on existing header items and updates their width based on the column width.
    Call this if a new model has been set or the column width was changed.
    */
    void UpdateHeaderItems();

    /// Container for header items.
    CGUIRepositionCompositeObject* m_pkHeader;

    /// Current model for this view.
    CGUITableModel* m_pkModel;

    /// The currently edited view item.
    CGUIViewItem* m_pkEditedViewItem;

    /// The current editor object.
    CSafeGUIObjectPtr m_pkCurrentEditorObject;

    /// list holds all view items of one column.
    typedef eC_TListDoubleLinked<CGUIViewItem*> ColumnItemList;

    /// Contains information about one column.
    struct ColumnInfo
    {
        ColumnItemList m_kViewItems; ///< all the elements of the column.
        eC_UInt m_uiModelColumnIndex; ///< Column index in the model.
        eC_Value m_vWidth; ///< Visual width of the column.
        eC_Value m_vMinWidth; ///< The minimum width of the column.
        TriggerEvent_t m_eSelectionTrigger; ///< Indicates if and how items of a column are selectable.
        TriggerEvent_t m_eEditTrigger; ///< Indicates if and how items of a column are editable.
        eC_Bool m_bVisible; ///< Indicates if the column is visible or not.
    };

    /// A column info list is a double-linked list of column info objects
    typedef eC_TListDoubleLinked<ColumnInfo> ColumnInfoList;

    /// Holds information about the columns and the ViewItem list.
    ColumnInfoList m_kColInfos;

    /// A row info list is a double-linked list of row info objects
    typedef eC_TListDoubleLinked<RowInfo> RowInfoList;

    /// Holds information about the rows.
    RowInfoList m_kRowInfos;

    /** Creates and initializes a new RowInfo
    @param uiRow
    @return RowInfo
    */
    virtual RowInfo CreateRowInfo(eC_UInt uiRow);

    /** Generates a view object for a cell using the associated generator.
    @param kViewIndex
    @return view-item
    */
    virtual CGUIViewItem* GenerateViewItem(const CGUIViewIndex& kViewIndex);

    /** Callback that informs inherited classes about a selection update.
    Override this method to react on changed selection.
    @see GetSelection()
    */
    virtual void SelectionUpdated(){}

    /**
        Helper function for DoClick and DoKeyDown to handle selection.
        @param rkIndex The index of the clicked / pressed object.
        @param uiModifiers The currently active key modifiers.
        @return True if the selection changed, False if not.
    */
    eC_Bool HandleSelection(const CGUIViewIndex& rkIndex, const eC_UInt uiModifiers);

     /** Internal helper for selection handling.  Adds a ViewItem to the selection list
     and sets it selected.
     This method does not trigger SelectionUpdated().
     @see RemoveItemFromSelection.
     @param rkIndex The Index of the ViewItem to set selected.
     @return True if the item was successfully added, otherwise False.
     */
    eC_Bool AddItemToSelection(const CGUIViewIndex& rkIndex);

    /** Internal helper for selection handling. Removes item from the selection list
     and sets it unselected.
     This method does not trigger SelectionUpdated().
     @see CGUIViewItem::AddItemToSelection.
     @param rkIndex The Index of the ViewItem to set unselected.
     @return True if the item was successfully added, otherwise False.
     */
    eC_Bool RemoveItemFromSelection(const CGUIViewIndex& rkIndex);

    /** Selects a range of cells.
    @param rkFromIndex The StartIndex of the selection.
    @param rkToIndex The index for the last item to select.
    @return True if the selection was successful, False else.
    */
    eC_Bool SelectBlock(const CGUIViewIndex& rkFromIndex, const CGUIViewIndex& rkToIndex);

    /** Toggles selection of an item.
    @param rkIndex The index of the item to toggle.
    @return True if the toggle operation was successful, False else.
    */
    eC_Bool ToggleSelection(const CGUIViewIndex& rkIndex);

    /** Toggles selection of a row.
    @param uiRow The row of the item to toggle.
    @return True if the toggle operation was successful, False else.
    */
    eC_Bool ToggleSelectionForRow(const eC_UInt uiRow);

    /** Retrieve the selection state of an item
    @param pkViewItem The pkViewItem to check.
    @return True if the item is currently in "selected mode", False else.*/
    eC_Bool IsItemSelected(CGUIViewItem* pkViewItem);

    /** Check if an item is selectable.
        @param rkIndex The index of the item to check.
        @return True if the item is selectable, False else.
    */
    eC_Bool ItemIsSelectable(const CGUIViewIndex& rkIndex);


    /** Check if an item is editable.
        @param rkIndex The index of the item to check.
        @return True if the item is editable, False else.
    */
    eC_Bool ItemIsEditable(const CGUIViewIndex& rkIndex);

    /** 
        The ViewItemGeneratorContainer is used to hold CGUIViewItemGenerator objects for the View.
        ViewItemGenerators can be set for the whole view, for a column or for a cell.
        @brief A helper class for CGUITableView
        @see CGUIViewItemGenerator.
    */
    class CViewItemGeneratorContainer
    {
        public:

            CViewItemGeneratorContainer();
            ~CViewItemGeneratorContainer();

            /** Set a default view item generator for the table.
                @param pkViewItemGenerator The view item generator
            */
            void SetDefaultViewGenerator(CGUIViewItemGenerator* pkViewItemGenerator);

            /** Set a default view item generator for a column
                @param uiColumn The column index
                @param pkViewItemGenerator The view item generator
            */
            void SetDefaultColumnViewGenerator(const eC_UInt uiColumn, CGUIViewItemGenerator* pkViewItemGenerator);

            /** Set a view item generator for a cell.
                @param rkIndex The cell index
                @param pkViewItemGenerator The view item generator
            */
            void SetCellGenerator(const CGUIViewIndex& rkIndex, CGUIViewItemGenerator* pkViewItemGenerator);

            /** Get the column generator if a column generator was set
                @param cuiColumn The column index
                @return The column generator
            */
            const CGUIViewItemGenerator* GetColumnViewItemGenerator(const eC_UInt cuiColumn);

            /** Get the generator for a view index.
                @param rkIndex The view index
                @return The generator
            */
            const CGUIViewItemGenerator* GetViewItemGenerator(const CGUIViewIndex& rkIndex);

        private:
            /**
            * Copy-constructor.
            * Dummy declaration with no implementation, just to hide the function.
            * @param kSource Source object to be copied.
            */
            CViewItemGeneratorContainer(const CViewItemGeneratorContainer& kSource);

        private:

            /// the default generator for the whole table.
            CGUIViewItemGenerator* m_pkDefaultViewGenerator;

            /// structure used to store generators for a column.
            struct columnGenerator_t
            {
                eC_UInt m_uiColumn;
                CGUIViewItemGenerator* m_pkGenerator;
            };

            /// holds generators which were set for columns.
            eC_TListDoubleLinked<columnGenerator_t> m_kColumnGenerators;

            /// structure used to store generators for a cell.
            struct cellGenerator_t
            {
                eC_UInt m_uiColumn;
                eC_UInt m_uiRow;
                CGUIViewItemGenerator* m_pkGenerator;
            };

            /// holds generators which were set for cells.
            eC_TListDoubleLinked<cellGenerator_t> m_kCellGenerators;
    };

    /// Container for all ViewItemGenerators that are set for the view, a column or a cell.
    CViewItemGeneratorContainer m_kGeneratorContainer;

private:
    /// Initialization helper for constructors.
    void Init();

    /// Table view-specific scroll bar initializations.
    void InitScrollBars();

    /// Clears the view (removes all children from the row container).
    void Clear();

    /** Check if auto resize of columns is necessary
        @param rvWidthToResizePerColumn The resize width of one auto resize column to resize.
               This will return the calculated resize pixel count per column.
        @return const eC_Bool True, if auto resize of columns is necessary
        */
    eC_Bool IsAutoResizeOfColumnsNecessary(eC_Value& rvWidthToResizePerColumn);

    /** Regenerate the column indices of the view items and the header item starting with column index.
    @param cuiColumnIndex The starting column index.
    */
    void    RegenerateItemsIndexFromColumn(const eC_UInt cuiColumnIndex);

    /** Container for all elements of the View except the HeaderItems.
    Used as scrolled object for vertical scrollbar */
    CGUICompositeObject* m_pkViewObjectContainer;

    /// The selectionmode for the table.
    SelectionMode_t m_eSelectionMode;

    /// The selection list holds all selected items.
    SelectionList m_kSelectionList;

    /// The index of the first item selected when selecting more than one element
    CGUIViewIndex m_kFirstSelected;

    /// Height of each row.
    eC_Value m_vRowHeight;

    /// Whether horizontal grid lines are drawn.
    eC_Bool m_bHorizontalLines;

    /// Whether vertical grid lines are drawn.
    eC_Bool m_bVerticalLines;

    /// Font for HeaderItems
    FontResource_t m_eHeaderTextFont;

    // Text-Color of HeaderItems
    eC_UInt m_uiHeaderTextColor;

    /// Background color of the header items.
    eC_UInt m_uiHeaderBkgColor;

    /// Color for grid lines.
    eC_UInt m_uiGridColor;

    /// Width of the grid line.
    eC_Value m_vGridLineWidth;

    /// The height of the header area.
    eC_Value m_vHeaderHeight;

    /// The auto resize column list holds all column auto resize candidates.
    AutoResizeColumnList m_AutoResizeColumnList;

    /// Indicates the last difference between clipper and container width.
    eC_Value m_vLastClipperContainerDifference;

    /// Indicates the gap from left and right border of all view items
    eC_Value m_vViewItemsGap;

    /// Indicates the gap from top and bottom border of all view items
    eC_Value m_vViewItemsGapVertical;

    /// Indicates if the current index has to be stored as the first element in the selected block
    eC_Bool m_bIgnoreIndexForSelection;
};

#endif // GUITABLEVIEW_H
