/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef GUITABCONTAINER__H_
#define GUITABCONTAINER__H_

#include "GUICompositeObject.h"
#include "GUIImageResource.h"
#include "GUIFontResource.h"

#include "GUIRadioButtonGroup.h"

#include "GUIBehaviourDecorator.h"

#include "GUINinePatch.h"
#include "GUICommonTypes.h"

/**
@brief Behaviour to switch between the children of a tab-container
*/
class CGUITabItemBehaviour : public CGUIBehaviourDecorator
{
public:
    /** Constructor
    @param uiTabItemIndex the child-index which will be activated when using the behaviour
    */
    CGUITabItemBehaviour(const eC_UInt& uiTabItemIndex);

    ///Default constructor
    CGUITabItemBehaviour();

    /** Set the attached container
    @param pkContainer
    */
    void SetContainer(CGUICompositeObject* pkContainer);

protected:
    virtual eC_Bool DoClick(
        const eC_Value &vAbsX,
        const eC_Value &vAbsY);

private:
    CGUICompositeObject* m_pkContainer;
    eC_UInt m_uiTabItemIndex;
};

class CGUITabContainer;
class CGUITabItem;

/**
@brief this class contains the client-area of the tab-container
it is on the same hierarchy-level as the button-box
*/
class CGUITabClientArea : public CGUICompositeObject
{
public:
    /** CGUITabClientArea constructor.
    @param pkParent Pointer to the designated parent object.
    @param vX X-position relative to its parent object
    @param vY Y-position relative to its parent object
    @param vWidth Width of the composite object
    @param vHeight Height of the composite object
    @param eID Object Identifier of this composite object (choose NO_HANDLE if none is required)
    */
    CGUITabClientArea(
        CGUICompositeObject *const pkParent,
        const eC_Value &vX, const eC_Value &vY,
        const eC_Value &vWidth, const eC_Value &vHeight,
        const ObjectHandle_t &eID = NO_HANDLE);

    CGUITabClientArea();

    virtual ~CGUITabClientArea() {}

    /**
    add a new TabItem to the Container
    @param pkTabItem TabItem to insert
    @return true if successful, else false
    */
    virtual eC_Bool AddObject(CGUITabItem *pkTabItem);

    /**
    add a new object to the Container
    @param pkObject object to insert
    @return true if successful, else false
    */
    virtual eC_Bool AddObject(CGUIObject *pkObject);

    virtual void RemoveObject(CGUIObject* pkObject);

    /** Refresh client-area
    */
    void Refresh();

    /** Set the attached container
    @param pkContainer
    */
    void SetContainer(CGUITabContainer* pkContainer);

#ifdef GUILIANI_STREAM_GUI
    virtual void ReadFromStream();
#endif // GUILIANI_STREAM_GUI

#ifdef GUILIANI_WRITE_GUI
    virtual void WriteToStream(const eC_Bool bWriteClassID = false);
#endif // GUILIANI_WRITE_GUI

private:
    CGUITabContainer* m_pkTabContainer;
};

/**
this Class defines a sub-object with is used within the CGUITabContainer
*/
class CGUITabItem : public CGUICompositeObject
{
public:
    /** CGUITabItem standard constructor.
    @see CGUIObject().
    */
    CGUITabItem();

    /** CGUITabItem constructor.
    @param vX X-position relative to its parent object
    @param vY Y-position relative to its parent object
    @param vWidth Width of the composite object
    @param vHeight Height of the composite object
    @param kHeadline headline of item
    @param eID Object Identifier of this composite object (choose NO_HANDLE if none is required)
    */
    CGUITabItem(
        const eC_Value &vX, const eC_Value &vY,
        const eC_Value &vWidth, const eC_Value &vHeight,
        const eC_String& kHeadline,
        const ObjectHandle_t &eID = NO_HANDLE);

    /** CGUITabItem constructor.
    @param vX X-position relative to its parent object
    @param vY Y-position relative to its parent object
    @param vWidth Width of the composite object
    @param vHeight Height of the composite object
    @param eHeadline headline of item
    @param eID Object Identifier of this composite object (choose NO_HANDLE if none is required)
    */
    CGUITabItem(
        const eC_Value &vX, const eC_Value &vY,
        const eC_Value &vWidth, const eC_Value &vHeight,
        const TextResource_t& eHeadline,
        const ObjectHandle_t &eID = NO_HANDLE);

    /** CGUITabItem constructor.
    @param kRect Bounding rectangle relative to parent.
    @param kHeadline headline of item
    @param eID Object Identifier of this composite object (choose NO_HANDLE if none is required)
    */
    CGUITabItem(
        const CGUIRect &kRect,
        const eC_String& kHeadline,
        const ObjectHandle_t &eID = NO_HANDLE);

    /** CGUITabItem constructor.
    @param kRect Bounding rectangle relative to parent.
    @param eHeadline headline of item
    @param eID Object Identifier of this composite object (choose NO_HANDLE if none is required)
    */
    CGUITabItem(
        const CGUIRect &kRect,
        const TextResource_t& eHeadline,
        const ObjectHandle_t &eID = NO_HANDLE);

    /// This destructor is automatically virtual, as the base class destructor is virtual
    virtual ~CGUITabItem();

    /** Set the headline-text as string
    @param kText headline
    */
    void SetText(const eC_String& kText);

    /** Set the headline-text as text-id
    @param eTextID text-id
    */
    void SetText(const TextResource_t& eTextID);

    /** Return the headline-text
    @return headline-text
    */
    eC_String GetText() const;

#ifdef GUILIANI_STREAM_GUI
    virtual void ReadFromStream();
#endif // GUILIANI_STREAM_GUI

#ifdef GUILIANI_WRITE_GUI
    virtual void WriteToStream(const eC_Bool bWriteClassID = false);
#endif // GUILIANI_WRITE_GUI

private:
    eC_Bool m_bUseTextID;
    eC_String m_kText;
    TextResource_t m_eTextID;
};

// @guiliani_doxygen toplevel_control Tab-Container
/**
<table border="0">
<tr>
<td width="200">@image html tab_container.png</td>
<td>
The Tab-Container can be used to group together various child-elements where only one child
will be displayed at a time. This can be used to create tabbed dialogs containing settings
for the application or to create sub-pages in a dialog to save up space and group them
into logical units.
</td>
</tr>
</table>
*/
// @endguiliani_doxygen

/**
This container will help grouping elements into to sub-groups while only one group will be displayed
and the others can be switched. Each child-element is a CGUITabItem and will be inserted into the
clientarea of the tab-container.
The placement and sizes of the buttons can globally set as well as the images and font used for the
buttons. Each CGUITabItem has a text-attribute which will be displayed on its corresponding tab-button.

@ingroup GUILIANI_CONTROLS
*/
class CGUITabContainer : public CGUICompositeObject
{
public:
    /** CGUICompositeObject standard constructor.
    @see CGUIObject().
    */
    CGUITabContainer();

    /** CGUICompositeObject constructor.
    @param pkParent Pointer to the designated parent object.
    @param vX X-position relative to its parent object
    @param vY Y-position relative to its parent object
    @param vWidth Width of the composite object
    @param vHeight Height of the composite object
    @param eID Object Identifier of this composite object (choose NO_HANDLE if none is required)
    */
    CGUITabContainer(
        CGUICompositeObject *const pkParent,
        const eC_Value &vX, const eC_Value &vY,
        const eC_Value &vWidth, const eC_Value &vHeight,
        const ObjectHandle_t &eID = NO_HANDLE);

    /** CGUICompositeObject constructor.
    @param pkParent Pointer to the designated parent object.
    @param kRect Bounding rectangle relative to parent.
    @param eID Object Identifier of this composite object (choose NO_HANDLE if none is required)
    */
    CGUITabContainer(
        CGUICompositeObject *const pkParent,
        const CGUIRect &kRect,
        const ObjectHandle_t &eID = NO_HANDLE);

    /// This destructor is automatically virtual, as the base class destructor is virtual
    virtual ~CGUITabContainer();

    /**
    add a new TabItem to the Container
    @param pkTabItem TabItem to insert
    @return true if successful, else false
    */
    virtual eC_Bool AddObject(CGUITabItem *pkTabItem);

    /**
    add a new object to the Container
    @param pkObject object to insert
    @return true if successful, else false
    */
    virtual eC_Bool AddObject(CGUIObject *pkObject);

    virtual void RemoveObject(CGUIObject* pkObject);

#ifdef GUILIANI_STREAM_GUI
    virtual void ReadFromStream();
#endif // GUILIANI_STREAM_GUI

#ifdef GUILIANI_WRITE_GUI
    virtual void WriteToStream(const eC_Bool bWriteClassID = false);
#endif // GUILIANI_WRITE_GUI

    virtual void SetWidth(const eC_Value &vX);
    virtual void SetHeight(const eC_Value &vY);

    /**
    Sets position of the buttonbox
    @param eTabButtonPosition position of buttonbox
    */
    void SetButtonPosition(const CGUICommonEnums::Alignment_t& eTabButtonPosition);

    /**
    Get the the position of the buttonbox
    @return position
    */
    CGUICommonEnums::Alignment_t GetButtonPosition() const;

    /**
    Set the current active tab-index
    @param uiIndex index of tab
    */
    void SetCurrentTabIndex(const eC_UInt& uiIndex);

    /**
    Get the index of currently visible tab
    @return index
    */
    eC_UInt GetCurrentTabIndex() const;

    /**
    sets sizes for buttons when auto-size is not active
    @param vWidth
    @param vHeight
    */
    void SetButtonSize(const eC_Value& vWidth, const eC_Value& vHeight);

    /**
    Get sizes for buttons
    @param vWidth
    @param vHeight
    */
    void GetButtonSize(eC_Value& vWidth, eC_Value& vHeight) const;

    /**
    sets maximum sizes for buttons when auto-size is active
    @param vWidth
    @param vHeight
    */
    void SetButtonMaxSize(const eC_Value& vWidth, const eC_Value& vHeight);

    /**
    get maximum sizes for buttons for auto-size
    @param vWidth
    @param vHeight
    */
    void GetButtonMaxSize(eC_Value& vWidth, eC_Value& vHeight) const;

    /**
    set auto-size active
    @param bButtonAutoSize
    */
    void SetButtonAutoSize(const eC_Bool& bButtonAutoSize);

    /**
    check if auto-size is active
    @return true if active
    */
    eC_Bool GetButtonAutoSize() const;

    /**
    set center buttons
    @param bCenterButtons if true the buttons are centered according to the currently set alignment
    */
    void SetCenterButtons(const eC_Bool& bCenterButtons);

    /**
    get if buttons are centered
    @return true if centered
    */
    eC_Bool GetCenterButtons() const;

    /**
    set images for buttons in the buttonbox
    @param eImageNotSelectedStandard image if not selected
    @param eImageNotSelectedHighlighted image if not selected
    @param eImageNotSelectedPressed image if not selected
    @param eImageNotSelectedFocussed image if not selected
    @param eImageNotSelectedGrayedOut image if not selected
    @param eImageSelectedStandard image if selected
    @param eImageSelectedHighlighted image if selected
    @param eImageSelectedPressed image if selected
    @param eImageSelectedFocussed image if selected
    @param eImageSelectedGrayedOut image if grayed out
    */
    void SetButtonImages(
        const ImageResource_t &eImageNotSelectedStandard,
        const ImageResource_t &eImageNotSelectedHighlighted,
        const ImageResource_t &eImageNotSelectedPressed,
        const ImageResource_t &eImageNotSelectedFocussed,
        const ImageResource_t &eImageNotSelectedGrayedOut,
        const ImageResource_t &eImageSelectedStandard,
        const ImageResource_t &eImageSelectedHighlighted,
        const ImageResource_t &eImageSelectedPressed,
        const ImageResource_t &eImageSelectedFocussed,
        const ImageResource_t &eImageSelectedGrayedOut);

    /**
    get images for buttons in the buttonbox
    @param eImageNotSelectedStandard image if not selected
    @param eImageNotSelectedHighlighted image if not selected
    @param eImageNotSelectedPressed image if not selected
    @param eImageNotSelectedFocussed image if not selected
    @param eImageNotSelectedGrayedOut image if not selected
    @param eImageSelectedStandard image if selected
    @param eImageSelectedHighlighted image if selected
    @param eImageSelectedPressed image if selected
    @param eImageSelectedFocussed image if selected
    @param eImageSelectedGrayedOut image if grayed out
    */
    void GetButtonImages(
        ImageResource_t &eImageNotSelectedStandard,
        ImageResource_t &eImageNotSelectedHighlighted,
        ImageResource_t &eImageNotSelectedPressed,
        ImageResource_t &eImageNotSelectedFocussed,
        ImageResource_t &eImageNotSelectedGrayedOut,
        ImageResource_t &eImageSelectedStandard,
        ImageResource_t &eImageSelectedHighlighted,
        ImageResource_t &eImageSelectedPressed,
        ImageResource_t &eImageSelectedFocussed,
        ImageResource_t &eImageSelectedGrayedOut);

    /**
    set the font-id used for the tab-buttons
    @param eFontID font-id
    */
    void SetFontID(const FontResource_t& eFontID);

    /**
    get the font-id used for the tab-buttons
    @return font-id
    */
    FontResource_t GetFontID() const;

    /**
    set the text-colors for tab-buttons
    @param kColorStandard text-color for standard
    @param kColorHighlighted text-color for highlighted
    @param kColorPressed text-color for pressed
    @param kColorGrayedOut text-color for grayed out
    */
    void SetButtonTextColor(
        const ColorValue_t& kColorStandard,
        const ColorValue_t& kColorHighlighted,
        const ColorValue_t& kColorPressed,
        const ColorValue_t& kColorGrayedOut);

    /**
    get the text-colors for tab-buttons
    @param kColorStandard text-color for standard
    @param kColorHighlighted text-color for highlighted
    @param kColorPressed text-color for pressed
    @param kColorGrayedOut text-color for grayed out
    */
    void GetButtonTextColor(
        ColorValue_t& kColorStandard,
        ColorValue_t& kColorHighlighted,
        ColorValue_t& kColorPressed,
        ColorValue_t& kColorGrayedOut);

    /**
    Get the number of total tabs in container
    @return number of tabs
    */
    eC_UInt GetNumberOfTabs() const;

    /**
    Returns a pointer to the client-area of the TabContainer
    @return pointer to client-area
    */
    CGUITabClientArea* GetClientArea();

    /**
    Return the nine-patch used for images
    @return nine-patch
    */
    CGUINinePatch& GetNinePatch();

    /** Refresh layout of buttons
    */
    void RefreshButtons();

private:
    /** initialize the container and buttons
    */
    void Init();

    void DeInit();

    /** adjust container sizes and positions according to tab-position
    */
    void AdjustContainers();

    void GetOrCreateSubObjects();

    void AdjustTabButtonCount();

    /** update buttons after layout or sizes have been changed
    */
    void UpdateTabButtons();

private:
    CGUICommonEnums::Alignment_t m_eTabButtonAlignment;
    eC_UInt m_uiActiveTabIndex;
    eC_Value m_vButtonWidth;
    eC_Value m_vButtonHeight;
    eC_Value m_vMaxButtonWidth;
    eC_Value m_vMaxButtonHeight;
    eC_Bool m_bAutoSizeButtons;
    eC_Bool m_bCenterButtons;

    ImageResource_t m_eImageNotSelectedStandard;
    ImageResource_t m_eImageNotSelectedHighlighted;
    ImageResource_t m_eImageNotSelectedPressed;
    ImageResource_t m_eImageNotSelectedFocussed;
    ImageResource_t m_eImageNotSelectedGrayedOut;
    ImageResource_t m_eImageSelectedStandard;
    ImageResource_t m_eImageSelectedHighlighted;
    ImageResource_t m_eImageSelectedPressed;
    ImageResource_t m_eImageSelectedFocussed;
    ImageResource_t m_eImageSelectedGrayedOut;
    CGUINinePatch m_kNinePatch;

    FontResource_t m_eFontID;

    ColorValue_t m_kTextColorStandard;
    ColorValue_t m_kTextColorHighlighted;
    ColorValue_t m_kTextColorPressed;
    ColorValue_t m_kTextColorGrayedOut;

    CGUIRadioButtonGroup* m_pkButtonBox;
    CGUITabClientArea* m_pkClientArea;

    eC_UInt m_uiNumberOfTabs;

    eC_Bool m_bInCreateSubObjects;
};

#endif
