/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef GUISPLITCONTAINER__H_
#define GUISPLITCONTAINER__H_

#include "GUICompositeObject.h"
#include "GUICommonEnums.h"
#include "GUIColorPropertyObserver.h"

#include "GUIImageResource.h"

/**
A CGUISplitContainer can be used to create container with two subsections and to arrange various controls.
The dividing handle can either be horizontal or vertical. Even more complex divisions can be created by nesting CGUISplitContainer inside the client-area.

For best results the use of either Repositioning-CompositeObjects or Layouters is recommended, so that the children adapt to the changed size of their parent.

To move the dividing handle simply drag it to the new position. When double-clicking on the handle it will be moved to the middle of the container.
*/
class CGUISplitContainer : public CGUICompositeObject, CGUIColorPropertyObserver
{
public:
    /** CGUICompositeObject standard constructor.
    @see CGUIObject().
    */
    CGUISplitContainer();

    /** CGUICompositeObject constructor.
    @param pkParent Pointer to the designated parent object.
    @param vX X-position relative to its parent object
    @param vY Y-position relative to its parent object
    @param vWidth Width of the composite object
    @param vHeight Height of the composite object
    @param eID Object Identifier of this composite object (choose NO_HANDLE if none is required)
    */
    CGUISplitContainer(
        CGUICompositeObject* const pkParent,
        const eC_Value& vX,
        const eC_Value& vY,
        const eC_Value& vWidth,
        const eC_Value& vHeight,
        const ObjectHandle_t &eID = NO_HANDLE);

    /** CGUICompositeObject constructor.
    @param pkParent Pointer to the designated parent object.
    @param kRect Bounding rectangle relative to parent.
    @param eID Object Identifier of this composite object (choose NO_HANDLE if none is required)
    */
    CGUISplitContainer(
        CGUICompositeObject* const pkParent,
        const CGUIRect& kRect,
        const ObjectHandle_t &eID = NO_HANDLE);

    /// This destructor is automatically virtual, as the base class destructor is virtual
    virtual ~CGUISplitContainer();

#ifdef GUILIANI_STREAM_GUI
    virtual void ReadFromStream();
#endif // GUILIANI_STREAM_GUI

#ifdef GUILIANI_WRITE_GUI
    virtual void WriteToStream(const eC_Bool bWriteClassID = false);
#endif // GUILIANI_WRITE_GUI

    virtual eC_Bool AddObject(CGUIObject* pkObject);

    virtual void RemoveObject(CGUIObject* pkObject);

    virtual void SetWidth(const eC_Value &vX);

    virtual void SetHeight(const eC_Value &vY);

    virtual eC_Bool DoMouseEnter(const eC_Value &vAbsX, const eC_Value &vAbsY);
    virtual eC_Bool DoMouseLeave(const eC_Value &vAbsX, const eC_Value &vAbsY);
    virtual eC_Bool DoMouseMove(const eC_Value &vAbsX, const eC_Value &vAbsY);
    virtual eC_Bool DoButtonDown(const eC_Value &vAbsX, const eC_Value &vAbsY);
    virtual eC_Bool DoButtonUp(const eC_Value &vAbsX, const eC_Value &vAbsY);
    virtual eC_Bool DoDrag(const eC_Value &vDeltaX, const eC_Value &vDeltaY, const eC_Value &vAbsX, const eC_Value &vAbsY);
    virtual eC_Bool DoDragEnd(const eC_Value &vAbsX, const eC_Value &vAbsY);
    virtual eC_Bool DoDoubleClick(const eC_Value &vAbsX, const eC_Value &vAbsY);

    virtual eC_Bool DoPostDraw();

    /**
    centers the position of the handle after a doubleclick
    */
    void CenterSplitterHandle();

    /**
    sets the size of the splitter-handle
    @param vSize size to use for the splitter-handle
    */
    void SetHandleSize(const eC_Value& vSize);

    /**
    get the size of the splitter-handle
    @return size of the splitter-handle
    */
    eC_Value GetHandleSize();

    /**
    set the orientation of the splitter-handle
    @param eOrientation orientation to use for the splitter-handle
    */
    void SetHandleOrientation(const CGUICommonEnums::Orientation_t& eOrientation);

    /**
    get the orientation of the splitter-handle
    @return orientation of the splitter-handle
    */
    CGUICommonEnums::Orientation_t GetHandleOrientation();

    /**
    set the position of the splitter-handle
    @param vPosition position to use for the splitter-handle
    */
    void SetHandlePosition(const eC_Value& vPosition);

    /**
    get the position of the splitter-handle
    @return position of the splitter-handle
    */
    eC_Value GetHandlePosition();

    /**
    Set the image for the handle. if no image is set,
    the color is used for drawing.
    @param eImage image for handle
    @param eImageHighlighted image for handle if highlighted
    */
    void SetHandleImage(const ImageResource_t& eImage, const ImageResource_t& eImageHighlighted);

    /**
    Get the currently set image for the handle
    @param eImage image for the handle
    @param eImageHighlighted image for the handle if highlighted
    */
    void GetHandleImage(ImageResource_t& eImage, ImageResource_t& eImageHighlighted) const;

    /**
    Set the color for the handle using color-code
    @param uiColor color for the handle
    @param uiColorHighlighted color for the handle if highlighted
    */
    void SetHandleColor(const eC_UInt& uiColor, const eC_UInt& uiColorHighlighted);

    /**
    Set the color for the handle using property
    @param eColor color-property for the handle
    @param eColorHighlighted color-property for the handle if highlighted
    */
    void SetHandleColor(const GlobalProperty_t& eColor, const GlobalProperty_t& eColorHighlighted);

    /**
    Get the currently set color for the handle
    @param uiColor color for the handle
    @param uiColorHighlighted color for the handle if highlighted
    */
    void GetHandleColor(eC_UInt& uiColor, eC_UInt& uiColorHighlighted) const;

    /**
    Enlarge the sensitive area of the splitter-handle
    @param uiHandleEnlarge enlarge in each direction
    */
    void SetHandleEnlarge(const eC_UInt& uiHandleEnlarge);

    /**
    Get the enlargement for the splitter-handle
    @return enlargement in pixel
    */
    eC_UInt GetHandleEnlarge() const;

private:
    /** Helper function for constructors. */
    void Init();

    void DeInit();

    /** adjusts the containers' position and size after something changed
    */
    void AdjustContainers();

    /**
    sets the position of the splitter-handle after a drag
    @param vAbsX x-position of drag
    @param vAbsY y-position of drag
    */
    void SetHandlePosition(const eC_Value &vAbsX, const eC_Value &vAbsY);

private:
    static const eC_UInt INDEX_COLOR;
    static const eC_UInt INDEX_COLORHIGHLIGHTED;

    CGUICommonEnums::Orientation_t m_eSplitterOrientation;
    eC_Value m_vSplitterPosition;
    eC_Value m_vSplitterSize;
    ImageResource_t m_eSplitterImage;
    ImageResource_t m_eSplitterImageH;
    eC_UInt m_uiSplitterEnlarge;

    CGUIObject* m_pkContainer1;
    CGUIObject* m_pkContainer2;
    CGUIRect m_kSplitterRect;
    eC_Bool m_bInsideHandle;
    eC_Bool m_bDragging;
};

#endif
