/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef GUISEGMENTBAR_H
#define GUISEGMENTBAR_H

#include "GUIAbstractBar.h"

#include "GUICommonEnums.h"
#include "GUIImageResource.h"
#include "GUINinePatch.h"

/**
This control can be used to display values in the form of a filled bar with distinct segments.
The images for inactive and active segments can be set. Additionally the margin and the size
of a segment and also the gap between each segment can be set.
There is also an auto-stretch-attribute which can be used to resize the bar without changing
the size of the segments.

During runtime the value can be adjusted by clicking or dragging on the control.
Minimum and maximum-values as well as step-size are inherited from CGUIAbstractBar.

To understand how the control is displayed:

@code
+-----------------------------------+
|                                   |  < vertical margin
|  ooo ooo ooo ooo ooo ooo ooo ooo  |  <\
|  ooo ooo ooo ooo ooo ooo ooo ooo  |  <- segment height
|  ooo ooo ooo ooo ooo ooo ooo ooo  |  </
|                                   |  < vertical margin
+-----------------------------------+
      ^ Gap between segments
   ^^^ segment width
 ^^ horizontal margin

if auto-stretch is active the segment-size is calculated by the size of the control minus the (margin * 2)
@endcode

@ingroup GUILIANI_CONTROLS
*/
class CGUISegmentBar : public CGUIAbstractBar
{
public:
    /** Constructor.
        @param pParent Pointer to the designated parent object.
        @param vX X-position relative to its parent object, i.e. the x-offset from the left border of the parent object
        @param vY Y-position relative to its parent object, i.e. the y-offset from the upper border of the parent object
        @param vWidth Width of the object
        @param vHeight Height of the object
        @param eID Object identifier of this object (choose NO_HANDLE if none is required).
    */
    CGUISegmentBar(
        CGUICompositeObject *const pParent,
        const eC_Value &vX, const eC_Value &vY,
        const eC_Value &vWidth, const eC_Value &vHeight,
        const ObjectHandle_t &eID = NO_HANDLE);

    CGUISegmentBar();

    virtual ~CGUISegmentBar();

    /** Sets the images used for the control
    @param eInactiveImage image-id for inactive segment
    @param eActiveImage image-id for active segment
    */
    void SetImages(const ImageResource_t& eInactiveImage, const ImageResource_t& eActiveImage);

    /** Return the images used for the control
    @param eInactiveImage image-id for inactive segment
    @param eActiveImage image-id for active segment
    */
    void GetImages(ImageResource_t& eInactiveImage, ImageResource_t& eActiveImage) const;

    /** Return the ninepatch which is used for each segment
    @return nine-patch
    */
    CGUINinePatch GetNinePatch();

    /** Sets the orientation of the bar
    @param eOrientation either horizontal or vertical
    */
    void SetOrientation(const CGUICommonEnums::Orientation_t& eOrientation);

    /** Return the orientation of the bar
    @return orientation
    */
    CGUICommonEnums::Orientation_t GetOrientation() const;

    /** Set the margin which is used to display the segments.
    @param vMarginX horizontal margin
    @param vMarginY vertical margin
    */
    void SetMargin(const eC_Value& vMarginX, const eC_Value& vMarginY);

    /** Return the margin used for the segments.
    @param vMarginX horizontal margin
    @param vMarginY vertical margin
    */
    void GetMargin(eC_Value& vMarginX, eC_Value& vMarginY) const;

    /** Set the gap between each segment.
    @param vGap gap
    */
    void SetGap(const eC_Value& vGap);

    /** Return the gap between each segment.
    @return gap
    */
    eC_Value GetGap() const;

    /** Set the size for a segment.
    If auto-stretch is false these sizes are used to display
    the segments.
    @param vSegmentWidth width of the segment
    @param vSegmentHeight height of the segment
    */
    void SetSegmentSize(const eC_Value& vSegmentWidth, const eC_Value& vSegmentHeight);

    /** Return the size for a segment.
    @param vSegmentWidth width of the segment
    @param vSegmentHeight height of the segment
    */
    void GetSegmentSize(eC_Value& vSegmentWidth, eC_Value& vSegmentHeight) const;

    /** Set if auto-stretch will be used.
    If auto-stretch is active the images for the segments will be stretched
    over the whole size of the object minus the margin. otherwise the segment-sizes
    will be used.
    @param bAutoStretch true for auto-stretch
    */
    void SetAutoStretch(const eC_Bool& bAutoStretch);

    /** Return whether auto-stretch is used.
    @return true if auto-stretch is used, or false
    */
    eC_Bool GetAutoStretch() const;

    eC_Bool OnValueChange();

    virtual void SetWidth(const eC_Value &vX);

    virtual void SetHeight(const eC_Value &vY);

    virtual eC_Bool DoButtonDown(
        const eC_Value &vAbsX,
        const eC_Value &vAbsY);

    virtual eC_Bool DoDrag(
        const eC_Value &vDeltaX,
        const eC_Value &vDeltaY,
        const eC_Value &vAbsX,
        const eC_Value &vAbsY);

    virtual eC_Bool DoClick(
        const eC_Value &vAbsX,
        const eC_Value &vAbsY);

#if defined(GUILIANI_STREAM_GUI)
    /** Reads all object attributes from streaming file.
    This method is called by CGUIFactoryManager after one of the registered
    factories has created an instance of this class.
    */
    virtual void ReadFromStream();
#endif

#if defined(GUILIANI_WRITE_GUI)
    /** Writes all object attributes to the streaming file. A CGUIStreamWriter
    has to be initialized first.
    @param bWriteClassID This flag is used to select if writing of ControlID,
    leading and trailing tags is performed.
    */
    virtual void WriteToStream(const eC_Bool bWriteClassID = false);
#endif

    virtual eC_Bool IsHighlightable() const { return false; }

protected:
    virtual eC_Bool DoDraw();

private:
    void Init();

    void DeInit();

    void UpdateValues();
    void CalculateValue(const eC_Value& vPosX, const eC_Value& vPosY);

private:
    // streamed members
    CGUICommonEnums::Orientation_t m_eOrientation; ///< orientation of the bar
    ImageResource_t m_eInactiveImage; ///< image-id for inactive segment
    ImageResource_t m_eActiveImage; ///< image-id for active segment
    CGUINinePatch m_kSegmentNinePatch; ///< nine-patch used for segments
    eC_Bool m_bAutoStretch; ///< auto-stretch segments
    eC_Value m_vMarginX; ///< x-margin
    eC_Value m_vMarginY; ///< y-margin
    eC_Value m_vSegmentWidth; ///< width of a segment
    eC_Value m_vSegmentHeight; ///< height of a segments
    eC_Value m_vGapBetweenSegments; ///< gap between segments

    // internal members
    eC_Value m_vScaledSegmentWidth; ///< width for a scaled segment
    eC_Value m_vScaledSegmentHeight; ///< height for a scaled segment
    eC_Value m_vAdvanceX; ///< x-advance
    eC_Value m_vAdvanceY; ///< y-advance
    eC_UInt m_uiNumberOfSegments; ///< number of segments
};

#endif
