/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#if !defined(EXAMPLE_CONTROL_H)
#define EXAMPLE_CONTROL_H

#include "GUIAbstractBar.h"
#include "GUIRange.h"
#include "GUINinePatch.h"
#include "GUICommonEnums.h"

#include "GUIImageResource.h"

/** An example CGUIObject implementation that draws a rectangle with
    configureable border width and configurable colors.
  */
class CGUIRangeSlider : public CGUIAbstractBar
{
public:
    /** Constructor.
        @param pParent Pointer to the designated parent object.
        @param vX X-position relative to its parent object, i.e. the x-offset from the left border of the parent object
        @param vY Y-position relative to its parent object, i.e. the y-offset from the upper border of the parent object
        @param vWidth Width of the object
        @param vHeight Height of the object
        @param eID Object identifier of this object (choose NO_HANDLE if none is required).
    */
    CGUIRangeSlider(
        CGUICompositeObject *const pParent,
        const eC_Value &vX, const eC_Value &vY,
        const eC_Value &vWidth, const eC_Value &vHeight,
        const ObjectHandle_t &eID = NO_HANDLE);

    CGUIRangeSlider();

    virtual ~CGUIRangeSlider();

    /** Sets the images for the control
    @param eBackgroundImage image-id of the background
    @param eHandle image-id of the handles
    @param eRange image-id of the range
    */
    void SetImages(
        const ImageResource_t& eBackgroundImage,
        const ImageResource_t& eHandle,
        const ImageResource_t& eRange);

    /** Return the images for the control
    @param eBackgroundImage image-id of the background
    @param eHandle image-id of the handles
    @param eRange image-id of the range
    */
    void GetImages(
        ImageResource_t& eBackgroundImage,
        ImageResource_t& eHandle,
        ImageResource_t& eRange) const;

    /** Set current orientation
    @param eOrientation
    */
    void SetOrientation(const CGUICommonEnums::Orientation_t& eOrientation);

    /** Return the current orientation
    @return orientation
    */
    CGUICommonEnums::Orientation_t GetOrientation() const;

    /** Set the margin used to display the slider
    @param vMarginX horizontal margin
    @param vMarginY vertical margin
    */
    void SetMargin(const eC_Value& vMarginX, const eC_Value& vMarginY);

    /** Get the margin used to display the slider
    @param vMarginX horizontal margin
    @param vMarginY vertical margin
    */
    void GetMargin(eC_Value& vMarginX, eC_Value& vMarginY);

    /** Return the value of the second range
    @return second range
    */
    const CGUIRange& GetRange2() const;

    virtual eC_Bool DoButtonDown(const eC_Value &vAbsX, const eC_Value &vAbsY);

    virtual eC_Bool DoDrag(const eC_Value &vDeltaX, const eC_Value &vDeltaY, const eC_Value &vAbsX, const eC_Value &vAbsY);

    virtual eC_Bool DoDragEnd(const eC_Value &vAbsX, const eC_Value &vAbsY);

    virtual eC_Bool DoClick(const eC_Value &vAbsX, const eC_Value &vAbsY);

    virtual eC_Bool SetValue(const CGUIValue& rkValue);

    /** Set the second value
    @param rkValue
    @return true if successful
    */
    virtual eC_Bool SetValue2(const CGUIValue& rkValue);

    virtual void SetWidth(const eC_Value &vWidth);
    virtual void SetHeight(const eC_Value &vHeight);

#if defined(GUILIANI_STREAM_GUI)
    /** Reads all object attributes from streaming file.
        This method is called by CGUIFactoryManager after one of the registered
        factories has created an instance of this class.
    */
    virtual void ReadFromStream();
#endif

#if defined(GUILIANI_WRITE_GUI)
    /** Writes all object attributes to the streaming file. A CGUIStreamWriter
        has to be initialized first.
        @param bWriteClassID This flag is used to select if writing of ControlID,
               leading and trailing tags is performed.
    */
    virtual void WriteToStream(const eC_Bool bWriteClassID=false);
#endif

    virtual eC_Bool IsHighlightable() const { return false; }

protected:
    virtual eC_Bool DoDraw();

private:
    enum MovingPart_t
    {
        MP_NONE,
        MP_HANDLE1,
        MP_HANDLE2,
        MP_MIDDLE
    };

    void Init();

    void DeInit();

    void CalculateImageSizes();
    void CalculateValues();
    void CalculateHandle();

    void MoveHandle(const eC_Value &vDeltaX, const eC_Value &vDeltaY, const eC_Value& vAbsX, const eC_Value& vAbsY);

private:
    // streamed members
    CGUICommonEnums::Orientation_t m_eOrientation;
    ImageResource_t m_eBackground;
    ImageResource_t m_eRange;
    ImageResource_t m_eHandle;

    eC_Value m_vMarginX;
    eC_Value m_vMarginY;

    // internal members
    MovingPart_t m_eMovingHandle;

    CGUIRect m_kHandle1;
    CGUIRect m_kHandle2;

    eC_Value m_vHandleWidth;
    eC_Value m_vHandleHeight;
    eC_Value m_vRangeWidth;
    eC_Value m_vRangeHeight;

    eC_Value m_vBackgroundWidth;
    eC_Value m_vBackgroundHeight;
    eC_Value m_vBackgroundPosX;
    eC_Value m_vBackgroundPosY;

    CGUIRect m_kRangeRect;

    CGUIRange m_kRange2;

    eC_Bool m_bSelfUpdate;
};

#endif
