/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef CGUIRADIALSLIDER__H_
#define CGUIRADIALSLIDER__H_

#include "GUIRadialBar.h"

#include "GUIImageResource.h"
#include "GUICommonEnums.h"
#include "GUINinePatch.h"

/// A primitive-based radial slider

// @guiliani_doxygen toplevel_control Radial Slider
/**
    <table border="0">
        <tr>
            <td width="200">@image html radial_slider.png</td>
            <td>
                The "radial slider" can control values in a radial manner.
            </td>
        </tr>
    </table>
*/
// @endguiliani_doxygen
/**
    @ingroup GUILIANI_CONTROLS
*/
class CGUIRadialSlider : public CGUIRadialBar
{
public:
    /** Constructor.
        By default, the progress bar type is set to be PBT_PROGRESSBAR_TYPE_NORMAL.
        The level range is from 0 to 100.
        The step size is set to 1 pixel.
        The dimension of the bar is determined after the
        foreground image is set. Here it is set to be 0 by 0.
        @param pParent Pointer to parent object.
        @param vX X position relative to its parent object.
        @param vY Y position relative to its parent object.
        @param vWidth Width of the control.
        @param vHeight Height of the control.
        @param eID object handle for the control.
    */
    CGUIRadialSlider(
        CGUICompositeObject *const pParent,
        const eC_Value &vX, const eC_Value &vY,
        const eC_Value &vWidth, const eC_Value &vHeight,
        const ObjectHandle_t &eID = NO_HANDLE);

    /** Constructor.
        By default, the progress bar type is set to be PBT_PROGRESSBAR_TYPE_NORMAL.
        The level range is from 0 to 100.
        The step size is set to 1 pixel.
        The dimension of the bar is determined after the
        foreground image is set. Here it is set to be 0 by 0.
        @param pParent Pointer to parent object.
        @param kRect rectangle area
        @param eID object handle for the control.
    */
    CGUIRadialSlider(
        CGUICompositeObject *const pParent,
        const CGUIRect& kRect,
        const ObjectHandle_t &eID = NO_HANDLE);

    /** Copy-constructor of the CGUIRadialSlider class.
        @param kSource Source object to be copied.
    */
    CGUIRadialSlider(const CGUIRadialSlider& kSource);

    /** Operator= method of the CGUIRadialSlider class.
        @param kSource Source object to be copied.
        @return Copied object.
    */
    CGUIRadialSlider& operator=(const CGUIRadialSlider& kSource);

    ///    Destructor
    virtual ~CGUIRadialSlider();

    /** Change the width of this control. The progressing area is changed proportionally.
        @param vWidth Width of the control.
    */
    virtual void SetWidth(const eC_Value &vWidth);

    /** Change the height of this control. The progressing area is changed proportionally.
        @param vHeight Height of the control.
    */
    virtual void SetHeight(const eC_Value &vHeight);

    virtual eC_Bool DoButtonDown(const eC_Value &vAbsX, const eC_Value &vAbsY);

    virtual eC_Bool DoDrag(const eC_Value &vDeltaX, const eC_Value &vDeltaY, const eC_Value &vAbsX, const eC_Value &vAbsY);

    virtual eC_Bool DoDragEnd(const eC_Value &vAbsX, const eC_Value &vAbsY);

    virtual eC_Bool DoClick(const eC_Value &vAbsX, const eC_Value &vAbsY);

    virtual eC_Bool SetValue(const CGUIValue& rkValue);

#ifdef GUILIANI_STREAM_GUI
    CGUIRadialSlider();

    virtual void ReadFromStream();
#endif

#ifdef GUILIANI_WRITE_GUI
    virtual void WriteToStream(const eC_Bool bWriteClassID = false);
#endif

protected:
    /** Help function to initialize the control.
    */
    void Init();

    /** Copy attributes from source control to this control.
        @param kSource Source object to be copied.
    */
    void CopyAttributes(const CGUIRadialSlider& kSource);

    /// Deinitialize the control and free the resources.
    void DeInit();

    /// calculate image-size
    void CalculateImageSizes();

    /// calculate current slider-values
    void CalculateValues();

    /// calculate position of handle
    void CalculateHandle();

    /** Move the handle to a specific position
    @param vDeltaX delta for x
    @param vDeltaY delta for y
    @param vAbsX absolute position for x
    @param vAbsY absolute position for y
    */
    void MoveHandle(const eC_Value &vDeltaX, const eC_Value &vDeltaY, const eC_Value& vAbsX, const eC_Value& vAbsY);

    /** Get the angle for a specific position
    @param vAbsX absolute position for x
    @param vAbsY absolute position for y
    @return angle
    */
    eC_Value GetAngle(const eC_Value& vAbsX, const eC_Value& vAbsY);

    /// draw filled part of bar
    virtual void DrawFilled();

private:
    enum MovingPart_t
    {
        MP_NONE,
        MP_HANDLE1,
        MP_HANDLE2,
        MP_MIDDLE
    };

private:
    ImageResource_t m_eImageKnob;

    eC_Bool m_bDoubleEnded;

    eC_Value m_vKnobWidth;
    eC_Value m_vKnobHeight;

    CGUIRect m_kHandle1;
    CGUIRect m_kHandle2;

    eC_Bool m_bSelfUpdate;

    // internal members
    MovingPart_t m_eMovingHandle;

    eC_Value m_vCurrentAngle;
    eC_Value m_vCurrentAngle2;

    eC_Value m_vCurrentDrawAngle;
    eC_Value m_vCurrentDrawAngle2;
};
#endif
