/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef GUIRADIALBAR__H_
#define GUIRADIALBAR__H_

#include "GUIAbstractBar.h"
#include "GUIColorPropertyObserver.h"

/// Base class for radial bars.

/**
    It is meant to be the base class for controls like progress bar and
    slider and does not include any visualization code therefore the
    class cannot be used standalone.

    A key feature of this class is that it calls InvalidateArea() for all
    functions that might change the status of the derived control in such
    a way that the visual appearance needs to be updated. Especially all
    get-functions do not imply such graphics updates therefore they do not
    have corresponding interface functions in this class but they may rather
    be called directly on the CGUIRange member of this class.

    @ingroup GUILIANI_CONTROLS
*/
class CGUIRadialBar : public CGUIAbstractBar, public CGUIColorPropertyObserver
{
public:
    /** Returns the currently set color for the unfilled segment
    @return currently set color */
    eC_UInt GetUnfilledColor() const;

    /** Sets the unfilled color as a 32 Bit hexadecimal value (0xAARRGGBB).
    @param uiColor Color of drawing */
    void SetUnfilledColor(const eC_UInt& uiColor);

    /** Sets the unfilled color as a property
    @param eColor property to use */
    void SetUnfilledColor(const GlobalProperty_t& eColor);

    /** Returns the currently set color for the filled segment
    @return currently set color */
    eC_UInt GetFilledColor() const;

    /** Sets the filled color as a 32 Bit hexadecimal value (0xAARRGGBB).
    @param uiColor Color of drawing */
    void SetFilledColor(const eC_UInt& uiColor);

    /** Sets the filled color as a property
    @param eColor property to use */
    void SetFilledColor(const GlobalProperty_t& eColor);

    /** Returns if center is filled
    @return true if filled
    */
    eC_Bool GetCenterFilled() const;

    /** Sets if the center will be filled
    @param bCenterFilled if true a circle is drawn, if false an arc
    */
    void SetCenterFilled(const eC_Bool& bCenterFilled);

    /** Get the width of the bar
    @return with of the bar
    */
    eC_Value GetBarWidth() const;

    /** Sets the width of the bar
    @param vBarWidth
    */
    void SetBarWidth(const eC_Value& vBarWidth);

    /** Returns the start-angle
    @return start-angle
    */
    eC_Value GetStartAngle() const;

    /** Sets the start-angle
    0 degrees is at the right, 90 degrees at the top
    @param vStartAngle
    */
    void SetStartAngle(const eC_Value& vStartAngle);

    /** Returns the end-angle
    @return end-angle
    */
    eC_Value GetEndAngle() const;

    /** Sets the end-angle
    0 degrees is at the right, 90 degrees at the top
    @param vEndAngle
    */
    void SetEndAngle(const eC_Value& vEndAngle);

    virtual eC_Bool DoDraw();

protected:
    /** CGUIRadialBar standard constructor.
        @see CGUIObject().
    */
    CGUIRadialBar();

    /** CGUIRadialBar constructor.
        @param pParent Pointer to parent object.
        @param vX X position relative to its parent object.
        @param vY Y position relative to its parent object.
        @param vWidth Width of this object.
        @param vHeight Height of this object.
        @param eID Object Identifier of this object
    */
    CGUIRadialBar(
        CGUICompositeObject *const pParent,
        const eC_Value &vX,
        const eC_Value &vY,
        const eC_Value &vWidth,
        const eC_Value &vHeight,
        const ObjectHandle_t &eID = NO_HANDLE);

    /** CGUIRadialBar constructor.
        @param pParent Pointer to parent object.
        @param kRect Structure containing x- and y-coordinates as well as width and height.
        @param eID Object Identifier of this object
    */
    CGUIRadialBar(
        CGUICompositeObject *const pParent,
        const CGUIRect &kRect,
        const ObjectHandle_t &eID = NO_HANDLE);

    /** Copy-constructor of the CGUIRadialBar class.
        @param kSource Source object to be copied.
    */
    CGUIRadialBar(const CGUIRadialBar& kSource);

    /** Assignment operator of the CGUIRadialBar class.
        @param kSource Source object to be copied.
        @return Copied object.
    */
    CGUIRadialBar& operator=(const CGUIRadialBar& kSource);

#ifdef GUILIANI_STREAM_GUI
    virtual void ReadFromStream();
#endif
#ifdef GUILIANI_WRITE_GUI
    virtual void WriteToStream(const eC_Bool bWriteClassID = false);
#endif

protected:
    /** Draw the unfilled part of the bar
    */
    virtual void DrawUnfilled();

    /** Draw the filled part of the bar
    */
    virtual void DrawFilled();

protected:
    eC_Bool m_bUseArcs; ///< draw arcs instead of rings
    eC_Bool m_bCenterFilled; ///< if center is filled or cut
    eC_Value m_vBarWidth; ///< width of the bar
    eC_Value m_vStartAngle; ///< start-angle
    eC_Value m_vEndAngle; ///< end-angle
    eC_Value m_vTotalAngle; ///< total angle

private:
    /** Helper function for constructors. */
    void Init();

    /** Copy attributes method of the CGUIRadialBar class.
        @param kSource Source object to be copied.
        @return Copied object.
    */
    void CopyAttributes(const CGUIRadialBar& kSource);

private:
    static const eC_UInt INDEX_UNFILLEDCOLOR;
    static const eC_UInt INDEX_FILLEDCOLOR;

};
#endif
