/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef GUIPLOT_H
#define GUIPLOT_H

#include "GUIObject.h"
#include "GUIColorPropertyObserver.h"
#include "GUIFontResource.h"

#include "GUICommonEnums.h"

#include "eC_TArray.h"

/**
@brief this control can be used to display continuous data over time.
It supports automatic and manual refresh.

Buffersize, number of data-sources and attributes for visualization
can be set.
*/
class CGUIPlot : public CGUIObject, public CGUIColorPropertyObserver
{
public:
    /** Constructor.
        @param pParent Pointer to the designated parent object.
        @param vX X-position relative to its parent object, i.e. the x-offset from the left border of the parent object
        @param vY Y-position relative to its parent object, i.e. the y-offset from the upper border of the parent object
        @param vWidth Width of the object
        @param vHeight Height of the object
        @param eID Object identifier of this object (choose NO_HANDLE if none is required).
    */
    CGUIPlot(
        CGUICompositeObject *const pParent,
        const eC_Value &vX, const eC_Value &vY,
        const eC_Value &vWidth, const eC_Value &vHeight,
        const ObjectHandle_t &eID = NO_HANDLE);

    /// Default constructor
    CGUIPlot();

    /// Destructor
    virtual ~CGUIPlot();

    /** Copy constructor
    @param kToBeCopied
    */
    CGUIPlot(const CGUIPlot& kToBeCopied);

    /** Assignment constructor
    @param kSource
    @return copy
    */
    CGUIPlot& operator=(const CGUIPlot& kSource);

    virtual void DoAnimate(const eC_Value& vTimes = eC_FromInt(1));

    virtual void SetWidth(const eC_Value& vX);
    virtual void SetHeight(const eC_Value& vY);

    /** Set the number of data-sources which are displayed.
    the maximum number is 10
    @param uiNumberOfDataSources
    */
    void SetNumberOfDataSources(const eC_UInt& uiNumberOfDataSources);

    /** Get the number of data-sources
    @return number of data-sources
    */
    eC_UInt GetNumberOfDataSources() const;

    /** Add new data to buffer for specified data-source.
    This data will be written at the current write-index.
    during manual refresh-mode the next sample can be advanced
    by a call to Refresh() and all currently received values
    will be persisted.
    @param uiIndex index of the data-source
    @param fValue data to add
    */
    void AddData(const eC_UInt& uiIndex, const eC_Float& fValue);

    /** Set the internal buffer-size. This is also the maximum number of values
    which can be displayed simultaneously.
    This will reset all data.
    @param uiBuffersize
    */
    void SetBufferSize(const eC_UInt& uiBuffersize);

    /** Return the buffer-size
    @return buffer-size
    */
    eC_UInt GetBufferSize() const;

    /** Set the refresh-method and interval.
    When automatic refresh is active the control will be refreshed
    in the given interval regardless of existing input.
    @param bAutomaticRefresh
    @param uiRefreshInterval interval in milliseconds for the refresh
    */
    void SetRefresh(const eC_Bool& bAutomaticRefresh, const eC_UInt& uiRefreshInterval);

    /** Get the refresh-mode and interval
    @param bAutomaticRefresh
    @param uiRefreshInterval
    */
    void GetRefresh(eC_Bool& bAutomaticRefresh, eC_UInt& uiRefreshInterval) const;

    /** Set if new values should be initialized with the last received value
    @param bRetainValues if true the last value is used, if false 0.0 is used
    */
    void SetRetainValues(const eC_Bool& bRetainValues);

    /** Get if values should be retained
    @return true if values are retained
    */
    eC_Bool GetRetainValues() const;

    /** Set spacing for the labels on the y-axis in pixels
    @param uiLabelSpacingY spacing in pixels (minimum 1)
    */
    void SetLabelSpacingY(const eC_UInt& uiLabelSpacingY);

    /** Return current spacing for y-axis
    @return spacing
    */
    eC_UInt GetLabelSpacingY() const;

    /** Set the scaling for the axis in pixels per value
    A setting of 10.0 will plot a value every 10 pixels.
    A setting below 1.0 for the x-axis is not accepted.
    The scaling for the x-axis is also used to display the scale
    @param vScalingX scaling for x-axis
    @param vScalingY scaling for y-axis
    */
    void SetScaling(const eC_Value& vScalingX, const eC_Value& vScalingY);
    
    /** Return the currently active scaling
    @param vScalingX
    @param vScalingY
    */
    void GetScaling(eC_Value& vScalingX, eC_Value& vScalingY) const;

    /** Set offset for Y-axis
    This is the number of pixels the center is moved in vertical direction,
    where negative values will move the x-axis up.
    @param vOffset offset in pixels
    */
    void SetOffset(const eC_Value& vOffset);

    /** Return the offset for Y-axis
    @return offset
    */
    eC_Value GetOffset() const;

    /** Set the padding
    @param vPaddingX
    @param vPaddingY
    */
    void SetPadding(const eC_Value& vPaddingX, const eC_Value& vPaddingY);

    /** Return the currently active padding
    @param vPaddingX
    @param vPaddingY
    */
    void GetPadding(eC_Value& vPaddingX, eC_Value& vPaddingY) const;

    /** Set the font used for labeling the axis
    @param eFontID
    */
    void SetFontID(const FontResource_t& eFontID);

    /** Return the font used for the labels
    @return font-id
    */
    FontResource_t GetFontID() const;

    /** Set the color for the axis as a value
    @param uiAxisColor
    */
    void SetAxisColor(const eC_UInt& uiAxisColor);

    /** Set the color for the axis as a property
    @param eAxisColor
    */
    void SetAxisColor(const GlobalProperty_t& eAxisColor);

    /** Return the color for the axis as a value
    @return color of axis
    */
    eC_UInt GetAxisColor() const;

    /** Set the width for the axis-lines
    @param vAxisLineWidth line-width
    */
    void SetAxisLineWidth(const eC_Value& vAxisLineWidth);

    /** Return the line-width for the axis
    @return line-width
    */
    eC_Value GetAxisLineWidth() const;

    /** Set the color for graph at specified index to color
    @param uiIndex index of graph
    @param uiLineColor color as value
    */
    void SetLineColor(const eC_UInt& uiIndex, const eC_UInt& uiLineColor);

    /** Set the color for graph at specified index to color
    @param uiIndex index of graph
    @param eLineColor color as property
    */
    void SetLineColor(const eC_UInt& uiIndex, const GlobalProperty_t& eLineColor);

    /** Return the color for graph at specified index
    @param uiIndex index of graph
    @return color as value
    */
    eC_UInt GetLineColor(const eC_UInt& uiIndex) const;

    /** Set the width for the plotted graph
    @param vLineWidth line-width
    */
    void SetLineWidth(const eC_Value& vLineWidth);

    /** Return the line-width for the plotted graph
    @return line-width
    */
    eC_Value GetLineWidth() const;

    /** Set interactive-mode.
    If true it will be possible to move the area of the control
    currently displayed by dragging the mouse. a double-click will reset
    to the center of the view.
    @param bInteractive
    */
    void SetInteractive(const eC_Bool& bInteractive);

    /** Get interactive-mode
    @return true if interactive-mode is set, else false
    */
    eC_Bool GetInteractive() const;

    /** Perform a refresh. This can be used to advance one sampling step,
    if automatic refresh has been deactivated.
    If more than one data-source is connected it is recommended to refresh the control
    after all data for this sample have been written to retain synchronisation.
    */
    void Refresh();

#if defined(GUILIANI_STREAM_GUI)
    /** Reads all object attributes from streaming file.
        This method is called by CGUIFactoryManager after one of the registered
        factories has created an instance of this class.
    */
    virtual void ReadFromStream();
#endif

#if defined(GUILIANI_WRITE_GUI)
    /** Writes all object attributes to the streaming file. A CGUIStreamWriter
        has to be initialized first.
        @param bWriteClassID This flag is used to select if writing of ControlID,
               leading and trailing tags is performed.
    */
    virtual void WriteToStream(const eC_Bool bWriteClassID=false);
#endif

    virtual eC_Bool DoDraw();

    virtual eC_Bool DoDrag(
        const eC_Value &vDeltaX,
        const eC_Value &vDeltaY,
        const eC_Value &vAbsX,
        const eC_Value &vAbsY);

    virtual eC_Bool DoDoubleClick(
        const eC_Value &vAbsX,
        const eC_Value &vAbsY);

    virtual eC_Bool IsHighlightable() const { return false; }

private:
    void Init();
    void DeInit();

    void DrawAxis();
    void DrawData(const eC_UInt& uiDataIndex);

    void UpdateWindow();
    void CalcDecimals();

private:
    static const eC_UInt INDEX_AXISCOLOR;
    static const eC_UInt INDEX_LINECOLOR;
    static const eC_UByte MAX_NUMBER_OF_DECIMALS;

    eC_TArray<eC_TArray<eC_Float> > m_kData;
    eC_UInt m_uiBufferSize;
    eC_UInt m_uiReadLength;
    eC_UInt m_uiReadIndex;
    eC_UInt m_uiWriteIndex;

    eC_Bool m_bAutomaticRefresh;
    eC_UInt m_uiRefreshInterval;
    eC_Bool m_bRetainValues;

    eC_Value m_vScalingX;
    eC_Value m_vScalingY;
    eC_Value m_vOffset;
    eC_Value m_vPaddingX;
    eC_Value m_vPaddingY;
    eC_UInt m_uiLabelSpacingY;

    FontResource_t m_eFontID;
    eC_UInt m_uiAxisColor;
    eC_Value m_vAxisLineWidth;
    eC_UInt m_uiLineColor;
    eC_Value m_vDrawLineWidth;

    eC_UInt m_uiNumberOfDataSources;
    eC_UInt m_uiNumberLineColors;

    eC_Bool m_bInteractive;

    eC_Bool m_bFloatForAxis;
    eC_UByte m_ubDecimals;
};

#endif
