/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef GUIPAGECONTAINER_H
#define GUIPAGECONTAINER_H

#include "GUICompositeObject.h"
#include "GUIEasing.h"
#include "GUIImageResource.h"
#include "GUINinePatch.h"

#include "GUICommonEnums.h"

// @guiliani_doxygen toplevel_control Page-Container
/**
<table border="0">
<tr>
<td width="200">@image html page_container.png</td>
<td>
The Page-Container can be used to group together various child-elements where only one child
will be displayed at a time and the others can be navigated to via dragging or the page-indicators.
</td>
</tr>
</table>
*/
// @endguiliani_doxygen

/**
Each child-element will automatically be resized to the size of the container and placed one after the other.
Depending on the orientation the elements can be dragged horizontally or vertically to switch
from one element to another. Also the indicator-icons can be clicked to switch to a specific page.

A general use-case would be displaying specific information in a page e.g. step-by-step instructions for
cooking something or setting up the application based on the user-input.

If manual navigation should not be permitted both dragging and the indicators can be deactivated.
The page-switch can be performed via the application depending on the actions on the current page.

This can help creating wizard-styled dialogs where on each page the data will be checked before
the next page will be shown.

@ingroup GUILIANI_CONTROLS
*/

/**
@brief This container can be used to create multiple pages of controls which can be swiped page-wise

this container is helpful if e.g. long texts should be displayed when having a small visible area, instead of using a scrolling-container.
*/
class CGUIPageContainer : public CGUICompositeObject
{
public:
    /** Constructor
        @param pParent Pointer to the parent object of this image.
        @param vX The x coordinate of the upper left corner of this image.
        @param vY The y coordinate of the upper left corner of this image.
        @param vWidth The width of this image.
        @param vHeight The height of this image.
        @param eID This wheel's object ID.
    */
    CGUIPageContainer(
        CGUICompositeObject *const pParent,
        const eC_Value &vX, const eC_Value &vY,
        const eC_Value &vWidth, const eC_Value &vHeight,
        const ObjectHandle_t &eID = NO_HANDLE);

    /// Default constructor
    CGUIPageContainer();
 
    /** Copy constructor
    @param kToBeCopied
    */
    CGUIPageContainer(const CGUIPageContainer& kToBeCopied);
 
    /** Assignment constructor
    @param kSource
    @return copy
    */
    CGUIPageContainer& operator=(const CGUIPageContainer& kSource);

    virtual ~CGUIPageContainer();
 
    virtual eC_Bool AddObject(CGUIObject* pkObject);

    virtual void RemoveObject(CGUIObject* pkObject);

    /** Scrolls to the given entry/node index.
    @param uiNodeIndex Index of entry/node.
    */
    virtual void ScrollTo(eC_UInt uiNodeIndex);

    /** Scrolls using animations to the given entry/node value.
    @param uiNodeIndex Index of entry/node.
    */
    virtual void ScrollToAnimated(eC_UInt uiNodeIndex);

    /** GetCurrentIndex returns the current focused entry value.
    @return eC_UInt containing the current focused index.
    */
    virtual eC_UInt GetCurrentIndex();

    // Methods derived from CGUIObject 
    virtual eC_Bool DoDraw();
    virtual eC_Bool DoPostDraw();

    virtual void DoAnimate(const eC_Value& vTimes = eC_FromInt(1));
    virtual eC_Bool DoClick(const eC_Value &vAbsX, const eC_Value &vAbsY);
    virtual eC_Bool DoButtonDown(const eC_Value &vAbsX, const eC_Value &vAbsY);
    virtual eC_Bool DoDrag(const eC_Value &vDeltaX, const eC_Value &vDeltaY, const eC_Value &vAbsX, const eC_Value &vAbsY);
    virtual eC_Bool DoDragEnd(const eC_Value &vAbsX, const eC_Value &vAbsY);
    virtual eC_Bool SetValue(const CGUIValue& rkValue);

    virtual void SetWidth(const eC_Value &vX);
    virtual void SetHeight(const eC_Value &vY);

#if defined(GUILIANI_STREAM_GUI)
    /** Reads all attributes from streaming file.
        This method is called by CGUIFactoryManager after one of the registered
        factories has created an instance of this class.
    */
    virtual void ReadFromStream();
#endif

#if defined(GUILIANI_WRITE_GUI)
    /** Writes all attributes to the streaming file. A CGUIStreamWriter
        has to be initialized first.
        @param bWriteClassID This flag is used to select if writing of control
               class ID, leading and trailing tags is performed.
    */
    virtual void WriteToStream(const eC_Bool bWriteClassID = false);
#endif

    /** Get the nine-patch which is used to draw the background of the wheel
        @return nine-patch
    */
    CGUINinePatch& GetNinePatch()
    {
        return m_kNinePatch;
    }

    /** Start animation.
    */
    virtual void StartScrollingAnimation();

    /** Stop animation.
    */
    virtual void StopScrollingAnimation();

public:
    /* Setter/Getter */

    /** Set direction for container
    @param eDirection
    */
    void SetDirection(const CGUICommonEnums::Orientation_t& eDirection);

    /** Get direction for container
    @return direction
    */
    CGUICommonEnums::Orientation_t GetDirection() const;

    /** Set if drag can be used for page-selection.
    If not pages can only be selected via indicators.
    @param bDragActive true or false
    */
    void SetDragActive(const eC_Bool& bDragActive);

    /** Get if drag is active
    @return true if active
    */
    eC_Bool GetDragActive() const;

    /** Set minimum drag-distance, which needs to exceeded before a drag happens
    @param vDragDistance
    */
    void SetDragDistance(const eC_Value& vDragDistance);

    /** Get drag-distance
    @return drag-distance
    */
    eC_Value GetDragDistance() const;

    /** Set swipe-distance, which needs to be exceeded before a page-switch happens
    @param vSwipeDistance
    */
    void SetSwipeDistance(const eC_Value& vSwipeDistance);

    /** Get swipe-distance
    @return swipe-distance
    */
    eC_Value GetSwipeDistance() const;

    /** Set overshoot for the container.
    This is the distance the pages on the edges of the container can be dragged beyond.
    @param vOvershoot
    */
    void SetOverShoot(const eC_Value& vOvershoot);

    /** Get overshoot
    @return overshoot
    */
    eC_Value GetOverShoot() const;

    /** Set the background-image
    @param eBGImage
    */
    void SetBGImage(const ImageResource_t& eBGImage);

    /** Get the background-image
    @return background-image
    */
    ImageResource_t GetBGImage() const;

    /** Set alignment of indicator-bar
    @param eAlignment
    */
    void SetIndicatorAlignment(const CGUICommonEnums::Alignment_t& eAlignment);

    /** Get alignment of indicator-bar
    @return alignment
    */
    CGUICommonEnums::Alignment_t GetIndicatorAlignment() const;

    /** Set the images which are used for the indicator-bar
    @param eImageInactive
    @param eImageActive
    */
    void SetIndicatorImages(
        const ImageResource_t& eImageInactive,
        const ImageResource_t& eImageActive);

    /** Get the images for the indicator-bar
    @param eImageInactive
    @param eImageActive
    */
    void GetIndicatorImages(
        ImageResource_t& eImageInactive,
        ImageResource_t& eImageActive) const;

    /** Set the size which is used to display the indicators
    @param vWidth if 0 the image-width will be used
    @param vHeight if 0 the image-height will be used
    */
    void SetIndicatorSize(const eC_Value& vWidth, const eC_Value& vHeight);

    /** Get the sizes for the indicator-bar
    @param vWidth
    @param vHeight
    */
    void GetIndicatorSize(eC_Value& vWidth, eC_Value& vHeight) const;

    /** Set navigation via indicators active
    @param bIndicatorsActive true if navigation is allowed
    */
    void SetIndicatorsActive(const eC_Bool& bIndicatorsActive);

    /** Return whether navigation via indicators is allowed
    @return true if active
    */
    eC_Bool GetIndicatorsActive() const;

    /** Set easing-parameters for normal easing
    this easing is used when animating from page to page
    @param eEasing
    @param vEasingTime
    */
    void SetNormalEasing(
        const CGUIEasing::EasingType_t& eEasing,
        const eC_Value& vEasingTime);

    /** Get easing-parameters for normal easing
    @param eEasing
    @param vEasingTime
    */
    void GetNormalEasing(
        CGUIEasing::EasingType_t& eEasing,
        eC_Value& vEasingTime) const;

    /** Set easing-parameters for bounce-easing
    this easing is used when pages at the edges of the container
    are dragged beyond and then nounce back
    @param eEasing
    @param vEasingTime
    */
    void SetBounceEasing(
        const CGUIEasing::EasingType_t& eEasing,
        const eC_Value& vEasingTime);

    /** Get the easing-parameters for bounce-easing
    @param eEasing
    @param vEasingTime
    */
    void GetBounceEasing(
        CGUIEasing::EasingType_t& eEasing,
        eC_Value& vEasingTime) const;

private:
    void Init();

    void DeInit();

    void ArrangeChildren();

    void UpdateChildPositions(const eC_Value& vPosition);

    void AdjustPosition(const eC_Value& vPositionDelta, const eC_Value& vPositionAbs);

    eC_Value GetValidPosition(const eC_Value& vPosition, const eC_Value& vDistanceForDirection);

private:
    CGUICommonEnums::Orientation_t m_eDirection; ///< Direction of the wheel. Either horizontal or vertical.
    eC_Bool m_bDragActive; ///< if swipe is active
    eC_Value m_vMinimumDragDistance; ///< minimum drag-distance
    eC_Value m_vSwipeDistance; ///< minimum distance for page-change
    eC_Value m_vOverShoot; ///< distance for dragging out of range

    ImageResource_t m_eBGImage; ///< image-id for the background
    CGUINinePatch m_kNinePatch; ///< nine-patch for the background

    CGUICommonEnums::Alignment_t m_eIndicatorAlignment; ///< alignment for the indicator-bar
    ImageResource_t m_ePageIndicatorInactive; ///< image-id for inactive indicator
    ImageResource_t m_ePageIndicatorActive; ///< image-id for actvie indicator
    eC_Value m_vIndicatorWidth; ///< width of one indicator
    eC_Value m_vIndicatorHeight; ///< height of one indicator
    eC_Bool m_bIndicatorsActive; ///< navigation via indicators possible

    eC_UInt m_uiDragTimeStamp;///< Time at start of animation and Timestamp of last received DoDrag()-Event
    eC_UInt m_uiDragStarted; ///< Holds the time when the last drag was started. Used to calculate the drag speed.
    CGUIPoint m_kDragStart; ///< Holds the start point of a drag.

    CGUIEasing::EasingType_t  m_eCurrentEasingType;
    eC_Value m_vCurrentEasingDuration;
    CGUIEasing::EasingType_t  m_eEasingTypeNormal;
    eC_Value m_vEasingDurationNormal;
    CGUIEasing::EasingType_t  m_eEasingTypeBounce;
    eC_Value m_vEasingDurationBounce;

    CGUIObject* m_pkCurrentElement;
    eC_UInt m_uiCurrentIndex;

    eC_Value m_vMaxPosition;
    eC_Value m_vCurrentPosition;
    eC_Value m_vTargetPosition;

    eC_TArray<CGUIRect> m_kIndicatorRects;
};

#endif
