/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef __GUI_LAYERCONTAINER_H__
#define __GUI_LAYERCONTAINER_H__

#include "GUICompositeObject.h"

/**
This container-control enables the user to define and access layers and use them for different purposes.
Its functionality depends on the presence and configuration of a DC-Wrapper. If no DC-Wrapper is currently active,
this control will behave like a normal CGUICompositeObject and will not be able to display user-content.

It can be used e.g. to display inputs from different sources directly on the screen without the need
of Guiliani to know how to handle the data. Data from camera-inputs can be used as well as generated image-data
from the application.

When activating the attribute "UserContent" Guiliani does not alter the contents of the container and expects the
application to update and maintain the contents. The DC-Wrapper is only told to refresh the respective layer on user behalf.

The attribute "LayerID" is a virtual identification which needs to correspond to the layer-configuration of the layer
contained in the DC-Wrapper @see CDCWrap()

It is possible to create complex layer-structures containing multiple layers containing both Guiliani-content
and user-controlled content. A possible scenario would be having a dialog containing Guiliani-controls with a placeholder
for a stream of video-data and on top of this another container with buttons (coming from Guiliani) to control this data.

@code

+---------------------------+
|0G                         |
|      +-------------+      |
|      |1U           |      |
|      |             |      |
|      |             |      |
|      +-------------+      |
|      |2G           |      |
|      +-------------+      |
+---------------------------+

@endcode

The numbers in the upper-left corner represent the ID for the layers, the following letter if this layer is controlled
by Guiliani (G) or the user (U).


*/
class CGUILayerContainer : public CGUICompositeObject
{
public:
    /** CGUILayerContainer constructor.
        @param pParent Pointer to the designated parent object.
        @param vX X-position relative to its parent object.
        @param vY Y-position relative to its parent object.
        @param vWidth Width of the composite object.
        @param vHeight Height of the composite object.
        @param eID Object Identifier of this composite object.
    */
    CGUILayerContainer(
        CGUICompositeObject *const pParent,
        const eC_Value &vX, const eC_Value &vY,
        const eC_Value &vWidth, const eC_Value &vHeight,
        const ObjectHandle_t &eID = NO_HANDLE);

    /** CGUILayerContainer constructor.
        @param pParent Pointer to the designated parent object.
        @param kRect Size and position relative to parent.
        @param eID Object Identifier of this composite object.
    */
    CGUILayerContainer(
        CGUICompositeObject *const pParent,
        const CGUIRect &kRect,
        const ObjectHandle_t &eID = NO_HANDLE);

    /** CGUILayerContainer destructor. */
    virtual ~CGUILayerContainer();

    /**    Standard constructor.
        @see CGUIObject().
    */
    CGUILayerContainer();

    /**
    Set the layer-id which should be used
    @param uiLayerID layer-id
    */
    void SetLayerID(const eC_UInt& uiLayerID);

    /**
    Returns the currently used layer-id
    @return layer-id
    */
    eC_UInt GetLayerID() const;

    /**
    Sets if the container is used to display user-content
    @param bUserContent
    */
    void SetUserContent(const eC_Bool& bUserContent);

    /**
    Returns if user-content is currently displayed
    @return true if user-content is present, otherwise false
    */
    eC_Bool GetUserContent() const;

    /**
    Sets if the container should apply clipping when included inside another layer-container
    @param bApplyClipping
    */
    void SetApplyClipping(const eC_Bool& bApplyClipping);

    /**
    Returns if clipping is currently used
    @return true if clipping is used, otherwise false
    */
    eC_Bool GetApplyClipping() const;

#ifdef GUILIANI_STREAM_GUI
    virtual void ReadFromStream();
#endif
#ifdef GUILIANI_WRITE_GUI
    virtual void WriteToStream(const eC_Bool bWriteClassID = false);
#endif

    virtual eC_Bool DoDraw();

    virtual eC_Bool DoPostDraw();

    virtual void HandleDraw(const CGUIRect &kClipRect);

    virtual void SetInvisible(const eC_Bool &bInvisible);

protected:

private:
    void Init();

    void CheckLayerID();

private:
    eC_Bool m_bLayerActive;
    eC_UInt m_uiLayerID;
    eC_Bool m_bUserContent;
    eC_Bool m_bApplyClipping;

    eC_Value m_vRenderingXPos;
    eC_Value m_vRenderingYPos;
    eC_UInt m_uiCurrentActiveLayer;
};
#endif
