/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/
#ifndef _GUIKNOB_H_
#define _GUIKNOB_H_

#include "GUIAbstractBar.h"
#include "GUIColorPropertyObserver.h"

#include "GUINinePatch.h"
#include "GUIRange.h"

#include "GUIPoint.h"
#include "GUIRect.h"

#include "GUICommonEnums.h"
#include "GUIImageResource.h"

/**
@brief This class represents a knoblike-control

The image of the handle is rotated around the center of the object according to the current value.

A Background-image can be specified, as well as an image representing the actual knob.
The neutral position of the image is assumed to be the indicator on the knob pointing upwards.

A start- and end-angle can be specified to define where the minimum and maximum value lie.
The mid-value will be displayed on the upper-most position of the control.

When using Axis-Control the value can be set either by dragging along the horizontal or vertical axis.
If not active the value is set according to the touch-position.

The Attribute "SmoothControl" can be used to display the handle in a smooth way, and not exactly where
the calculated values are.

@ingroup GUILIANI_CONTROLS
*/
class CGUIKnob : public CGUIAbstractBar, public CGUIColorPropertyObserver
{
public:
    /** Constructor
    Please use the SetImages method to set the correct images.
    @param pParent Pointer to the designated parent object
    @param vX X-position relative to its parent object
    @param vY Y-position relative to its parent object
    @param vWidth Width of the slider
    @param vHeight Height of the slider
    @param iValue Current value of the slider
    @param iMin Minimum value of the slider
    @param iMax Maximum value of the slider
    @param uiStepSize The step size in units of the range
    @param vStartAngle StartAngle of the slider
    @param vEndAngle EndAngle of the slider
    @param eBase Steps are based at minimum or maximum.
    @param eID Object identifier of this slider (chose NO_HANDLE if none is required) */
    CGUIKnob(
        CGUICompositeObject *const pParent,
        const eC_Value &vX, const eC_Value &vY,
        const eC_Value &vWidth, const eC_Value &vHeight,
        const eC_Value &vStartAngle,
        const eC_Value &vEndAngle,
        const eC_Int &iValue,
        const eC_Int &iMin,
        const eC_Int &iMax,
        const eC_UInt &uiStepSize,
        const CGUIRange::Base_t& eBase = CGUIRange::BASE_AT_MINIMUM,
        const ObjectHandle_t &eID = NO_HANDLE);

    virtual ~CGUIKnob();

    /// sets all images to standart images
    void SetStandardImages();

    virtual eC_Bool DoDragEnd(
        const eC_Value &vAbsX,
        const eC_Value &vAbsY);

    virtual eC_Bool DoDrag(
        const eC_Value &vDeltaX,
        const eC_Value &vDeltaY,
        const eC_Value &vAbsX,
        const eC_Value &vAbsY);

    virtual eC_Bool DoDraw();

    virtual eC_Bool IsHighlightable() const { return false; }

    /** Sets if knob is changed via circular or axis-drag
    @param bAxisControl if set to true knob is controlled according to DragOrientation
    */
    void SetAxisControl(const eC_Bool& bAxisControl);

    /** Get state of axis-control
    @return True if active
    */
    eC_Bool GetAxisControl() const;

    /** Sets the draggin orientation for axis-control
    @param eOrientation horizontal or vertical
    */
    void SetDragOrientation(const CGUICommonEnums::Orientation_t& eOrientation);

    /** Get dragging-orientation
    @return Drag-Orientation
    */
    CGUICommonEnums::Orientation_t GetDragOrientation() const;

    /** Sets smooth-control
    @param bSmoothControl if true knob-position is not neccessarily set to values
    */
    void SetSmoothControl(const eC_Bool& bSmoothControl);

    /** Return if smooth-control is set
    @return True if smooth-control is active
    */
    eC_Bool GetSmoothControl() const;

    /** Sets the images for the control
    @param eImageBG background-image for the control
    @param eImageKnob image for the knob
    */
    void SetImages(const ImageResource_t &eImageBG, const ImageResource_t &eImageKnob);

    /** Sets the images for the control
    @param eKnobBG background-image for the control
    @param eKnobNormal normal-image for the knob
    @param eKnobHighlighted highlighted-image for the knob
    @param eKnobPressed pressed-image for the knob
    @param eKnobGrayedOut grayedout-image for the knob
    */
    void SetImages(
        const ImageResource_t &eKnobBG,
        const ImageResource_t &eKnobNormal,
        const ImageResource_t &eKnobHighlighted,
        const ImageResource_t &eKnobPressed,
        const ImageResource_t &eKnobGrayedOut = DUMMY_IMAGE)
    {
        SetImages(eKnobBG, eKnobNormal);
    }

    /** Returns the currently set color
    @return currently set color */
    eC_UInt GetKnobColor() const;

    /** Sets the color for the knob as a 32 Bit hexadecimal value (0xAARRGGBB).
    @param uiKnobColor color for the knob */
    void SetKnobColor(const eC_UInt& uiKnobColor);

    /** Sets the color of the primitive as a property
    @param eKnobColor property to use */
    void SetKnobColor(const GlobalProperty_t& eKnobColor);

    /** Set the radius for the knob
    @param vKnobRadius
    */
    void SetKnobRadius(const eC_Value& vKnobRadius);

    /** Returns the radius used for the knob
    @return knob-radius
    */
    eC_Value GetKnobRadius() const;

    /** Returns the currently set color
    @return currently set color */
    eC_UInt GetLineColor() const;

    /** Sets the color of the line
    @param uiLineColor Color of drawing */
    void SetLineColor(const eC_UInt& uiLineColor);

    /** Sets the color of the line
    @param eLineColor property to use */
    void SetLineColor(const GlobalProperty_t& eLineColor);

    /** Sets the width of the line
    @param vLineWidth
    */
    void SetLineWidth(const eC_Value& vLineWidth);

    /** Return the width of the line
    @return line-width
    */
    eC_Value GetLineWidth() const;

    /** Sets the length of the line
    @param vLineLength
    */
    void SetLineLength(const eC_Value& vLineLength);

    /** Return the length of the line
    @return line-length
    */
    eC_Value GetLineLength() const;

    /** Sets Start Angle of our slider track. If start angle is bigger than EndAngle then slider will be in reverse
    @param vStartAngle the starting angle of our slider beginning from down middle and going left
    */
    void SetStartAngle(const eC_Value& vStartAngle);

    /** Returns the starting-angle
    @return start-angle
    */
    eC_Value GetStartAngle() const;

    /** Sets End Angle of our slider track. Also calls - SetSliderTrackDistance();
    @param vEndAngle the end angle of our slider starting from down middle and going left
    */
    void SetEndAngle(const eC_Value& vEndAngle);

    /** Returns the Angle at witch our slider ends.
    @return end-angle
    */
    eC_Value GetEndAngle() const;

    /** Sets Slider Value.
    Overrides SetValue function from Range class. may only be used outside of this class. 
    @param rkValue altrought the input parameter can be eC_Value(float) it is later converted to the int.
    @return boolean
    */
    virtual eC_Bool SetValue(const CGUIValue& rkValue);

    /** Sets the Range of CGUIAbstractBar class.
    @param iMin minimal value
    @param iMax maximal value
    */
    virtual void SetRange(const eC_Int& iMin, const eC_Int& iMax);


#ifdef GUILIANI_STREAM_GUI
    /** Standard constructor. Only to be called by factory.
    No user code should call this constructor, not even in
    streaming mode (that is, when GUILIANI_STREAM_GUI is defined)!
    @see CGUIKnob() */
    CGUIKnob();

    virtual void ReadFromStream();
#endif
#ifdef GUILIANI_WRITE_GUI
    virtual void WriteToStream(const eC_Bool bWriteClassID = false);
#endif

protected:
#ifndef GUILIANI_STREAM_GUI
    /// Only called by initialization list of the standard constructor.
    /// No custom user code should call this, therefore it is protected.
    CGUIKnob();
#endif

private:
    eC_Int GetSliderValue(const eC_Value &vAngle);

    void UpdateValue();

    /** this will adjust the draw angle either
    to the next valid value or to the currently set angle
    */
    void AdjustDrawAngle();

    eC_Value GetAngle(const eC_Value &vAbsX, const eC_Value &vAbsY);

    void CalculateLinePoints();

    /** initialization
    Please use the SetImages method to set the correct images.
    @param vRadius Radius of the slider
    @param iValue Current value of the slider
    @param iMin Minimum value of the slider
    @param iMax Maximum value of the slider
    @param vStartAngle StartAngle of the slider
    @param vEndAngle EndAngle of the slider
    @param uiStepSize The step size in units of the range
    @param eBase Steps are based at minimum or maximum.
    */
    void Init(
        const eC_Value &vStartAngle,
        const eC_Value &vEndAngle,
        const eC_Int &iValue,
        const eC_Int &iMin,
        const eC_Int &iMax,
        const eC_UInt &uiStepSize,
        const CGUIRange::Base_t &eBase);

    void DeInit();

private:
    static const eC_UInt INDEX_KNOBCOLOR;
    static const eC_UInt INDEX_LINECOLOR;

    CGUICommonEnums::Orientation_t m_eDragOrientation;
    eC_Bool m_bAxisControl; ///< control horizontally/vertically
    eC_Bool m_bSmoothControl; ///< smooth control and do not fit to values

    ImageResource_t m_eImageBG; ///< Knob background
    ImageResource_t m_eImageKnob; ///< Knob image

    // stuff which is used if m_eKnob == DUMMY_IMAGE
    eC_UInt m_uiKnobColor; ///< color of the knob circle
    eC_Value m_vKnobRadius; ///< radius of the knob-circle
    eC_UInt m_uiLineColor; ///< color of line
    eC_Value m_vLineWidth; ///< width of line
    eC_Value m_vLineLength; ///< length of line

    eC_Value m_vStartAngle; ///< start-angle
    eC_Value m_vEndAngle; ///< end-angle

    eC_Value m_vDrawAngle; ///< angle
    eC_Value m_vSmoothAngle; ///< smooth-angle
    eC_Value m_vLineStartX;
    eC_Value m_vLineEndX;
    eC_Value m_vLineStartY;
    eC_Value m_vLineEndY;

    eC_Int m_iImageWidth; ///< image-widht
    eC_Int m_iImageHeight; ///< image-height
};
#endif
