/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/
#ifndef GUIKEYBOARD_H_
#define GUIKEYBOARD_H_

#include "GUIColorPropertyObserver.h"
#include "GUICompositeObject.h"
#include "GUIBehaviourDecorator.h"
#include "GUIFontResource.h"
#include "GUIGeneralResource.h"
#include "GUIGeneralResourceManager.h"
#include "GUIImageResource.h"

class CGUIButton;

/// A basic on-screen keyboard control

// @guiliani_doxygen toplevel_control Keyboard
/**
<table border="0">
<tr>
<td width="200">@image html onscreen_keyboard.png</td>
<td>
The "keyboard" control is a fully customizable. Offering all necessary means for standard QWERTY layout up to phone-style keyboards.
Application specific symbols or images can be integrated easily.
There are no limitations in the number or sizes of the keyboard buttons,
making usage of Guiliani's intelligent layouters for a perfect positioning (Class: CGUIKeyboard).
</td>
</tr>
</table>
*/
// @endguiliani_doxygen
/**
This implements a basic OnScreen-Keyboard with four different layouts. (ASCII characters, ASCII capital Letters, Numerical, Symbolic)
The keyboard's buttons send their corresponding keycodes to a defineable CGUIEdit in the GUI. The CGUIEdit is chosen via the CGUIKeyboard::SetTargetObject() API and
will be the target for ALL subsequent key presses within ALL existing keyboard layouts.

The four default layouts can be created via the CreateDefaultLayouts() method.
Alternatively you can create custom layouts by attaching a CGUIKeyboardBehaviour to your own custom controls.
If you wish to use the existing default layouts, but change their visualization, you may inherit your own keyboard from CGUIKeyboard and override the
CreateKeyboardButton() method. This will be called for all the buttons by CreateDefaultLayouts().

Example showing the creation of a keyboard with default layouts:
@code
// Create keyboard
CGUIKeyboard* pkKeyboard = new CGUIKeyboard(pkMainWin, 0,0, eC_FromInt(300), eC_FromInt(100), OBJ_ID_OF_GUIEDIT, OBJ_KEYBOARD);
// Use default keyboard layouts
pkKeyboard->CreateDefaultLayouts();
// Set object which shall receive the keyboard output
CGUIKeyboard::SetTargetObject(OBJ_ID_OF_CGUIEDIT);
@endcode

Example of a custom keyboard which uses CGUIBlendButton instead of plain CGUIButton:
@code
class CMyKeyboard : public CGUIKeyboard
{
public:
CMyKeyboard(
CGUICompositeObject *const pParent,
const eC_Value &vX, const eC_Value &vY,
const eC_Value &vWidth, const eC_Value &vHeight,
const ObjectHandle_t eTargetObjectID,
const ObjectHandle_t &eID = NO_HANDLE)
: CGUIKeyboard(pParent, vX, vY, vWidth, vHeight, eTargetObjectID, eID)
{};

// Gets called for every button during the creation of the default layouts...
CGUIButton* CreateKeyboardButton(const eC_String& kLabel, const eC_UInt uiKeyCode, CGUICompositeObject* pParent)
{
// Create a CGUIBlendButton
CGUIBlendButton* pButton = new CGUIBlendButton(pParent, 0,0,0,0, kLabel, NULL, NO_HANDLE);
if (pButton)
{
// Set button attributes, such as Font, Images, NinePatch...
if (pButton->GetLabel())
{
pButton->GetLabel()->SetFont(FNT_KEYBOARD);
}
pButton->SetImages(IMG_STDCTRL_KEYBOARDBTN_STANDARD, IMG_STDCTRL_KEYBOARDBTN_PRESSED, IMG_STDCTRL_KEYBOARDBTN_HIGHLIGHTED, IMG_STDCTRL_KEYBOARDBTN_GRAYED_OUT, IMG_STDCTRL_KEYBOARDBTN_FOCUSED);
pButton->GetNinePatch().Assign(5,5,5,5);
// Assign a CGUIKeyboardBehaviour which sends the given uiKeyCode when being pressed
pButton->SetBehaviour(new CGUIKeyboardBehaviour(uiKeyCode));
}
return pButton;
}
};
@endcode


@ingroup GUILIANI_CONTROLS
*/
class CGUIKeyboard : public CGUICompositeObject, public CGUIColorPropertyObserver
{
public:
    /** CGUIKeyboard constructor.
    @param vX X-position relative to its parent object
    @param vY Y-position relative to its parent object
    @param vWidth Width of the object
    @param vHeight Height of the object
    @param pParent Pointer to the designated parent object
    @param eTargetObjectID Handle to a CGUIEdit which shall receive the output of the keyboard
    @param eID of this object
    */
    CGUIKeyboard(
        CGUICompositeObject *const pParent,
        const eC_Value &vX, const eC_Value &vY,
        const eC_Value &vWidth, const eC_Value &vHeight,
        const ObjectHandle_t eTargetObjectID,
        const ObjectHandle_t &eID = NO_HANDLE);

    /** Standard constructor
    */
    CGUIKeyboard();

    virtual ~CGUIKeyboard();

    /**
    Load a keyboard-layout for the keyboard. This can be used to change the keyboard-layout according to the active language.
    This will replace all existing keys and load them from the file.
    The content of the layout-file is structured as follows:

    @code
    <KeyboardLayout>
    <Version>1</Version>
    <Group> -> there are four groups in the keyboard representing the different types (uppercase, lowercase, numbers, symbols)
    <RowCount>XXX</RowCount> -> this is the number of rows for the current group
    <Row> -> definition for the current row
    <CharCount>XXX</CharCount> -> number of characters in the current row
    <Char>XXX</Char> -> the character representing the key (either single character, also UTF-16 or a special keyword -> below)
    </Row>
    </Group>
    ...
    </KeyboardLayout>
    @endcode

    Special Keywords:
    BACK -> is the backspace-key
    SPACE -> spacebar
    ENTER -> enter-key
    ABC -> switch to uppercase-group
    abc -> switch to lowercase-group
    123 -> switch to number-group
    SYM -> switch to symbol-group

    @param rkLayoutFile the file containing the description of the layout
    @return true on success, otherwise false
    */
    eC_Bool LoadLayout(const eC_String& rkLayoutFile);

    /**
    Load a keyboard-layout using general resources
    @param reResource the ID of the resource to use
    */
    void LoadLayout(const GeneralResource_t& reResource);

    /**
    Load a keyboard-layout from a file
    @param pkFile file to load from
    @param kFileExtension extension of file to choose correct loader
    @return true on success, otherwise false
    */
    eC_Bool LoadLayout(eC_File* pkFile, const eC_String& kFileExtension);

    /** Convert an xml-layout into a binary layout to reduce size
    @param rkInputFile input-file
    @param rkOutputFile output-file
    */
    void ConvertLayout(const eC_String& rkInputFile, const eC_String& rkOutputFile);

    /** Sets the CGUIEdit which will receive the output from all existing keyboards (or to be precise, from all CGUIKeyboardBehaviour instances).
    If no ID was explicitly given, the output will instead be sent to the currently focused object.
    @note If you choose to send the output to the currently focused object, you should make sure that they keyboard and its buttons are NOT focusable!
    @param eTargetObject ID of an instance of CGUIEdit
    */
    void SetTargetObject(const ObjectHandle_t eTargetObject) { m_eTargetObject = eTargetObject; }

    /** Forwards the given KeyCode into the corresponding DoKeyDown and DoChar slots of the CGUIEdit identified by SetTargetObject().<br/>
    Note that the following keycodes will be handled by calling DoKeyDown on the CGUIEdit:<br/>
    <table>
    <tr><th>KeyCode</th><th>Name</th>
    <tr><td>0x0008</td><td>GK_BACKSPACE</td>
    <tr><td>0x000D</td><td>GK_ACTION</td>
    <tr><td>0x007F</td><td>GK_DELETE</td>
    <tr><td>0x2190</td><td>GK_LEFT</td>
    <tr><td>0x2192</td><td>GK_RIGHT</td>
    </table>
    @param uiKeycode UTF16 KeyCode which shall be handled.
    */
    void HandleKey(const eC_UInt uiKeycode);

    /** Set the current layout
    @param eLayout Resource-ID
    */
    void SetLayout(const GeneralResource_t& eLayout);

    /** Get the currently used layout
    @return Resource-ID
    */
    GeneralResource_t GetLayout() const;

    /** Set font which is used by keyboard-buttons
    @param eFont font-id
    */
    void SetFont(const FontResource_t& eFont);

    /** Sets the text-colors for the keyboard-buttons
    @param uiColorStandard standard-color
    @param uiColorHighlighted highlighted-color
    @param uiColorGrayedOut grayedout-color
    @param uiColorPressed pressed-color
    */
    void SetColors(
        const eC_UInt& uiColorStandard,
        const eC_UInt& uiColorHighlighted,
        const eC_UInt& uiColorGrayedOut,
        const eC_UInt& uiColorPressed);

    /** Sets the text-colors for the keyboard-buttons
    @param eColorStandard standard-color
    @param eColorHighlighted highlighted-color
    @param eColorGrayedOut grayedout-color
    @param eColorPressed pressed-color
    */
    void SetColors(
        const GlobalProperty_t& eColorStandard,
        const GlobalProperty_t& eColorHighlighted,
        const GlobalProperty_t& eColorGrayedOut,
        const GlobalProperty_t& eColorPressed);

    /** Return the text-colors for the keyboard-buttons
    @param uiColorStandard standard-color
    @param uiColorHighlighted highlighted-color
    @param uiColorGrayedOut grayedout-color
    @param uiColorPressed pressed-color
    */
    void GetColors(
        eC_UInt& uiColorStandard,
        eC_UInt& uiColorHighlighted,
        eC_UInt& uiColorGrayedOut,
        eC_UInt& uiColorPressed);

    /** Sets the images for keyboard-buttons
    @param eImageNormal normal-image
    @param eImageHighlighted highlighted-image
    @param eImagePressed pressed-image
    @param eImageGrayedOut grayedout-image
    @param eImageFocused focused-image
    */
    void SetImages(
        const ImageResource_t& eImageNormal,
        const ImageResource_t& eImageHighlighted,
        const ImageResource_t& eImagePressed,
        const ImageResource_t& eImageGrayedOut,
        const ImageResource_t& eImageFocused);

    /** Return the images for keyboard-buttons
    @param eImageNormal normal-image
    @param eImageHighlighted highlighted-image
    @param eImagePressed pressed-image
    @param eImageGrayedOut grayedout-image
    @param eImageFocused focused-image
    */
    void GetImages(
        ImageResource_t& eImageNormal,
        ImageResource_t& eImageHighlighted,
        ImageResource_t& eImagePressed,
        ImageResource_t& eImageGrayedOut,
        ImageResource_t& eImageFocused);

#ifdef GUILIANI_STREAM_GUI
    virtual void ReadFromStream();
#endif
#ifdef GUILIANI_WRITE_GUI
    virtual void WriteToStream(const eC_Bool bWriteClassID = false);
#endif

protected:
    /// Initializes the keyboard.
    void Init();

    /// ID of CGUIEdit which shall receive the keyboard's output
    ObjectHandle_t m_eTargetObject;

private:
    /** Dummy implementation to avoid unintended use of compiler-generated default */
    CGUIKeyboard(const CGUIKeyboard& kSource);

    /** Dummy implementation to avoid unintended use of compiler-generated default */
    CGUIKeyboard& operator=(const CGUIKeyboard& kSource);

    /** Creates a button with the default keyboard-button design, which has a CGUIKeyboardBehaviour attached to it.
    The CGUIKeyboardBehaviour will write the first letter of the button's label into the target CGUIEdit
    @param kLabel Label to show on the button AND letter to be written into the CGUIEdit.
    @param pParent Parent object into which to add the button.
    @return newly created button object
    */
    virtual CGUIButton* CreateKeyboardButton(const eC_String& kLabel, CGUICompositeObject* pParent);

    /** Creates a button with the default keyboard-button design, which has a CGUIKeyboardBehaviour attached to it.
    The CGUIKeyboardBehaviour will write uiKeyCode into the target CGUIEdit.
    It is recommended to reimplement this method in derived classes to realize customized keyboard designs.
    @param kLabel Label to show on the button
    @param uiKeyCode UTF16 key code to write into the CGUIEdit
    @param pParent Parent object into which to add the button.
    @return newly created button object
    */
    virtual CGUIButton* CreateKeyboardButton(const eC_String& kLabel, const eC_UInt uiKeyCode, CGUICompositeObject* pParent);

    /** Creates one row of buttons in the keyboard.
    @param pParent Parent object to which the newly created buttons will be attached
    @param uiTotalNOFRowsInKeyboard Total number of rows in the keyboard
    @param uiRowIndex Index of the new row in the keyboard (Must be >= 0 and smaller than uiTotalNOFRowsInKeyboard)
    @param vXOffset Horizontal offset of the first button in pixels from left border of the parent object
    @param uiNOFButtons Number of buttons in this row.
    @param pWidths Array of eC_Values defining the width of the buttons. The array MUST have EXACTLY uiNOFButtons entries and the sum of all widths
    must equal 1. The widths define the percentual width of the buttons relative to their parent. (see CGUILayouterPercentage)
    @param pLabels Array of Strings defining the labels to be shown on the buttons. The array MUST have EXACTLY uiNOFButtons entries .
    @return Index of the first created button within pParent's list of child objects.
    */
    eC_UInt CreateKeyboardRow(CGUICompositeObject* pParent, const eC_UInt uiTotalNOFRowsInKeyboard, const eC_UInt uiRowIndex, const eC_Value vXOffset, const eC_UInt uiNOFButtons, const eC_Value* pWidths, const eC_String* pLabels);

    void UpdateButtons();

private:
    static const eC_UInt INDEX_COLOR_STANDARD;
    static const eC_UInt INDEX_COLOR_HIGHLIGHTED;
    static const eC_UInt INDEX_COLOR_GRAYEDOUT;
    static const eC_UInt INDEX_COLOR_PRESSED;

    GeneralResource_t m_eLayout; ///< used layout

    FontResource_t m_eFont; ///< used font

    ImageResource_t m_eImageNormal; ///< used images
    ImageResource_t m_eImageHighlighted; ///< used images
    ImageResource_t m_eImagePressed; ///< used images
    ImageResource_t m_eImageGrayedOut; ///< used images
    ImageResource_t m_eImageFocused; ///< used images
};

/** This class is used to interpret the layout-file on loading
*/
class KeyboardLayoutInterpreter : public CGUIGeneralResourceInterpreter
{
public:
    /** Constructor
    @param pkObject target-keyboard
    @param kFileExtension extension of the file (xml or bin)
    */
    KeyboardLayoutInterpreter(CGUIKeyboard* pkObject, const eC_String& kFileExtension) :
        CGUIGeneralResourceInterpreter(),
        m_pkObject(pkObject),
        m_kFileExtension(kFileExtension)
    {}

    /** interpret the resource-file containing the keyboard-layout
    @param pkFile input-file
    */
    virtual void Interpret(eC_File* pkFile)
    {
        if (NULL != m_pkObject)
            m_pkObject->LoadLayout(pkFile, m_kFileExtension);
    }

private:
    CGUIKeyboard * m_pkObject; ///< attached keyboard
    eC_String m_kFileExtension; ///< extension of file
};

#endif
