/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef GUIGROUPCONTAINER__H_
#define GUIGROUPCONTAINER__H_

#include "GUICompositeObject.h"
#include "GUITextResource.h"

class CGUIText;

/**
@brief This control is used to group several controls together in a group.
*/
class CGUIGroupContainer : public CGUICompositeObject
{
public:
    /** CGUIAbstractBar constructor.
    @param pParent Pointer to parent object.
    @param vX X position relative to its parent object.
    @param vY Y position relative to its parent object.
    @param vWidth Width of this object.
    @param vHeight Height of this object.
    @param eID Object Identifier of this object
    */
    CGUIGroupContainer(
        CGUICompositeObject *const pParent,
        const eC_Value &vX,
        const eC_Value &vY,
        const eC_Value &vWidth,
        const eC_Value &vHeight,
        const ObjectHandle_t &eID = NO_HANDLE);

    /** standard constructor.
    @see CGUIObject().
    */
    CGUIGroupContainer();

    virtual ~CGUIGroupContainer();

    /** Add object into container. if the container is set to auto-size, it will resize to fit the inserted object completely
    @param pkObject object to be added
    @return true if successful
    */
    virtual eC_Bool AddObject(CGUIObject* pkObject);

    /** @return A pointer to the text object that represents the caption of the button.
    To accomplish a "SetLabel" functionality to change the label, you can use
    these calls: GetLabel()->SetTextString(newString)
    */
    CGUIText *GetLabel() const;

    /** Changes the label on the button to a new CGUIText with string.
    The attributes of the text such as its font, color, size etc. will be
    copied. Subsequent calls to GetLabel() will return a CGUIText
    object.
    @param kString The new caption of the button.
    @remark If this button does not have a label at the time this method is
    called, a new CGUIText object with default attributes is created.
    */
    void SetLabel(const eC_String& kString);

    /** Changes the label on the button to a new CGUIText with ID.
    The attributes of the text such as its font, color, size etc. will be
    copied. Subsequent calls to GetLabel() will return a CGUIText
    object.
    @param eTextID The new caption of the button.
    @remark If this button does not have a label at the time this method is
    called, a new CGUIText object with default attributes is created.
    */
    void SetLabel(const TextResource_t& eTextID);

    /** Change the internal text object. This method deletes the old text.
    Use this to exchange the default CGUIText control created by this
    control object with another (possibly user-implemented) text.
    If you only want to change the text content, use SetLabel().
    @param pkText The new text object.
    */
    void SetText(CGUIText* pkText);

    /** Set x-margin of frame
    @param vFrameMarginX x-margin
    */
    void SetFrameMarginX(const eC_Value& vFrameMarginX);

    /** Get the currently set x-margin
    @return x-margin */
    eC_Value GetFrameMarginX() const;

    /** Set y-margin of frame
    @param vFrameMarginY y-margin
    */
    void SetFrameMarginY(const eC_Value& vFrameMarginY);

    /** Get the currently set y-margin
    @return y-margin */
    eC_Value GetFrameMarginY() const;

    /** Set margin of frame
    @param vFrameMarginX x-margin
    @param vFrameMarginY y-margin
    */
    void SetFrameMargin(const eC_Value& vFrameMarginX, const eC_Value& vFrameMarginY);

    /** Sets the color of the frame
    @param uiFrameColor
    */
    void SetFrameColor(const eC_UInt& uiFrameColor);

    /** Returns the currently set color of the frame
    @return frame-color
    */
    eC_UInt GetFrameColor() const;

    /** Set if autosize should be used
    @param bAutoSize if true control is resized after adding objects
    */
    void SetAutoSize(const eC_Bool& bAutoSize);

    /** Returns if autosize is active
    @return true if autosize is active
    */
    eC_Bool GetAutoSize() const;

    virtual eC_Bool DoDraw();

protected:
    /** CGUIAbstractBar constructor.
        @param pParent Pointer to parent object.
        @param kRect Structure containing x- and y-coordinates as well as width and height.
        @param eID Object Identifier of this object
    */
    CGUIGroupContainer(
        CGUICompositeObject *const pParent,
        const CGUIRect &kRect,
        const ObjectHandle_t &eID = NO_HANDLE);

    /** Copy-constructor of the CGUIAbstractBar class.
        @param kSource Source object to be copied.
    */
    CGUIGroupContainer(const CGUIGroupContainer& kSource);

    /** Assignment operator of the CGUIAbstractBar class.
        @param kSource Source object to be copied.
        @return Copied object.
    */
    CGUIGroupContainer& operator=(const CGUIGroupContainer& kSource);

    /** Copy attributes
    @param kSource source-object
    */
    void CopyAttributes(const CGUIGroupContainer& kSource);

#ifdef GUILIANI_STREAM_GUI
    virtual void ReadFromStream();
#endif
#ifdef GUILIANI_WRITE_GUI
    virtual void WriteToStream(const eC_Bool bWriteClassID = false);
#endif

private:
    void Init();

    void DeInit();

private:
    eC_Value m_vFrameMarginX; ///< x-margin of frame
    eC_Value m_vFrameMarginY; ///< y-margin of frame

    eC_Value m_vFramePaddingX; ///< x-padding of frame
    eC_Value m_vFramePaddingY; ///< y-padding of frame

    eC_UInt m_uiFrameColor; ///< color used for the frame

    CGUIText* m_pkLabel; ///< caption-text

    eC_Bool m_bAutoSize; ///< autosize after adding/removing objects
};
#endif
