/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef GUIGRAPH_H
#define GUIGRAPH_H

#include "GUIObject.h"
#include "GUIColorPropertyObserver.h"
#include "GUIFontResource.h"

#include "GUICommonEnums.h"

#include "eC_TArray.h"

/**
@brief this class is used by CGUIGraph to calculate the necessary values
for displaying the function-graph
*/
class CGUIGraphFunction
{
public:
    /** Calculate function-value
    @param fInput input to the function
    @return the value of the function
    */
    virtual eC_Float Calculate(const eC_Float& fInput) = 0;
};

/**
@brief This control can be used to draw multiple function-graphs.
Attributes for both axis and the visualization of data can be set.

With SetFunction() the corresponding functions can be set which will
be called when redrawing the graphs.

The functions must implement CGUIGraphFunction.
*/
class CGUIGraph : public CGUIObject, public CGUIColorPropertyObserver
{
public:
    /** Constructor.
        @param pParent Pointer to the designated parent object.
        @param vX X-position relative to its parent object, i.e. the x-offset from the left border of the parent object
        @param vY Y-position relative to its parent object, i.e. the y-offset from the upper border of the parent object
        @param vWidth Width of the object
        @param vHeight Height of the object
        @param eID Object identifier of this object (choose NO_HANDLE if none is required).
    */
    CGUIGraph(
        CGUICompositeObject *const pParent,
        const eC_Value &vX, const eC_Value &vY,
        const eC_Value &vWidth, const eC_Value &vHeight,
        const ObjectHandle_t &eID = NO_HANDLE);

    /// Default constructor
    CGUIGraph();

    /** Copy constructor
    @param kToBeCopied
    */
    CGUIGraph(const CGUIGraph& kToBeCopied);

    /** Assignment constructor
    @param kSource
    @return copy
    */
    CGUIGraph& operator=(const CGUIGraph& kSource);

    virtual ~CGUIGraph();

    /** Set the axis-style for the y-axis
    @param eAxisStyleX
    @param eAxisStyleY
    */
    void SetAxisStyle(
        const CGUICommonEnums::AxisStyle_t& eAxisStyleX,
        const CGUICommonEnums::AxisStyle_t& eAxisStyleY);

    /** Get the axis-style of the y-axis
    @param eAxisStyleX
    @param eAxisStyleY
    */
    void GetAxisStyle(
        CGUICommonEnums::AxisStyle_t& eAxisStyleX,
        CGUICommonEnums::AxisStyle_t& eAxisStyleY) const;

    /** Set the scaling for the axis in pixels per value
    A setting of 10.0 will plot a value every 10 pixels.
    A setting below 1.0 for the x-axis is not accepted.
    The scaling for the x-axis is also used to display the scale
    @param vScalingX scaling for x-axis
    @param vScalingY scaling for y-axis
    */
    void SetScaling(const eC_Value& vScalingX, const eC_Value& vScalingY);

    /** Return the currently active scaling
    @param vScalingX
    @param vScalingY
    */
    void GetScaling(eC_Value& vScalingX, eC_Value& vScalingY) const;

    /** Set spacing for the labels in pixels
    @param uiLabelSpacingX spacing for x-axis in pixels
    @param uiLabelSpacingY spacing for y-axis in pixels
    */
    void SetLabelSpacing(const eC_UInt& uiLabelSpacingX, const eC_UInt& uiLabelSpacingY);

    /** Return current spacing for y-axis
    @param uiLabelSpacingX spacing for x-axis in pixels
    @param uiLabelSpacingY spacing for y-axis in pixels
    */
    void GetLabelSpacing(eC_UInt& uiLabelSpacingX, eC_UInt& uiLabelSpacingY) const;

    /** Set spacing for the values in pixels
    The spacing will be used avoid calculating every value for the function.
    Instead the calculation will only be done in intervals of this value in pixels
    @param uiValueSpacing value spacing in pixels
    */
    void SetValueSpacing(const eC_UInt& uiValueSpacing);

    /** Return current value-spacing for y-axis
    @return uiValueSpacingX value-spacing in pixels
    */
    eC_UInt GetValueSpacing() const;

    /** Set base for logarithmic scale
    @param uiLogBaseX base for x-axis
    @param uiLogBaseY base for y-axis
    */
    void SetLogBase(const eC_UInt& uiLogBaseX, const eC_UInt& uiLogBaseY);

    /** Return base for logarithmic scale
    @param uiLogBaseX base for x-axis
    @param uiLogBaseY base for y-axis
    */
    void GetLogBase(eC_UInt& uiLogBaseX, eC_UInt& uiLogBaseY);

    /** Set offset for Y-axis
    This is the number of pixels the center is moved
    in horizontal/vertical direction.
    @param vOffsetX offset in pixels for x-axis
    @param vOffsetY offset in pixels for y-axis
    */
    void SetOffset(const eC_Value& vOffsetX, const eC_Value& vOffsetY);

    /** Return the offset for Y-axis
    @param vOffsetX offset in pixels for x-axis
    @param vOffsetY offset in pixels for y-axis
    */
    void GetOffset(eC_Value& vOffsetX, eC_Value& vOffsetY) const;

    /** Set the font used for labeling the axis
    @param eFontID
    */
    void SetFontID(const FontResource_t& eFontID);

    /** Return the font used for the labels
    @return font-id
    */
    FontResource_t GetFontID() const;

    /** Set the color for the axis as a value
    @param uiAxisColor
    */
    void SetAxisColor(const eC_UInt& uiAxisColor);

    /** Set the color for the axis as a property
    @param eAxisColor
    */
    void SetAxisColor(const GlobalProperty_t& eAxisColor);

    /** Return the color for the axis as a value
    @return color of axis
    */
    eC_UInt GetAxisColor() const;

    /** Set the width for the axis-lines
    @param vAxisLineWidth line-width
    */
    void SetAxisLineWidth(const eC_Value& vAxisLineWidth);

    /** Return the line-width for the axis
    @return line-width
    */
    eC_Value GetAxisLineWidth() const;

    /** Set the color for graph at specified index to color
    @param uiIndex index of graph
    @param uiLineColor color as value
    */
    void SetLineColor(const eC_UInt& uiIndex, const eC_UInt& uiLineColor);

    /** Set the color for graph at specified index to color
    @param uiIndex index of graph
    @param eLineColor color as property
    */
    void SetLineColor(const eC_UInt& uiIndex, const GlobalProperty_t& eLineColor);

    /** Return the color for graph at specified index
    @param uiIndex index of graph
    @return color as value
    */
    eC_UInt GetLineColor(const eC_UInt& uiIndex) const;

    /** Set the width for the plotted graph
    @param vLineWidth line-width
    */
    void SetLineWidth(const eC_Value& vLineWidth);

    /** Return the line-width for the plotted graph
    @return line-width
    */
    eC_Value GetLineWidth() const;

    /** Set visibility of helper-lines
    @param eHelperLines
    */
    void SetHelperLines(const CGUICommonEnums::AxisAlignment_t& eHelperLines);

    /** Return alignment of helper-lines
    @return alignment
    */
    CGUICommonEnums::AxisAlignment_t GetHelperLines() const;

    /** Set the color for the helper-lines as a value
    @param uiHelperLinesColor
    */
    void SetHelperLinesColor(const eC_UInt& uiHelperLinesColor);

    /** Set the color for the helper-lines as a property
    @param eHelperLinesColor
    */
    void SetHelperLinesColor(const GlobalProperty_t& eHelperLinesColor);

    /** Return the color for the helper-lines as a value
    @return color of helper-lines
    */
    eC_UInt GetHelperLinesColor() const;

    /** Set the width for the helper-lines
    @param vHelperLineWidth line-width
    */
    void SetHelperLineWidth(const eC_Value& vHelperLineWidth);

    /** Return the line-width for the helper-lines
    @return line-width
    */
    eC_Value GetHelperLineWidth() const;

    /** Set function for graph at specified index
    @param uiIndex index of graph
    @param pkFunction function for graph
    */
    void SetFunction(const eC_UInt& uiIndex, CGUIGraphFunction* pkFunction);

    /** Set interactive-mode.
    If true it will be possible to move the area of the control
    currently displayed by dragging the mouse. a double-click will reset
    to the center of the view.
    @param bInteractive
    */
    void SetInteractive(const eC_Bool& bInteractive);

    /** Get interactive-mode
    @return true if interactive-mode is set, else false
    */
    eC_Bool GetInteractive() const;

#if defined(GUILIANI_STREAM_GUI)
    /** Reads all object attributes from streaming file.
        This method is called by CGUIFactoryManager after one of the registered
        factories has created an instance of this class.
    */
    virtual void ReadFromStream();
#endif

#if defined(GUILIANI_WRITE_GUI)
    /** Writes all object attributes to the streaming file. A CGUIStreamWriter
        has to be initialized first.
        @param bWriteClassID This flag is used to select if writing of ControlID,
               leading and trailing tags is performed.
    */
    virtual void WriteToStream(const eC_Bool bWriteClassID=false);
#endif

    virtual eC_Bool DoDraw();

    virtual eC_Bool DoDrag(
        const eC_Value &vDeltaX,
        const eC_Value &vDeltaY,
        const eC_Value &vAbsX,
        const eC_Value &vAbsY);

    virtual eC_Bool DoDoubleClick(
        const eC_Value &vAbsX,
        const eC_Value &vAbsY);

    virtual eC_Bool IsHighlightable() const { return false; }

private:
    void Init();
    void DeInit();

    void DrawAxis();
    void DrawData(const eC_UInt& uiIndex);

    static eC_Float Lin2Log(
        const eC_UInt& uiLogFactor,
        const eC_Value& vValue);

    static eC_Float Log2Lin(
        const eC_UInt& uiLogFactor,
        const eC_Value vValue);

    void CalcDecimals();

private:
    static const eC_UInt INDEX_AXISCOLOR;
    static const eC_UInt INDEX_HELPERCOLOR;
    static const eC_UInt INDEX_LINECOLOR;
    static const eC_UByte MAX_NUMBER_OF_DECIMALS;

    CGUICommonEnums::AxisStyle_t m_eAxisStyleX;
    eC_Value m_vScalingX;
    eC_UInt m_uiLabelSpacingX;
    eC_UInt m_uiValueSpacingX;
    eC_UInt m_uiLogBaseX;
    eC_Bool m_bFloatForX;
    eC_UByte m_ubDecimalsX;

    CGUICommonEnums::AxisStyle_t m_eAxisStyleY;
    eC_Value m_vScalingY;
    eC_UInt m_uiLabelSpacingY;
    eC_UInt m_uiLogBaseY;
    eC_Bool m_bFloatForY;
    eC_UByte m_ubDecimalsY;

    eC_Value m_vOffsetX;
    eC_Value m_vOffsetY;

    FontResource_t m_eFontID;
    eC_UInt m_uiAxisColor;
    eC_Value m_vAxisLineWidth;
    eC_UInt m_uiLineColor;
    eC_Value m_vDrawLineWidth;
    CGUICommonEnums::AxisAlignment_t m_eHelpLines;
    eC_UInt m_uiHelpLineColor;
    eC_Value m_vHelpLineWidth;
    eC_TArray<CGUIGraphFunction*> m_pkFunction;

    eC_UInt m_uiNumberLineColors;

    eC_Bool m_bInteractive;
};

#endif
