/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef GUIGEOMETRYOBJECT_H_
#define GUIGEOMETRYOBJECT_H_

#include "GUIObject.h"
#include "GUIColorPropertyObserver.h"
#include "GUIColorResource.h"
#include "GUIPropertyResource.h"

/// Simple control for displaying geometrical primitives.

// @guiliani_doxygen toplevel_control Geometry Object Control
/**
<table border="0">
<tr>
<td width="200">@image html geometry_object.png</td>
<td>
The "geometry control" control offers a simple way to add geometrical primitives, such as lines or rectangles, to your user-interface
without writing customized drawing code (Class: CGUIGeometryObject).
</td>
</tr>
</table>
*/
// @endguiliani_doxygen

/** This class offers a simple way to add geometrical primitives, such as lines or rectangles, to your user-interface
without writing customized drawing code.
*/
class CGUIGeometryObject : public CGUIObject, public CGUIColorPropertyObserver
{
public:
    /// Enumeration for object shapes
    enum PrimitiveShape_t
    {
        PS_LINE1,  ///< Diagonal line, from top/left to bottom/right corner of the object
        PS_LINE2,  ///< Diagonal line, from top/right to bottom/left corner of the object
        PS_VERTICAL_LINE, ///< Vertical line, centered within the object
        PS_HORIZONTAL_LINE, ///< Horizontal line, centered within the object
        PS_ELLIPSE, ///< Ellipse, centered within the object
        PS_RECTANGLE, ///< Rectangle, Covering the entire object
        PS_ARC, ///< Arc, centered within the object
        PS_RING ///< Ring, centered within the object with specific width
    };

    /** Constructor
    @param pParent Pointer to the designated parent object
    @param vX X-position relative to its parent object
    @param vY Y-position relative to its parent object
    @param vWidth Width of the geometry object
    @param vHeight Height of the geometry object
    @param uiColor Color of drawing
    @param eShape Shape descriptor. Valid values are defined in CGUIGeometryObject::PrimitiveShape_t
    @param vThickness Thickness of pen
    @param bFilled Fill-out object (Currently only supported for rectangles)
    @param eID Object identifier of this object (choose NO_HANDLE if none is required)
    */
    CGUIGeometryObject(
        CGUICompositeObject* const pParent,
        const eC_Value& vX,
        const eC_Value& vY,
        const eC_Value& vWidth,
        const eC_Value& vHeight,
        const eC_UInt&  uiColor,
        const PrimitiveShape_t eShape,
        const eC_Value& vThickness = eC_FromInt(1),
        const eC_Bool&  bFilled = false,
        const ObjectHandle_t& eID = NO_HANDLE);

    /** Constructor
    @param pParent Pointer to the designated parent object
    @param vX X-position relative to its parent object
    @param vY Y-position relative to its parent object
    @param vWidth Width of the geometry object
    @param vHeight Height of the geometry object
    @param uiColor Color of drawing
    @param uiBorderColor boder-color of drawing
    @param eShape Shape descriptor. Valid values are defined in CGUIGeometryObject::PrimitiveShape_t
    @param vThickness Thickness of pen
    @param bFilled Fill-out object (Currently only supported for rectangles)
    @param eID Object identifier of this object (choose NO_HANDLE if none is required)
    */
    CGUIGeometryObject(
        CGUICompositeObject* const pParent,
        const eC_Value& vX,
        const eC_Value& vY,
        const eC_Value& vWidth,
        const eC_Value& vHeight,
        const eC_UInt&  uiColor,
        const eC_UInt& uiBorderColor,
        const PrimitiveShape_t eShape,
        const eC_Value& vThickness = eC_FromInt(1),
        const eC_Bool&  bFilled = false,
        const ObjectHandle_t& eID = NO_HANDLE);

#if defined(GUILIANI_STREAM_GUI)
    virtual void ReadFromStream();
#endif

#if defined(GUILIANI_WRITE_GUI)
    virtual void WriteToStream(const eC_Bool bWriteClassID = false);
#endif

    /** Standard constructor. */
    CGUIGeometryObject();

    /** Destructor */
    virtual ~CGUIGeometryObject();

    virtual eC_Bool DoDraw();

    virtual eC_Bool IsHighlightable() const { return false; }

    /** Returns the currently set border-color
    @return currently set border-color */
    eC_UInt GetBorderColor() const;

    /** Sets the border-color of the primitive as a 32 Bit hexadecimal value (0xAARRGGBB).
    @param uiBorderColor Border-Color of drawing */
    void SetBorderColor(const eC_UInt& uiBorderColor);

    /** Sets the border-color of the primitive as a property
    @param eBorderColor property to use */
    void SetBorderColor(const GlobalProperty_t& eBorderColor);

    /** Returns the currently set color
    @return currently set color */
    eC_UInt GetColor() const;

    /** Sets the color of the primitive as a 32 Bit hexadecimal value (0xAARRGGBB).
    @param uiColor Color of drawing */
    void SetColor(const eC_UInt& uiColor);

    /** Sets the color of the primitive as a property
    @param eColor property to use */
    void SetColor(const GlobalProperty_t& eColor);

    /** Sets the shape of the primitive as an enum.
    @param eShape Shape descriptor. Valid values are defined in CGUIGeometryObject::PrimitiveShape_t */
    void SetShape(const PrimitiveShape_t eShape);

    /** Gets the shape of the primitive as an enum.
    @return eShape Shape descriptor. Valid values are defined in CGUIGeometryObject::PrimitiveShape_t */
    PrimitiveShape_t GetShape() const;

    /** Sets the drawing thickness of the primitive.
    @param vThickness Thickness (in pixels) of pen */
    void SetThickness(const eC_Value& vThickness);

    /** Gets the drawing thickness of the primitive.
    @return Thickness (in pixels) of pen */
    eC_Value GetThickness() const;

    /** Determines whether the object should be filled.
    @param bFilled Fill-out object. (Currently only supported for rectangles). */
    void SetFilled(const eC_Bool& bFilled);

    /** Determines whether the object should be filled.
    @return Fill-out object. (Currently only supported for rectangles). */
    eC_Bool GetFilled() const;

    /** Sets the starting-angle
    @param vStartAngle
    */
    void SetStartAngle(const eC_Value& vStartAngle);

    /** Returns the start-angle
    @return start-angle
    */
    eC_Value GetStartAngle() const;

    /** Sets the end-angle
    @param vEndAngle
    */
    void SetEndAngle(const eC_Value& vEndAngle);

    /** Returns the end-angle
    @return end-angle
    */
    eC_Value GetEndAngle() const;

    /** Sets the rotation-angle
    @param vRotationAngle
    */
    void SetRotationAngle(const eC_Value& vRotationAngle);

    /** Return the rotation-angle
    @return rotation-angle
    */
    eC_Value GetRotationAngle() const;

    /** Sets the width for the ring
    @param vRingWidth
    */
    void SetRingWidth(const eC_Value& vRingWidth);

    /** Return the width for the ring
    @return width of the ring
    */
    eC_Value GetRingWidth() const;

private:
    void Init();

private:
    static const eC_UInt INDEX_BORDERCOLOR;
    static const eC_UInt INDEX_COLOR;

    PrimitiveShape_t m_eShape; ///< shape of the object
    eC_Value m_vThickness; ///< thickness of border
    eC_Bool m_bFilled; ///< filled
    eC_Value m_vStartAngle; ///< starting angle
    eC_Value m_vEndAngle; ///< ending angle
    eC_Value m_vRotationAngle; ///< rotation angle
    eC_Value m_vRingWidth; ///< width for ring
};

#endif
