/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef GUIFRAGMENTCONTAINER_H__
#define GUIFRAGMENTCONTAINER_H__

#include "GUICompositeObject.h"
#include "GUIImageResource.h"

// @guiliani_doxygen toplevel_control Fragment-Container
/**
<table border="0">
<tr>
<td width="200">@image html fragment.png</td>
<td>
The Fragment-Container can be used to define a part of the GUI once and re-use it in different places.
This is done by creating a dialog containing the elements of a logical group which will appear in different places
in the GUI and select the dialog name as the source for the Fragment-Container.

The Fragment-Container will load the corresponding dialog and display it inside itself. All events will be handled
as usual, so menus, listitems or other recurring parts can be defined once in a seperate place and re-used multiple times.
</td>
</tr>
</table>
*/
// @endguiliani_doxygen

/**
This control is used to embed the contents of a dialog into another dialog without the need of re-defining the content.
The AutoSize-attribute can be used to extend the size of the container to the size of the content it should display.

<b>Note:</b> It is not possible to include a Dialog1 into a fragment located in Dialog2, if Dialog1 contains a Fragment-Container
refering to Dialog2. This will create an endless recursion and will cancel with an error. Also it is not possible to reference
the parent-dialog of the current Fragment-Container.

<b>Note:</b> Since the loaded dialogs will share the same Object-IDs a call to GetObjectById will always get the elements
contained in the dialog loaded first. To modify controls inside a specific instances it is recommended to get a pointer to the
Fragment-Container by using its Object-ID and reference the child-elements via a call of GetObjectById on this pointer.

@ingroup GUILIANI_CONTROLS
*/
class CGUIFragmentContainer : public CGUICompositeObject
{
public:
#if defined(GUILIANI_STREAM_GUI) || defined(GUILIANI_WRITE_GUI)
    /** XML tag used to indicate a target dialog filename to load. */
    static const eC_Char XMLTAG_FRAGMENTFILENAME[];
#endif

    /**    Standard constructor
        @see CGUIObject().
    */
    CGUIFragmentContainer();

    // Destructor
    virtual ~CGUIFragmentContainer();

    /** CGUICheckBox constructor.
    To set the standard Guiliani images, please call the SetStandardImages() method.
    To Set the user specific images, please use the SetImages() method.
    @param pParent Pointer to parent object.
    @param vX X position relative to its parent object.
    @param vY Y position relative to its parent object.
    @param vWidth Width of the image checkbox.
    @param vHeight Height of the image checkbox.
    @param eID Object Identifier of this image checkbox
    */
    CGUIFragmentContainer(
        CGUICompositeObject *const pParent,
        const eC_Value &vX,
        const eC_Value &vY,
        const eC_Value &vWidth,
        const eC_Value &vHeight,
        const ObjectHandle_t &eID = NO_HANDLE);

   /** CGUICheckBox constructor.
    @param pParent Pointer to parent object.
    @param kRect Structure containing x- and y-coordinates as well as width and height.
    @param eID Object Identifier of this image checkbox
    */
    CGUIFragmentContainer(
        CGUICompositeObject *const pParent,
        const CGUIRect &kRect,
        const ObjectHandle_t &eID = NO_HANDLE);

    /** Copy-constructor of the CGUICheckBox class.
        @param kSource Source object to be copied.
    */
    CGUIFragmentContainer(const CGUIFragmentContainer& kSource);

    /** Operator= method of the CGUICheckBox class.
        @param kSource Source object to be copied.
        @return Destination of copy
    */
    CGUIFragmentContainer& operator=(const CGUIFragmentContainer& kSource);

    virtual void SetWidth(const eC_Value &vX);

    virtual void SetHeight(const eC_Value &vY);

    eC_Bool DoDraw();

    eC_Bool DoPostDraw();

    /** Sets the name of the fragment
    @param kFragmentName
    */
    void SetFragmentName(const eC_String& kFragmentName);

    /** Get the fragment-name
    @return fragment-name
    */
    eC_String GetFragmentName() const;

    /** set if container should resize itself
    @param bAutoSize if true, auto-resize
    */
    void SetAutoSize(const eC_Bool& bAutoSize);

    /** return if auto-resize is active
    @return true if auto-resize is active
    */
    eC_Bool GetAutoSize() const;

#ifdef GUILIANI_STREAM_GUI
    virtual void ReadFromStream();
#endif

#ifdef GUILIANI_WRITE_GUI
    virtual void WriteToStream(const eC_Bool bWriteClassID = false);
#endif

protected:
    /** Helper function for constructors. */
    void Init();

private:
    /// Helper for destructor and ReadFromStream.
    void DeInit();

    /** load the fragment
    */
    void LoadFragment();

private:
    static const eC_String ms_kOverlay;
    eC_String m_kFragmentName; ///< name of fragment
    eC_Bool m_bAutoSize; ///< resize to size of fragment

    CGUIObject* m_pkContainedFragment; ///< pointer to object

    eC_Bool m_bUseSnapshot;
    GUIBitmapPlaneID_t m_eBitmapPlaneID;

    eC_Bool m_bFragmentNameChanged;
};

#endif
