/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef GUIEXPANDINGCOMPSITE__H_
#define GUIEXPANDINGCOMPSITE__H_

#include "GUIRepositionCompositeObject.h"
#include "GUIBehaviourDecorator.h"
#include "GUICheckBox.h"

#include "GUIImageResource.h"
#include "GUIFontResource.h"
#include "GUIEasing.h"

/**
This behaviour is used to process click-events on the group-heading to expand/collapse the client-area
*/
class CGUIExpandingCompositeBehaviour : public CGUIBehaviourDecorator
{
public:
    /**
    Special constructor for the group-heading
    */
    CGUIExpandingCompositeBehaviour() :
        CGUIBehaviourDecorator()
    {
    }

    virtual eC_Bool DoClick(const eC_Value &vAbsX = eC_FromInt(-1), const eC_Value &vAbsY = eC_FromInt(-1));
};

/**
A CGUIExpandingComposite is a container which can expand/collapse via a click-event on its group-heading.
This can be used to save space when different groups with many child-elements are displayed.
*/
class CGUIExpandingComposite : public CGUICompositeObject
{
public:
    /** CGUICompositeObject standard constructor.
    @see CGUIObject().
    */
    CGUIExpandingComposite();

    /** CGUICompositeObject constructor.
    @param pParent Pointer to the designated parent object.
    @param vX X-position relative to its parent object
    @param vY Y-position relative to its parent object
    @param vWidth Width of the composite object
    @param vHeight Height of the composite object
    @param kHeadline Text to be displayed in the group-heading
    @param eID Object Identifier of this composite object (choose NO_HANDLE if none is required)
    */
    CGUIExpandingComposite(
        CGUICompositeObject* const pParent,
        const eC_Value &vX,
        const eC_Value &vY,
        const eC_Value &vWidth,
        const eC_Value &vHeight,
        const eC_String& kHeadline = "Headline",
        const ObjectHandle_t &eID = NO_HANDLE);

    /** CGUICompositeObject constructor.
    @param pParent Pointer to the designated parent object.
    @param kRect Bounding rectangle relative to parent.
    @param kHeadline Text to be displayed in the group-heading
    @param eID Object Identifier of this composite object (choose NO_HANDLE if none is required)
    */
    CGUIExpandingComposite(
        CGUICompositeObject* const pParent,
        const CGUIRect &kRect,
        const eC_String& kHeadline = "Headline",
        const ObjectHandle_t &eID = NO_HANDLE);

    /**
    Add a new object to the container. This method is overridden, since new objects should be inserted into the client-area
    @param pkObject object to add
    @return true if successful, else false
    */
    eC_Bool AddObject(CGUIObject* pkObject);

    /// This destructor is automatically virtual, as the base class destructor is virtual
    virtual ~CGUIExpandingComposite();

    virtual void SetWidth(const eC_Value &vX);

    virtual void SetHeight(const eC_Value &vY);

    /** Expand the container
    */
    void Expand();

    /** Get the current expanded-state
    @return true if expanded
    */
    eC_Bool IsExpanded() const;

    /** Set string for the healdine
    @param kHeadline
    */
    void SetHeadline(const eC_String& kHeadline);

    /** Return headline
    @return
    */
    eC_String GetHeadline() const;

    /**
    Set font for headline
    @param eFontID font-id
    */
    void SetFontID(const FontResource_t& eFontID);

    /**
    Get font for headline
    @return font-id
    */
    FontResource_t GetFontID() const;

    /**
    Set images for expand-button
    @param eImageCollapsed collapsed image
    @param eImageExpanded expanded image
    */
    void SetImages(
        const ImageResource_t& eImageCollapsed,
        const ImageResource_t& eImageExpanded);

    /**
    Get images for expand-button
    @param eImageCollapsed collapsed image
    @param eImageExpanded expanded image
    */
    void GetImages(
        ImageResource_t& eImageCollapsed,
        ImageResource_t& eImageExpanded) const;

    /**
    Set if collapse/expand will be animated
    @param bAnimated if true will animate
    */
    void SetAnimated(const eC_Bool& bAnimated);

    /**
    Return animated-state
    @return true if animated
    */
    eC_Bool GetAnimated() const;

    /**
    Set animation-duration
    @param uiDurationCollapse duration for collapse in ms
    @param uiDurationExpand duration for expand in ms
    */
    void SetAnimationDuration(
        const eC_UInt& uiDurationCollapse,
        const eC_UInt& uiDurationExpand);

    /**
    Get animation-duration
    @param uiDurationCollapse duration for collapse in ms
    @param uiDurationExpand duration for expand in ms
    */
    void GetAnimationDuration(
        eC_UInt& uiDurationCollapse,
        eC_UInt& uiDurationExpand) const;

    /**
    Get the group-heading (CGUICheckBox) to modify attributes
    @return pointer to the group-heading
    */
    CGUICheckBox* GetExpandButton() const;

    /**
    Get the group-cotainer
    @return pointer to the group-container
    */
    CGUIRepositionCompositeObject* GetContainer() const;

    virtual void DoAnimate(const eC_Value& vTimes = eC_FromInt(1));

#ifdef GUILIANI_STREAM_GUI
    virtual void ReadFromStream();
#endif // GUILIANI_STREAM_GUI

#ifdef GUILIANI_WRITE_GUI
    virtual void WriteToStream(const eC_Bool bWriteClassID = false);
#endif // GUILIANI_WRITE_GUI

private:
    /** Helper function for constructors. */
    void Init();

    /**
    Update internal elements which is the CGUICheckBox for the group-heading and the CGUICompositeObject for the client-area-container
    */
    void UpdateElements();

private:
    eC_String m_kHeadline;
    FontResource_t m_eFontID;
    ImageResource_t m_eImageCollapsed;
    ImageResource_t m_eImageExpanded;
    eC_Bool m_bAnimated;
    eC_UInt m_uiDurationCollapse;
    eC_UInt m_uiDurationExpand;

    CGUICheckBox* m_pkExpandButton;
    CGUIRepositionCompositeObject* m_pkContainer;
    CGUIExpandingCompositeBehaviour* m_pkExpandBehaviour;

    eC_Bool m_bExpanded;
    CGUIEasing m_kEasing;
    eC_Value m_vElapsedTime;
    eC_Value m_vCurrentDuration;
    eC_Value m_vEaseHeightStart;
    eC_Value m_vEaseHeightEnd;

    eC_Value m_vHeadlineHeight;
};

#endif
