/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef _GUICLOCK_H_
#define _GUICLOCK_H_

#include "GUIObject.h"
#include "GUIColorPropertyObserver.h"
#include "GUIImageResource.h"

/**
@brief This control can be used to display an analog clock.
*/
class CGUIClock : public CGUIObject, public CGUIColorPropertyObserver
{
public:

    /** Constructor
    Please use the SetImages method to set the correct images.
    @param pParent Pointer to the designated parent object
    @param vX X-position relative to its parent object
    @param vY Y-position relative to its parent object
    @param vWidth Width of the geometry object
    @param vHeight Height of the geometry object
    @param eID Object-ID
    */
    CGUIClock(
        CGUICompositeObject* const pParent,
        const eC_Value& vX,
        const eC_Value& vY,
        const eC_Value& vWidth,
        const eC_Value& vHeight,
        const ObjectHandle_t& eID = NO_HANDLE);

    /** Constructor
    Please use the SetImages method to set the correct images.
    @param pParent Pointer to the designated parent object
    @param vX X-position relative to its parent object
    @param vY Y-position relative to its parent object
    @param vWidth Width of the geometry object
    @param vHeight Height of the geometry object
    @param ubHours hours of the clock
    @param ubMinutes minutes of the clock
    @param ubSeconds seconds of the clock
    @param eID Object-ID
    */
    CGUIClock(
        CGUICompositeObject* const pParent,
        const eC_Value& vX,
        const eC_Value& vY,
        const eC_Value& vWidth,
        const eC_Value& vHeight,
        const eC_UByte& ubHours,
        const eC_UByte& ubMinutes,
        const eC_UByte& ubSeconds,
        const ObjectHandle_t& eID = NO_HANDLE);

    /** Constructor
    Please use the SetImages method to set the correct images.
    @param pParent Pointer to the designated parent object
    @param vX X-position relative to its parent object
    @param vY Y-position relative to its parent object
    @param vWidth Width of the geometry object
    @param vHeight Height of the geometry object
    @param ubHours hours of the clock
    @param ubMinutes minutes of the clock
    @param ubSeconds seconds of the clock
    @param uiHourHandleColor hour handle color
    @param uiMinuteHandleColor minute handle color
    @param uiSecondHandleColor second handle color
    @param eID Object-ID
    */
    CGUIClock(
        CGUICompositeObject* const pParent,
        const eC_Value& vX,
        const eC_Value& vY,
        const eC_Value& vWidth,
        const eC_Value& vHeight,
        const eC_UByte& ubHours,
        const eC_UByte& ubMinutes,
        const eC_UByte& ubSeconds,
        const eC_UInt& uiHourHandleColor, 
        const eC_UInt& uiMinuteHandleColor, 
        const eC_UInt& uiSecondHandleColor,
        const ObjectHandle_t& eID = NO_HANDLE);

    /** Constructor
    Please use the SetImages method to set the correct images.
    @param pParent Pointer to the designated parent object
    @param vX X-position relative to its parent object
    @param vY Y-position relative to its parent object
    @param vWidth Width of the geometry object
    @param vHeight Height of the geometry object
    @param ubHours hours of the clock
    @param ubMinutes minutes of the clock
    @param ubSeconds seconds of the clock
    @param uiHourHandleColor hour handle color
    @param uiMinuteHandleColor minute handle color
    @param uiSecondHandleColor second handle color
    @param vHourNeedleLength hour handle length
    @param vMinuteNeedleLength minute handle length
    @param vSecondNeedleLength second handle length
    @param eID Object-ID
    */
    CGUIClock(
        CGUICompositeObject* const pParent,
        const eC_Value& vX,
        const eC_Value& vY,
        const eC_Value& vWidth,
        const eC_Value& vHeight,
        const eC_UByte& ubHours,
        const eC_UByte& ubMinutes,
        const eC_UByte& ubSeconds,
        const eC_UInt& uiHourHandleColor,
        const eC_UInt& uiMinuteHandleColor,
        const eC_UInt& uiSecondHandleColor,
        const eC_Value& vHourNeedleLength, 
        const eC_Value& vMinuteNeedleLength, 
        const eC_Value& vSecondNeedleLength,
        const ObjectHandle_t& eID = NO_HANDLE);

    /** Constructor
    Please use the SetImages method to set the correct images.
    @param pParent Pointer to the designated parent object
    @param vX X-position relative to its parent object
    @param vY Y-position relative to its parent object
    @param vWidth Width of the geometry object
    @param vHeight Height of the geometry object
    @param ubHours hours of the clock
    @param ubMinutes minutes of the clock
    @param ubSeconds seconds of the clock
    @param uiHourHandleColor hour handle color
    @param uiMinuteHandleColor minute handle color
    @param uiSecondHandleColor second handle color
    @param vHourNeedleLength hour handle length
    @param vMinuteNeedleLength minute handle length
    @param vSecondNeedleLength second handle length
    @param vHourNeedleWidth hour handle width
    @param vMinuteNeedleWidth minute handle width
    @param vSecondNeedleWidth second handle width
    @param eID Object-ID
    */
    CGUIClock(
        CGUICompositeObject* const pParent,
        const eC_Value& vX,
        const eC_Value& vY,
        const eC_Value& vWidth,
        const eC_Value& vHeight,
        const eC_UByte& ubHours,
        const eC_UByte& ubMinutes,
        const eC_UByte& ubSeconds,
        const eC_UInt& uiHourHandleColor,
        const eC_UInt& uiMinuteHandleColor,
        const eC_UInt& uiSecondHandleColor,
        const eC_Value& vHourNeedleLength,
        const eC_Value& vMinuteNeedleLength,
        const eC_Value& vSecondNeedleLength,
        const eC_Value& vHourNeedleWidth, 
        const eC_Value& vMinuteNeedleWidth, 
        const eC_Value& vSecondNeedleWidth,
        const ObjectHandle_t& eID = NO_HANDLE);

    /// Destructor
    virtual ~CGUIClock();

    virtual eC_Bool DoDraw();

    virtual void DoAnimate(const eC_Value& vTimes);

    virtual eC_Bool IsHighlightable() const { return false; }

    /** SetImages
    Method to change the images used for the chart background.
    If the chart size varies from the supplied image's size, the image will automatically split into
    three portions. These are either a left/middle/right part for horizontal, or a top/middle/bottom part
    for vertical sliders. The left/top images will be constructed from the left/top half of the original
    image. Analogue, The right/bottom images will be constructor from the right/bottom half of the original image.
    The pixel row/column in the middle of the original image will be stretched to fill up the space, if the chart
    is to be displayed bigger than the original image.
    @param eImageBG Image-ID of background image
    @param eImageHour Image-ID for hour-handle
    @param eImageMinute Image-ID for minute-handle
    @param eImageSecond Image-ID for second-handle
    */
    void SetImages(
        const ImageResource_t& eImageBG,
        const ImageResource_t& eImageHour,
        const ImageResource_t& eImageMinute,
        const ImageResource_t& eImageSecond);

    /** GetImages
    @param eImageBG Image-ID of background image
    @param eImageHour Image-ID for hour-handle
    @param eImageMinute Image-ID for minute-handle
    @param eImageSecond Image-ID for second-handle
    */
    void GetImages(
        ImageResource_t& eImageBG,
        ImageResource_t& eImageHour,
        ImageResource_t& eImageMinute,
        ImageResource_t& eImageSecond) const;

    /** SetStandardImages
    Sets standard images.
    */
    void SetStandardImages();

    // needle color
    /** SetHourNeedleColor
    This method sets the color of the hour handle
    @param uiNeedleColor HEX color example - 0xFFFF0000 for red
    */
    void SetHourNeedleColor(const eC_UInt& uiNeedleColor);

    /** SetHourNeedleColor
    This method sets the color of the hour handle
    @param eNeedleColor
    */
    void SetHourNeedleColor(const GlobalProperty_t& eNeedleColor);

    /** Get Color for hour-needle
    @return color
    */
    eC_UInt GetHourNeedleColor() const;

    /** SetMinuteNeedleColor
    This method sets the color of the minute handle
    @param uiNeedleColor HEX color example - 0xFFFF0000 for red
    */
    void SetMinuteNeedleColor(const eC_UInt& uiNeedleColor);

    /** SetMinuteNeedleColor
    This method sets the color of the minute handle
    @param eNeedleColor
    */
    void SetMinuteNeedleColor(const GlobalProperty_t& eNeedleColor);

    /** Get Color for hour-needle
    @return color
    */
    eC_UInt GetMinuteNeedleColor() const;

    /** SetSecondNeedleColor
    This method sets the color of the second handle
    @param uiNeedleColor HEX color example - 0xFFFF0000 for red
    */
    void SetSecondNeedleColor(const eC_UInt& uiNeedleColor);

    /** SetSecondNeedleColor
    This method sets the color of the second handle
    @param eNeedleColor
    */
    void SetSecondNeedleColor(const GlobalProperty_t& eNeedleColor);

    /** Get Color for hour-needle
    @return color
    */
    eC_UInt GetSecondNeedleColor() const;

    /** This method sets the length of the hour handle
    @param vNeedleLength the length of the needle
    */
    void SetHourNeedleLength(const eC_Value& vNeedleLength);

    /** get length of hour-needle
    @return the length of the needle
    */
    eC_Value GetHourNeedleLength() const;

    /** This method sets the length of the minute handle
    @param vNeedleLength the length of the needle
    */
    void SetMinuteNeedleLength(const eC_Value& vNeedleLength);

    /** get length of minute-needle
    @return the length of the needle
    */
    eC_Value GetMinuteNeedleLength() const;

    /** SetSecondNeedlelength
    This method sets the length of the second handle
    @param vNeedleLength the length of the needle
    */
    void SetSecondNeedleLength(const eC_Value& vNeedleLength);

    /** get length of second-needle
    @return the length of the needle
    */
    eC_Value GetSecondNeedleLength() const;

    /** This method sets the width of the hour handle
    @param vNeedleWidth the width of the needle
    */
    void SetHourNeedleWidth(const eC_Value& vNeedleWidth);

    /** This method gets the width of the hour handle
    @return the width of the needle
    */
    eC_Value GetHourNeedleWidth() const;

    /** This method sets the width of the minute handle
    @param vNeedleWidth the width of the needle
    */
    void SetMinuteNeedleWidth(const eC_Value& vNeedleWidth);

    /** This method gets the width of the minute handle
    @return the width of the needle
    */
    eC_Value GetMinuteNeedleWidth() const;

    /** This method sets the width of the second handle
    @param vNeedleWidth the width of the needle
    */
    void SetSecondNeedleWidth(const eC_Value& vNeedleWidth);

    /** This method gets the width of the second handle
    @return the width of the needle
    */
    eC_Value GetSecondNeedleWidth() const;

    /** ToggleShadows
    This method turns the shadows of the clock handles on and off
    @param bShowShadow True to show shadows
    */
    void ShowShadows(const eC_Bool& bShowShadow);

    /** Return if shadows are shown
    @return true if shadows are shown
    */
    eC_Bool ShowShadows() const;

    /** This method sets the color of the shadows (in case they are turned on)
    @param uiShadowColor the color of the shadow in HEX example - 0xFF000000 for black (which is default)
    */
    void SetShadowColor(const eC_UInt& uiShadowColor);

    /** This method sets the color of the shadows (in case they are turned on)
    @param eShadowColor the color of the shadow as a property
    */
    void SetShadowColor(const GlobalProperty_t& eShadowColor);

    /** This method gets the color of the shadows (in case they are turned on)
    @return the color of the shadow as a property
    */
    eC_UInt GetShadowColor() const;

    /** This method turns the second handle on and off
    @param bShowSecondHandle True if second-handle should be displayed
    */
    void ShowSecondHandle(const eC_Bool& bShowSecondHandle);

    /** Return if the second-handle is shown
    @return true if second-handle is shown
    */
    eC_Bool ShowSecondHandle() const;

    /** Set autostart
    @param bAutoStart set to true if autostart
    */
    void SetAutoStart(const eC_Bool& bAutoStart);

    /** Return the autostart-state
    @return True if autostart is active
    */
    eC_Bool GetAutoStart() const;

    /** Sets Clock Value.
    Converts rkValue to hours minutes and seconds
    @param rkValue altrought the input parameter can be eC_Value(float) it is later converted to the uint.
    @return boolean
    */
    virtual eC_Bool SetValue(const CGUIValue& rkValue);

    /** Set time in seconds
    @param uiTimeInSeconds number of seconds till midnight
    */
    void SetTimeInSeconds(const eC_UInt& uiTimeInSeconds);

    /** Get time in seconds
    @return time in seconds till midnight
    */
    eC_UInt GetTimeInSeconds() const;

    /** SetTime
    This method sets the time of the clock
    @param ubHours for hours
    @param ubMinutes for minutes
    @param ubSeconds for seconds
    */
    void SetTime(const eC_UByte& ubHours, const eC_UByte& ubMinutes, const eC_UByte& ubSeconds);

    /** GetTime
    This method returns the time of the clock
    @param ubHours
    @param ubMinutes
    @param ubSeconds
    */
    void GetTime(eC_UByte& ubHours, eC_UByte& ubMinutes, eC_UByte& ubSeconds) const;

    /** setHours
    This method sets the hours
    @param uiHour for hours
    */
    void SetHour(const eC_UByte& uiHour);

    /** GetHours
    This method gets the hours
    @return hours
    */
    eC_UByte GetHour() const;

    /** SetMinutes
    This method sets the minutes
    @param uiMinute for minutes
    */
    void SetMinute(const eC_UByte& uiMinute);

    /** GetMinutes
    This method gets the minutes
    @return minutes
    */
    eC_UByte GetMinute() const;

    /** SetSeconds
    This method sets the seconds
    @param uiSecond for seconds
    */
    void SetSecond(const eC_UByte& uiSecond);

    /** GetSeconds
    This method gets the seconds
    @return m_ubSecond for seconds
    */
    eC_UByte GetSecond() const;

#ifdef GUILIANI_STREAM_GUI
    /** Standard constructor. Only to be called by factory.
    No user code should call this constructor, not even in
    streaming mode (that is, when GUILIANI_STREAM_GUI is defined)!
    @see CGUIClock() */
    CGUIClock();

    virtual void ReadFromStream();
#endif

#ifdef GUILIANI_WRITE_GUI
    virtual void WriteToStream(const eC_Bool bWriteClassID = false);
#endif

protected:
#ifndef GUILIANI_STREAM_GUI
    /// Only called by initialization list of the standard constructor.
    /// No custom user code should call this, therefore it is protected.
    CGUIClock();
#endif

private:
    void Init(
        const eC_UByte& ubHours, const eC_UByte& ubMinutes, const eC_UByte& ubSeconds,
        const eC_UInt& uiHourHandleColor, const eC_UInt& uiMinuteHandleColor, const eC_UInt& uiSecondHandleColor,
        const eC_Value& vHourNeedlelength, const eC_Value& vMinuteNeedlelength, const eC_Value& vSecondNeedlelength,
        const eC_Value& vHourNeedleWidth, const eC_Value& vMinuteNeedleWidth, const eC_Value& vSecondNeedleWidth
   );

    void DeInit();

    eC_Value GetHourAngle() const;
    eC_Value GetMinuteAngle() const;
    eC_Value GetSecondAngle() const;

    void DrawNeedle(eC_Value vAngle, eC_Value vWidth, eC_Value vlength, eC_UInt uiColor, eC_UInt uiShadowColor, ImageResource_t m_eImage);

    void CalculateEndpoint(eC_Value vAngle, eC_Value vlength);
    void RecalculateAngles();

private:
    static const eC_UInt INDEX_COLOR_HOUR;
    static const eC_UInt INDEX_COLOR_MINUTE;
    static const eC_UInt INDEX_COLOR_SECOND;
    static const eC_UInt INDEX_COLOR_SHADOW;

    //Clock backgroung
    ImageResource_t m_eImageBackground;              ///< Background Image
    //Handle Images
    ImageResource_t m_eImageHourHandle;              ///< Hour Handle Image
    ImageResource_t m_eImageMinuteHandle;            ///< Minute Handle Image
    ImageResource_t m_eImageSecondHandle;            ///< Seconds Handle Image

    //time
    eC_UByte m_ubHour;
    eC_UByte m_ubMinute;
    eC_UByte m_ubSecond;

    eC_UByte m_ubStartHours;
    eC_UByte m_ubStartMinutes;
    eC_UByte m_ubStartSeconds;

    //angles
    eC_Value m_vHourAngle;
    eC_Value m_vMinuteAngle;
    eC_Value m_vSecondAngle;

    eC_Value m_vExactSeconds;

    // ------------NEEDLES------------
    // End of the needle to be drawn
    eC_Value m_vEndX;
    eC_Value m_vEndY;

    // Shadows
    eC_Bool m_bShowShadows;

    // lengths
    eC_Value m_vHourNeedleLength;
    eC_Value m_vMinuteNeedleLength;
    eC_Value m_vSecondNeedleLength;

    // Widths
    eC_Value m_vHourNeedleWidth;
    eC_Value m_vMinuteNeedleWidth;
    eC_Value m_vSecondNeedleWidth;

    eC_Bool m_bShowSecondHandle;

    eC_Bool m_bAutoStart;
};
#endif // _GUICLOCK_H_
