/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef _GUICIRCULARSLIDER_H_
#define _GUICIRCULARSLIDER_H_

#include "GUIAbstractBar.h"
#include "GUINinePatch.h"
#include "GUIRange.h"

#include "GUIPoint.h"
#include "GUIRect.h"

#include "GUIImageResource.h"

/**
@brief This class represents a circular slider.

Like a normal slider this control has a background-image and several different images for the handle.
A Start- and Ending-Angle can be defined where 0 is sitting in the middle pointing up.

The Radius is used to define the distance where the handle is displayed around the center of the control.

The Attribute "SmoothControl" can be used to display the handle in a smooth way, and not exactly where
the calculated values are.

@ingroup GUILIANI_CONTROLS
*/
class CGUICircularSlider : public CGUIAbstractBar
{
public:
    /** Constructor
    Please use the SetImages method to set the correct images.
    @param pParent Pointer to the designated parent object
    @param vX X-position relative to its parent object
    @param vY Y-position relative to its parent object
    @param vWidth Width of the slider
    @param vHeight Height of the slider
    @param vRadius Radius of the slider
    @param vStartAngle StartAngle of the slider
    @param vEndAngle EndAngle of the slider
    @param iValue Current value of the slider
    @param iMin Minimum value of the slider
    @param iMax Maximum value of the slider
    @param uiStepSize The step size in units of the range
    @param eBase Steps are based at minimum or maximum.
    @param eID Object identifier of this slider (chose NO_HANDLE if none
    is required) */
    CGUICircularSlider(
        CGUICompositeObject *const pParent,
        const eC_Value& vX, const eC_Value &vY,
        const eC_Value& vWidth, const eC_Value &vHeight,
        const eC_Value& vRadius,
        const eC_Value& vStartAngle,
        const eC_Value& vEndAngle,
        const eC_Int& iValue,
        const eC_Int& iMin,
        const eC_Int& iMax,
        const eC_UInt& uiStepSize,
        const CGUIRange::Base_t& eBase = CGUIRange::BASE_AT_MINIMUM,
        const ObjectHandle_t& eID = NO_HANDLE);

    virtual ~CGUICircularSlider();

    /// sets all images to standart images
    void SetStandardImages();

    virtual eC_Bool DoButtonDown(const eC_Value& vAbsX, const eC_Value& vAbsY);

    virtual eC_Bool DoDrag(
        const eC_Value& vDeltaX,
        const eC_Value& vDeltaY,
        const eC_Value& vAbsX,
        const eC_Value& vAbsY);

    virtual eC_Bool DoDragEnd(
        const eC_Value& vAbsX,
        const eC_Value& vAbsY);

    virtual eC_Bool DoClick(
        const eC_Value& vAbsX,
        const eC_Value& vAbsY);

    virtual eC_Bool DoDraw();

    /** alculates the position of the knob based on its value
    Has no return. Just changes m_kKnobRect.
    @param iValue value of our slider from Range class
    */
    void CalculateKnobRect(const eC_Int& iValue);

    /** Gets angle of any x and y coordinates
    @param vAbsX X-position in screen coordinates
    @param vAbsY Y-position in screen coordinates
    @return vAngle
    */
    eC_Value GetAngle(const eC_Value& vAbsX, const eC_Value& vAbsY);

    /** SetImages
    Method to change the images used for the slider representation.
    The sizes of the knob images may differ.
    If the knob size varies from the supplied image's size, the image will automatically split into
    three portions. These are either a left/middle/right part for horizontal, or a top/middle/bottom part
    for vertical sliders. The left/top images will be constructed from the left/top half of the original
    image. Analogue, The right/bottom images will be constructor from the right/bottom half of the original image.
    The pixel row/column in the middle of the original image will be stretched to fill up the space, if the knob
    is to be displayed bigger than the original image.
    @param eSldBG ID of background image
    @param eSldKnobNormal ID of image used to represent the normal state of the slider
    @param eSldKnobHighlighted ID of image used to represent the highlighted slider
    @param eSldKnobPressed ID of image used to represent the pressed slider
    @param eSldKnobGrayedOut ID of image used to represent the grayed out slider (compat: set to DUMMY_IMAGE) */
    void SetImages(
        const ImageResource_t& eSldBG,
        const ImageResource_t& eSldKnobNormal,
        const ImageResource_t& eSldKnobHighlighted,
        const ImageResource_t& eSldKnobPressed,
        const ImageResource_t& eSldKnobGrayedOut = DUMMY_IMAGE);

    // set functions

    /** sets knob size
    @param vKnobSize knob-size
    */
    void SetKnobSize(const eC_Value& vKnobSize);

    /** gets the knobsize
    @return knob-size
    */
    eC_Value GetKnobSize() const;

    /** Sets Start Angle of our slider track. If start angle is bigger than EndAngle then slider will be in reverse
    @param vStartAngle the starting angle of our slider beginning from down middle and going left
    */
    void SetStartAngle(const eC_Value& vStartAngle);

    /** Returns the Angle at witch our slider starts.
    @return m_vStartAngle.
    */
    eC_Value GetStartAngle() const;

    /** Sets End Angle of our slider track. Also calls - SetSliderTrackDistance();
    @param vEndAngle the end angle of our slider starting from down middle and going left
    */
    void SetEndAngle(const eC_Value& vEndAngle);

    /** Returns the Angle at witch our slider ends.
    @return m_vEndAngle.
    */
    eC_Value GetEndAngle() const;

    /** Sets Radius of our circle. Also calls - SetSliderTrackDistance();
    @param vRadius the radius of our circular slider. must be equal or smaller than the half of the height or width of our object
    */
    void SetRadius(const eC_Value& vRadius);

    /** Returns the Circle radius.
    @return m_vRadius.
    */
    eC_Value GetRadius() const;

    /** Sets Slider Value.
    Overrides SetValue function from Range class. may only be used outside of this class. 
    @param rkValue altrought the input parameter can be eC_Value(float) it is later converted to the int.
    @return boolean
    */
    virtual eC_Bool SetValue(const CGUIValue& rkValue);

    /** Sets the Range of CGUIAbstractBar class.
    @param iMin minimal value
    @param iMax maximal value
    */
    void SetRange(const eC_Int& iMin, const eC_Int& iMax);

    /** Sets smooth-control
    @param bSmoothControl if true knob-position is not neccessarily set to values
    */
    void SetSmoothControl(const eC_Bool& bSmoothControl);

    /** Return if smooth-control is set
    @return True if smooth-control is active
    */
    eC_Bool GetSmoothControl() const;

#ifdef GUILIANI_STREAM_GUI
    /** Standard constructor. Only to be called by factory.
    No user code should call this constructor, not even in
    streaming mode (that is, when GUILIANI_STREAM_GUI is defined)!
    @see CGUICircularSlider() */
    CGUICircularSlider();

    virtual void ReadFromStream();
#endif
#ifdef GUILIANI_WRITE_GUI
    virtual void WriteToStream(const eC_Bool bWriteClassID = false);
#endif

protected:
#ifndef GUILIANI_STREAM_GUI
    /// Only called by initialization list of the standard constructor.
    /// No custom user code should call this, therefore it is protected.
    CGUICircularSlider();
#endif

private:
    void Init(
        const eC_Value& vRadius,
        const eC_Value& vStartAngle,
        const eC_Value& vEndAngle,
        const eC_Int& iValue,
        const eC_Int& iMin,
        const eC_Int& iMax,
        const eC_UInt& uiStepSize,
        const CGUIRange::Base_t& eBase);

    void DeInit();

    eC_Int GetSliderValue(const eC_Value &vAngle);

    void UpdateValue();
    void UpdateAngle();

private:
    CGUIRect m_kKnobRect;

    ImageResource_t m_eCSldBG;              ///< Knob background
    ImageResource_t m_eCSldKnobNormal;      ///< Knob knob normal state
    ImageResource_t m_eCSldKnobHighlighted; ///< Knob knob highlighted
    ImageResource_t m_eCSldKnobPressed;     ///< Knob knob pressed
    ImageResource_t m_eCSldKnobGrayedOut;   ///< Knob knob grayed out

    /** Size of knob in horizontal direction of the control in pixel.
    This value is set via SetKnobSizeHorizontal() in GUISlider and
    GUIScrollBar for example. */
    eC_Value m_vKnobSize;

    /// Minimum allowed knob size (in direction of slider orientation) in pixels (Defaults to 10)
    eC_Value m_vMinimumKnobSize;

    eC_Bool m_bSmoothControl; ///< smooth control and do not fit to values

    //radius of ricle inside Rect Object class
    eC_Value m_vRadius;

    // angles for user input
    eC_Value m_vStartAngle;
    eC_Value m_vEndAngle;

    eC_Value m_vAngle;
};
#endif
