/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef _GUICHART_H_
#define _GUICHART_H_

#include "GUIObject.h"
#include "GUIColorPropertyObserver.h"

#include "GUIImageResource.h"
#include "GUIFontResource.h"
#include "GUIPropertyResource.h"

#include "eC_TArray.h"

#include "GUICommonEnums.h"

/**
@brief This control can be used to visualize multiple data-sources in line- or bar-charts
*/
class CGUIChart : public CGUIObject, CGUIColorPropertyObserver
{
public:
    /// enum how to disply values
    enum DataRepresentation_t
    {
        POINTS, ///< only points
        POINTS_AND_LINES, /// points and lines
        LINES, ///< Dots of data connected with 1 line
        BARS, ///< Bars Raising out of x-axis
        PIE, ///< pie-chart (only one dataset)
        RING, ///< ring-chart (stacked datasets)
        STACKED_LINES, ///< stacked lines on top of each other
        STACKED_BARS, ///< stacked bars on top of each other
    };

    /// enum how the axis are displayed
    enum AxisRepresentation_t
    {
        NONE, ///< do not show axis
        LINE, ///< show only line of axis
        LINE_AND_SCALING, ///< Dots of data connected with 1 line
        LINE_SCALING_AND_TEXT, ///< Bars Raising out of x-axis
    };

    /** internal management
    */
    struct Data_t
    {
        eC_Value vData; ///< data-value
        eC_Value vOffset; ///< offset values for stacked charts
        eC_Int iNumberForLabel; ///< current index
    };

    /** Constructor
    Please use the SetImages method to set the correct images.
    @param pParent Pointer to the designated parent object
    @param vX X-position relative to its parent object
    @param vY Y-position relative to its parent object
    @param vWidth Width of the geometry object
    @param vHeight Height of the geometry object
    @param iAxisColor Color of the axis Ex(0xff000000)
    @param TextLabelColor Color of the Text Lables Ex(0xff000000)
    @param DrawLineColor Color of the Line that connects data points Ex(0xff000000)
    @param DrawPointColor Color of the Data Points Ex(0xff123456)
    @param BarColor Color of the Bars Ex(0xff000000)
    @param AxisLineWidth Witdh of axis lines
    @param DrawLineWidth Witdh of lines that connect Data Points
    @param vScaleLength Length of the lines that go thru X and Y axis
    @param vPointSize Size of the point to be drawn
    @param uiScalingXAxis Number of parts in X axis (lines that cross X axis)
    @param uiScalingYAxis Number of parts in Y axis (lines that cross Y axis)
    @param vMaxShownNumberY Maximum lable number on Y axis (is not a constant AND changes dinamically depending on user interaction)
    @param vMinShownNumberY Minimum lable number on Y axis (is not a constant AND changes dinamically depending on user interaction)
    @param vZoomFactorY How much zoom will enlarge Y axis (best choice 2)
    @param uiZoomFactorX How much zoom will enlarge X axis (best choice 2)
    @param eID Object identifier of this object (choose NO_HANDLE if none is required)
    */
    CGUIChart(
        CGUICompositeObject* const pParent,
        const eC_Value& vX,
        const eC_Value& vY,
        const eC_Value& vWidth,
        const eC_Value& vHeight,
        eC_UInt iAxisColor, eC_UInt TextLabelColor, eC_UInt DrawLineColor, eC_UInt DrawPointColor, eC_UInt BarColor,
        eC_Value AxisLineWidth, eC_Value DrawLineWidth, eC_Value vScaleLength, eC_Value vPointSize,
        eC_UInt uiScalingXAxis, eC_UInt uiScalingYAxis,
        eC_Value vMaxShownNumberY, eC_Value vMinShownNumberY,
        eC_Value vZoomFactorY, eC_UInt uiZoomFactorX,
        const ObjectHandle_t& eID = NO_HANDLE);

    /** Constructor
    Please use the SetImages method to set the correct images.
    @param pParent Pointer to the designated parent object
    @param vX X-position relative to its parent object
    @param vY Y-position relative to its parent object
    @param vWidth Width of the geometry object
    @param vHeight Height of the geometry object
    @param eID Object-ID
    Parameters that will set up automatically:
    (iAxisColor, TextLabelColor, DrawLineColor, DrawPointColor, BarColor,
    AxisLineWidth, DrawLineWidth, vScaleLength, vPointSize
    uiScalingXAxis, uiScalingYAxis,
    vMaxShownNumberY, vMinShownNumberY,
    vZoomFactorY, uiZoomFactorX)
    */
    CGUIChart(
        CGUICompositeObject* const pParent,
        const eC_Value& vX,
        const eC_Value& vY,
        const eC_Value& vWidth,
        const eC_Value& vHeight,
        const ObjectHandle_t& eID = NO_HANDLE);

    /** Constructor
    Please use the SetImages method to set the correct images.
    @param pParent Pointer to the designated parent object
    @param vX X-position relative to its parent object
    @param vY Y-position relative to its parent object
    @param vWidth Width of the geometry object
    @param vHeight Height of the geometry object
    @param iScalingXAxis initial scaling of x-axis
    @param iScalingYAxis initial scaling of y-axis
    @param vMaxShownNumberY Maximum lable number on Y axis (is not a constant AND changes dinamically depending on user interaction)
    @param vMinShownNumberY Minimum lable number on Y axis (is not a constant AND changes dinamically depending on user interaction)
    @param eID Object-ID
    Parameters that will set up automatically:
    (iAxisColor, TextLabelColor, DrawLineColor, DrawPointColor, BarColor,
    AxisLineWidth, DrawLineWidth, vScaleLength, vPointSize
    uiScalingXAxis, uiScalingYAxis,
    vZoomFactorY, uiZoomFactorX)
    */
    CGUIChart(
        CGUICompositeObject* const pParent,
        const eC_Value& vX,
        const eC_Value& vY,
        const eC_Value& vWidth,
        const eC_Value& vHeight,
        eC_Int iScalingXAxis, eC_Int iScalingYAxis,
        eC_Value vMaxShownNumberY, eC_Value vMinShownNumberY,
        const ObjectHandle_t& eID = NO_HANDLE);

    /** Destructor */
    virtual ~CGUIChart();

    virtual eC_Bool DoDraw();

    virtual eC_Bool DoDragEnd(const eC_Value &vAbsX, const eC_Value &vAbsY);

    virtual eC_Bool DoDrag(const eC_Value &vDeltaX, const eC_Value &vDeltaY, const eC_Value &vAbsX, const eC_Value &vAbsY);

    virtual eC_Bool DoButtonDown(const eC_Value &vAbsX, const eC_Value &vAbsY);

    virtual eC_Bool DoClick(const eC_Value& vAbsX, const eC_Value& vAbsY);

    virtual void SetWidth(const eC_Value& vX);

    virtual void SetHeight(const eC_Value& vY);

    virtual eC_Bool IsHighlightable() const { return false; }

#ifndef GUILIANI_NO_DATAPOOL
    /** Generic interface used by the DataPool for setting values in GUIObjects.
    By default, this maps directly onto the SetValue() API dealing with a single CGUIValue.
    @param rkValue Reference to the DataPoolEntry used for retrieving the value(s)
    @return The return value of the called API (Usually True if OK and False in case of error).
    */
    virtual eC_Bool SetValue(CDataPoolEntry& rkValue);
#endif

    // data
    /** AddData
    Adds data to the array. Note that the dimension starts at 0. For example if only one line of data is displayed
    then its dimension is 0. If user wish to display two lines of data, then the user needs to append second dimension (1) and so on.
    @param vDataValue eC_Value
    @param uiDimension eC_UInt
    */
    void AddData(eC_Value vDataValue, eC_UInt uiDimension = 0);

    // sets all images to standart images
    /** SetStandardImages
    Sets standard images.
    */
    void SetStandardImages();
    // set images

    /** SetImages
    Method to change the images used for the chart background.
    If the chart size varies from the supplied image's size, the image will automatically split into
    three portions. These are either a left/middle/right part for horizontal, or a top/middle/bottom part
    for vertical sliders. The left/top images will be constructed from the left/top half of the original
    image. Analogue, The right/bottom images will be constructor from the right/bottom half of the original image.
    The pixel row/column in the middle of the original image will be stretched to fill up the space, if the chart
    is to be displayed bigger than the original image.
    @param eImageBG ID of background image */
    void SetImages(const ImageResource_t &eImageBG);

    /** Sets the font used for the axis
    @param eFontId FontID
    */
    void SetFont(const FontResource_t& eFontId);

    // potentially public methods
    /** ZoomInY
    This method zooms out in y-axis (which means that the range between min and max is smaller)
    */
    void ZoomInY();

    /** ZoomOutY
    This method zooms out of y-axis (which means that the range between min and max is bigger)
    */
    void ZoomOutY();

    // increase/decrease x

    /** ZoomInX
    This method zooms in of x-axis (which means that less data points will be drawn in a single chart frame)
    */
    void ZoomInX();

    /** ZoomOutX
    This method zooms out of x-axis (which means that more data points will be drawn in a single chart frame)
    */
    void ZoomOutX();

    /**
    This method changes the representation of data points. It can be points, points and lines, lines or bars.
    @param eDataRepresentation data representation
    */
    void SetDataRepresentation(const DataRepresentation_t& eDataRepresentation);

    /**
    This method returns the currently set representation of data points. It can be points, points and lines, lines or bars.
    @return data representation
    */
    DataRepresentation_t GetDataRepresentation() const;

    /**
    This method changes the representation of x-axis. (invisible, only-axis, axis and scaling, axis scaling and labels)
    @param eXAxis
    */
    void SetRepresentationXAxis(const AxisRepresentation_t& eXAxis);

    /**
    This method return the currently set representation of x-axis. (invisible, only-axis, axis and scaling, axis scaling and labels)
    @return horizontal axis-representation
    */
    AxisRepresentation_t GetRepresentationXAxis() const;

    /**
    This method changes the representation of y-axis. (invisible, only-axis, axis and scaling, axis scaling and labels)
    @param eYAxis
    */
    void SetRepresentationYAxis(const AxisRepresentation_t& eYAxis);

    /**
    This method return the currently set representation of y-axis. (invisible, only-axis, axis and scaling, axis scaling and labels)
    @return vertical axis-representation
    */
    AxisRepresentation_t GetRepresentationYAxis() const;

    /**
    This method sets display of helper-lines (none, hor, ver or both)
    @param eRepresentation type of helper-lines
    */
    void SetHelperLineRepresentation(const CGUICommonEnums::AxisAlignment_t& eRepresentation);

    /**
    This method returns the representation type of the helper lines
    @return representation type of the helper lines
    */
    CGUICommonEnums::AxisAlignment_t GetHelperLineRepresentation() const;

    /**
    This method sets the color of the horizontal helper lines.
    @param uiColor color
    */
    void SetHorizontalHelperLineColor(const eC_UInt& uiColor);

    /**
    This method sets the color of the horizontal helper lines.
    @param eColor color
    */
    void SetHorizontalHelperLineColor(const GlobalProperty_t& eColor);

    /**
    This method returns the color of the horizontal helper lines
    @return color
    */
    eC_UInt GetHorizontalHelperLineColor() const;

    /**
    This method sets the color of the horizontal helper lines
    @param uiColor color
    */
    void SetVerticalHelperLineColor(const eC_UInt& uiColor);

    /**
    This method sets the color of the horizontal helper lines
    @param eColor color
    */
    void SetVerticalHelperLineColor(const GlobalProperty_t& eColor);

    /**
    This method returns the color of the horizontal helper lines
    @return color
    */
    eC_UInt GetVerticalHelperLineColor() const;

    /**
    This method turns on and off the value feedback method. ONLY works in bar-chart representation.
    Value feedback method shows the data value of the bar that is clicked by the user.
    @param bShowBarData
    */
    void SetShowBarData(const eC_Bool& bShowBarData);

    /** Return if bar-data is currently shown
    @return true if active
    */
    eC_Bool GetShowBarData() const;

    /** SetAxisColor
    @param uiAxisColor
    */
    void SetAxisColor(const eC_UInt& uiAxisColor);

    /** SetAxisColor
    @param eAxisColor
    */
    void SetAxisColor(const GlobalProperty_t& eAxisColor);

    /** GetAxisColor
    This method returns the color of the x- and y-axis (scaling color is also included).
    @return Axis-color
    */
    eC_UInt GetAxisColor() const;

    /** SetTextLabelColor
    @param uiTextLabelColor
    */
    void SetTextLabelColor(const eC_UInt& uiTextLabelColor);

    /** SetTextLabelColor
    @param eTextLabelColor
    */
    void SetTextLabelColor(const GlobalProperty_t& eTextLabelColor);

    /** GetDrawLineColor
    This method returns the color of the text that is used to draw all labels.
    @return Color of text-label
    */
    eC_UInt GetTextLabelColor() const;

    /** SetLineColor
    @param uiIndex eC_Uint is the indes of the array of colors (starts from 0)
    @param uiLineColor
    */
    void SetLineColor(const eC_UInt& uiIndex, const eC_UInt& uiLineColor);

    /** SetLineColor
    @param uiIndex eC_Uint is the indes of the array of colors (starts from 0)
    @param eLineColor
    */
    void SetLineColor(const eC_UInt& uiIndex, const GlobalProperty_t& eLineColor);

    /** GetLineColor
    This method returns the color of the line (line-chart representation) depending on the dymensions.
    Default is 0 (first data array)
    @param uiIndex
    @return line-color of index
    */
    eC_UInt GetLineColor(const eC_UInt& uiIndex) const;

    /** SetPointColor
    @param uiIndex eC_Uint is the indes of the array of colors (starts from 0)
    @param uiPointColor
    */
    void SetPointColor(const eC_UInt& uiIndex, const eC_UInt& uiPointColor);

    /** SetPointColor
    @param uiIndex eC_Uint is the indes of the array of colors (starts from 0)
    @param ePointColor
    */
    void SetPointColor(const eC_UInt& uiIndex, const GlobalProperty_t& ePointColor);

    /** GetPointColor
    This method returns the color of the point (point-chart representation) depending on the dymensions.
    Default is 0 (first data array)
    @param uiIndex eC_UInt
    @return point-color of index
    */
    eC_UInt GetPointColor(const eC_UInt& uiIndex) const;

    /** SetBarColor
    @param uiIndex is the indes of the array of colors (starts from 0)
    @param uiBarColor
    */
    void SetBarColor(const eC_UInt& uiIndex, const eC_UInt& uiBarColor);

    /** SetBarColor
    @param uiIndex is the indes of the array of colors (starts from 0)
    @param eBarColor
    */
    void SetBarColor(const eC_UInt& uiIndex, const GlobalProperty_t& eBarColor);

    /** GetBarColor
    This method returns the color of the bar (bar-chart representation) depending on the dymensions.
    Default is 0 (first data array)
    @param uiIndex eC_UInt
    @return bar-color of index
    */
    eC_UInt GetBarColor(const eC_UInt& uiIndex) const;

    /** SetSelectionColor
    @param uiSelectionColor
    */
    void SetSelectionColor(const eC_UInt& uiSelectionColor);

    /** SetSelectionColor
    @param eSelectionColor
    */
    void SetSelectionColor(const GlobalProperty_t& eSelectionColor);

    /** GetBarColor
    This method returns the color of the selection
    Default is 0 (first data array)
    @return Selection-color
    */
    eC_UInt GetSelectionColor() const;

    /** SetZoomFactorX
    This method sets the zoom factor of x-axis. That means that each time when the user zooms out the x-axis the ignore factor
    (data displayed per scale (default is 1)) will be multiplied by this number
    @param vZoomFactorX for x-axis
    */
    void SetZoomFactorX(const eC_Value& vZoomFactorX);

    /** GetZoomFactorX
    This method returns the zoom factor of X-axis.
    @return m_uiZoomFactorX eC_Value
    */
    eC_Value GetZoomFactorX() const;

    /** SetZoomFactorY
    This method sets the zoom factor of y-axis. That means that each time when the user zooms out the y-axis the min and max
    numbers will be multiplied by this number.
    @param vZoomFactorY for y-axis
    */
    void SetZoomFactorY(const eC_Value& vZoomFactorY);

    /** GetZoomFactorY
    This method returns the zoom factor of Y-axis.
    @return m_vZoomFactorY eC_Value
    */
    eC_Value GetZoomFactorY() const;

    /** SetScalingXAxis
    This method sets the scaling of x-axis (amount of partitions - in how many parts will be x-axis divided)
    This dictetes how many datapoints (if datapoints > scaling) will be displayed in one chart frame
    @param uiScalingXAxis eC_UInt
    */
    void SetScalingXAxis(const eC_UInt& uiScalingXAxis);

    /** GetScalingXAxis
    This method returns the scaling of X-axis.
    @return m_uiScalingXAxis eC_UInt
    */
    eC_UInt GetScalingXAxis() const;

    /** SetScalingYAxis
    This method sets the scaling of y-axis (amount of partitions - in how many parts will be y-axis divided)
    This dictetes how many labels will be on y-axis (not including the origin label)
    @param uiScalingYAxis eC_UInt
    */
    void SetScalingYAxis(const eC_UInt& uiScalingYAxis);

    /** GetScalingYAxis
    This method returns the scaling of Y-axis.
    @return m_uiScalingYAxis eC_UInt
    */
    eC_UInt GetScalingYAxis() const;

    /** SetAxisLineWidth
    This method sets the with of axis lines (that also includes the width of scaling lines)
    @param vAxisLineWidth eC_Value
    */
    void SetAxisLineWidth(const eC_Value& vAxisLineWidth);

    /** GetAxisLineWidth
    This method returns the visual width of the axis lines (scaling lines are also included)
    @return m_vAxisLineWidth eC_Value
    */
    eC_Value GetAxisLineWidth() const;

    /** SetDrawLineWidth
    This method sets the with of lines that are used to connect data points
    @param vDrawLineWidth eC_Value
    */
    void SetDrawLineWidth(const eC_Value& vDrawLineWidth);

    /** GetDrawLineWidth
    This method returns the visual width of the draw lines (line-chart representation)
    @return m_vDrawLineWidth eC_Value
    */
    eC_Value GetDrawLineWidth() const;

    /** SetScaleLength
    This method sets the lenght of scale lines (both x- and y-axis)
    @param vScaleLength eC_Value
    */
    void SetScaleLength(const eC_Value& vScaleLength);

    /** GetScaleLength
    This method returns the visual lenght of the scales (x- and y-axis scaling)
    @return m_vScaleLength eC_Value
    */
    eC_Value GetScaleLength() const;

    /** SetPointSize
    This method sets the size of points that are used to draw data on the screen
    @param vPointSize eC_Value
    */
    void SetPointSize(const eC_Value& vPointSize);

    /** GetPointSize
    This method returns the visual size of the datapoints
    @return m_vPointSize eC_Value
    */
    eC_Value GetPointSize() const;

    /** SetStartLabelValue
    This method sets the value label at which x-axis starts
    @param iStartValue eC_Int
    */
    void SetStartLabelValue(const eC_Int& iStartValue);

    /** Get the starting value for x-axis
    @return starting-value
    */
    eC_Int GetStartLabelValue() const;

    /** Sets the minimum number of Y axis scaling
    @param vMinNumber is the minimum number
    */
    void SetMinValueY(const eC_Value& vMinNumber);

    /** Sets the maximum number of Y axis scaling
    @param vMaxNumber is the maximum number
    */
    void SetMaxValueY(const eC_Value& vMaxNumber);

    /** Sets the range for shown data points
    It will start always from the first parameter, but if the last parameter will exceed normal scaling of x axis
    then the chart will display more datapoints than the input suggests.
    @param iMinNumber is the data point to be shown first
    @param iMaxNumber is the data point to be shown last
    */
    void SetRangeX(const eC_Int& iMinNumber, const eC_Int& iMaxNumber);

    /** Get the range for shown datapoints
    @param iMinNumber is the data point to be shown first
    @param iMaxNumber is the data point to be shown last
    */
    void GetRangeX(eC_Int& iMinNumber, eC_Int& iMaxNumber) const;

    /** Sets minimum and maximum value for y-axis
    @param vMinValue minimum value
    @param vMaxValue maximum value
    */
    void SetRangeY(const eC_Value& vMinValue, const eC_Value& vMaxValue);

    /** Get the range for y-axis
    @param vMinValue minimum value
    @param vMaxValue maximum value
    */
    void GetRangeY(eC_Value& vMinValue, eC_Value& vMaxValue) const;

    /** Get the currently focussed area of data
    @return focus-area
    */
    CGUIRect GetFocusArea() const;

    /** Sets the minimum data point as first datapoint of the x axis
    For example - 100 data points and scaling of x axis is 10
    SetMinShownData(20) will make it so that the x axis will display 20th data point first and 29th data point last
    @param iMinShownData is the data point to be shown first
    */
    void JumpToDataPoint(const eC_Int& iMinShownData);

    //Data managment
    /** JumpToTheEnd
    Whenever this method will be called the data-point view will jump to the end. That means that only the last datapoints will be
    displayed.
    */
    void JumpToTheEnd();

    /** WipeAllData
    This method deletes all data points that are saved in this class. Note, if you are using DataPool then when the DataPool will notify this class again,
    all data points will come back.
    */
    void WipeAllData();

    /** KeepLastXElemenets
    This method keeps x amount of data points that are saved in this class. Note, if you are using DataPool then when the DataPool will notify this class again,
    all data points will come back.
    @param uiDataAmountToKepp eC_UInt
    */
    void KeepLastXElemenets(const eC_UInt& uiDataAmountToKepp);

private:
    void Init(eC_UInt iAxisColor, eC_UInt TextLabelColor, eC_UInt DrawLineColor, eC_UInt DrawPointColor, eC_UInt BarColor,
        eC_Value AxisLineWidth, eC_Value DrawLineWidth, eC_Value vScaleLength, eC_Value vPointSize,
        eC_UInt uiScalingXAxis, eC_UInt uiScalingYAxis,
        eC_Value vMaxShownNumberY, eC_Value vMinShownNumberY,
        eC_Value vZoomFactorY, eC_UInt uiZoomFactorX);

    void DeInit();

    // add colors
    /** AddDrawLineColor
    This method simply appends colors to the color pull
    If new dimension for chart data is created
    The user should append color so that the second line chart chould have its own color
    If the color is not appended then the second line chart will use the same color as the first
    @param uiDrawLineColor eC_UInt
    */
    void AddDrawLineColor(const eC_UInt& uiDrawLineColor);

    /** AddDrawPointColor
    This method simply appends colors to the color pull
    If new dimension for chart data is created
    The user should append color so that the second point chart chould have its own color
    If the color is not appended then the second point chart will use the same color as the first
    @param uiDrawPointColor eC_UInt
    */
    void AddDrawPointColor(const eC_UInt& uiDrawPointColor);

    /** AddBarColor
    This method simply appends colors to the color pull
    If new dimension for chart data is created
    The user should append color so that the second bar chart chould have its own color
    If the color is not appended then the second bar chart will use the same color as the first
    @param uiBarColor eC_UInt
    */
    void AddBarColor(const eC_UInt& uiBarColor);

    /**
    Draws all charts and decides which visualization is used
    */
    void DrawAllCharts();

    /**
    Draws line chart.
    @param uiChartIndex index of the chart data
    */
    void DrawLineChart(const eC_UInt& uiChartIndex);

    /**
    Draws stacked line chart.
    @param uiChartIndex index of the chart data
    */
    void DrawStackedLineChart(const eC_UInt& uiChartIndex);

    /**
    Draws point chart.
    @param uiChartIndex index of the chart data
    */
    void DrawPointChart(const eC_UInt& uiChartIndex);

    /**
    Draws stacked point chart.
    @param uiChartIndex index of the chart data
    */
    void DrawStackedPointChart(const eC_UInt& uiChartIndex);

    /**
    Draws bar chart.
    @param uiChartIndex index of the chart data
    */
    void DrawBarChart(const eC_UInt& uiChartIndex);

    /**
    Draws stacked bar chart.
    @param uiChartIndex index of the chart data
    */
    void DrawStackedBarChart(const eC_UInt& uiChartIndex);

    /**
    Decides which color to choose for a given chart index.
    This function has two fallbacks:
    1 Fallback: The first color defined by the user.
    2 Fallback: black
    @return color code
    */
    eC_UInt GetChartColor(const eC_UInt& uiChartIndex, const eC_TArray<eC_UInt>& kColors);

    /**
    Zooms the data focus window by the given ratio and keeps the center position.
    @param vZoomFactorX scale for the x dimension
    @param vZoomFactorY scale for the y dimension
    */
    void Zoom(const eC_Value& vZoomFactorX, const eC_Value& vZoomFactorY);

    /**
    Draw the labels and lines for the x axis
    */
    void DrawXAxis();

    /**
    Calculates the numerical step between y labels.
    Caution the step size is returned negative
    @return negative step size
    */
    eC_Value CalculateYLabelStepSizeInData();

    /**
    Calculates the pixel step between y labels.
    @return step size pixel
    */
    eC_Value CalculateYLabelStepSizeInPixel();

    /**
    Draw the labels and lines for the y axis
    */
    void DrawYAxis();

    /**
    Calculates data x position based on the given x position relative to the chart area.
    @param vXPos relative x position
    @return data x position
    */
    eC_Value TranslateRelXPosToDataPos(const eC_Value& vXPos);

    /**
    Calculates data y position based on the given y position relative to the chart area.
    @param vYPos relative y position
    @return data y position
    */
    eC_Value TranslateRelYPosToDataPos(const eC_Value& vYPos);

    /**
    Calculates absolute x position based on the given data x position.
    @param vXPos data x position
    @return absolute x position
    */
    eC_Value TranslateDataPosToAbsXPos(const eC_Value& vXPos);

    /**
    Calculates absolute y position based on the given data y position.
    @param vYPos data y position
    @return absolute y position
    */
    eC_Value TranslateDataPosToAbsYPos(const eC_Value& vYPos);

    // labeling data
    /** DrawYLabel
    This method draws the label of Y-axis. This method is different from the other label methods in that it draws the label
    so that is is centered in relation to y-axis scaling
    @param vXPos eC_Value
    @param vYPos eC_Value
    @param kLabel eC_String
    @param uiPreviousColor eC_Int
    */
    void DrawYLabel(
        const eC_Value& vXPos,
        const eC_Value& vYPos,
        const eC_String& kLabel,
        const eC_UInt& uiPreviousColor);

    /** DrawXLabel
    This method draws the label of X-axis. This method is different from the other label methods in that it draws the label
    so that is is centered in relation to x-axis scaling
    @param vXPos eC_Value
    @param vYPos eC_Value
    @param kLabel eC_String
    @param uiPreviousColor eC_Int
    */
    void DrawXLabel(
        const eC_Value& vXPos,
        const eC_Value& vYPos,
        const eC_String& kLabel,
        const eC_UInt& uiPreviousColor);

    /**
    Draws the coordinate of the selected value as text
    */
    void DrawSelectionLabel();

    // formats the long float string into something more readable
    /** FormatString
    This method formats strings that are used to label the y-axis. Because all y-axis scalings are eC_Value, this
    method removes all unneeded zeroes at the end.
    @param vStringToFormat eC_String
    */
    void FormatString(eC_String &vStringToFormat);

    /** RecalculateChartArea
    This method recalculates the chart area.
    */
    void RecalculateChartArea();

    /** RecalculateChartOffsets
    This method recalculates the chart offset.
    */
    void RecalculateChartOffsets();

    /**
    Searches for the data visualization at the given position
    @param vXAbsPos absolute x position
    @param vYAbsPos absolute y position
    @param uiChartIndex output parameter for chart index of the found data vis
    @param uiValueIndex output parameter for value index of the found data vis
    @return true if data is visualized at the given point
    */
    eC_Bool DetectDataEntry(
        const eC_Value& vXAbsPos,
        const eC_Value& vYAbsPos,
        eC_UInt& uiChartIndex,
        eC_UInt& uiValueIndex);

    /**
    Creates a point visualization rectangle for given chart and value
    @param uiChartIndex chart index
    @param uiValueIndex value index
    @param vPointSize point radius
    @param pkRect output parameter rectable for the point
    */
    void GetPointRect(
        const eC_UInt& uiChartIndex,
        const eC_UInt& uiValueIndex,
        const eC_Value& vPointSize,
        CGUIRect& pkRect);

    /**
    Creates a stacked point visualization rectangle for given chart and value
    @param uiChartIndex chart index
    @param uiValueIndex value index
    @param vPointSize point radius
    @param pkRect output parameter rectable for the point
    */
    void GetStackedPointRect(
        const eC_UInt& uiChartIndex,
        const eC_UInt& uiValueIndex,
        const eC_Value& vPointSize,
        CGUIRect& pkRect);

    /**
    Creates a line visualization rectangle for given chart and value
    @param uiChartIndex chart index
    @param uiValueIndex value index
    @param pkRect output parameter rectable for the point
    */
    void GetLineRect(
        const eC_UInt& uiChartIndex,
        const eC_UInt& uiValueIndex,
        CGUIRect& pkRect);

    /**
    Creates a stacked line visualization rectangle for given chart and value
    @param uiChartIndex chart index
    @param uiValueIndex value index
    @param pkRect output parameter rectable for the point
    */
    void GetStackedLineRect(
        const eC_UInt& uiChartIndex,
        const eC_UInt& uiValueIndex,
        CGUIRect& pkRect);

    /**
    Creates a bar visualization rectangle for given chart and value
    @param uiChartIndex chart index
    @param uiValueIndex value index
    @param pkRect output parameter rectable for the point
    */
    void GetBarRect(
        const eC_UInt& uiChartIndex,
        const eC_UInt& uiValueIndex,
        CGUIRect& pkRect);

    /**
    Creates a bar visualization rectangle for given chart and value
    @param uiChartIndex chart index
    @param uiValueIndex value index
    @param pkRect output parameter rectable for the point
    */
    void GetStackedBarRect(
        const eC_UInt& uiChartIndex,
        const eC_UInt& uiValueIndex,
        CGUIRect& pkRect);

    /**
    Calculate offsets for drawing stacked versions
    */
    void CalculateOffsets();

    eC_UInt GetStartValue();

    eC_UInt GetEndValue(const eC_UInt& uiMaxValue);

    void UpdateFocussedData();

public:
#ifdef GUILIANI_STREAM_GUI
    /** Standard constructor. Only to be called by factory.
    No user code should call this constructor, not even in
    streaming mode (that is, when GUILIANI_STREAM_GUI is defined)!
    @see CGUIChart() */
    CGUIChart();

    virtual void ReadFromStream();
#endif
#ifdef GUILIANI_WRITE_GUI
    virtual void WriteToStream(const eC_Bool bWriteClassID = false);
#endif

protected:
#ifndef GUILIANI_STREAM_GUI
    /// Only called by initialization list of the standard constructor.
    /// No custom user code should call this, therefore it is protected.
    CGUIChart();
#endif

private:
    /**
    Updates m_kDataBoundingBox if the m_bIsDataAreaDirty flag is true.
    */
    void UpdateDataArea();


private:
    static const eC_UInt INDEX_AXISCOLOR;
    static const eC_UInt INDEX_TEXTCOLOR;
    static const eC_UInt INDEX_SELECTIONCOLOR;
    static const eC_UInt INDEX_HELPERHORCOLOR;
    static const eC_UInt INDEX_HELPERVERCOLOR;
    static const eC_UInt INDEX_POINTCOLOR;
    static const eC_UInt INDEX_LINECOLOR;
    static const eC_UInt INDEX_BARCOLOR;

    eC_UInt m_uiNumberPointColors;
    eC_UInt m_uiNumberLineColors;
    eC_UInt m_uiNumberBarColors;

    ///< Background Image
    ImageResource_t m_eImageBackground;

    ///<The area in which the chart is drawn
    CGUIRect m_kChartArea;

    /// If this flag is true the navigation is restricted; otherwise its free
    eC_Bool m_bRestrictNavigation;

    ///<The area in data space wich is shown
    CGUIRect m_kFocusDataArea;

    ///<The bounding box of m_kAllData in data space 
    CGUIRect m_kDataBoundingBox;

    ///<Dirty flag for m_kDataBoundingBox. If this flag is true the member m_kDataBoundingBox is not valid.
    eC_Bool m_bIsDataAreaDirty;

    /// Font for text
    FontResource_t m_eFont;

    /// Array of chart data
    eC_TArray<eC_TArray<Data_t> > m_kAllData;

    //enums
    DataRepresentation_t m_eDataRepresentation;
    AxisRepresentation_t m_eXAxisRepresentation;
    AxisRepresentation_t m_eYAxisRepresentation;

    ///<this flag is true if chart is dragged
    eC_Bool m_bDraggingMode;

    /// Helper line configuration
    CGUICommonEnums::AxisAlignment_t m_eHelperLinesRepresentation;

    /// Flag that enables the selection of the data by the user
    eC_Bool m_bEnableSelection;

    //in how many parts is axis cut
    eC_UInt m_uiScalingXAxis;
    eC_UInt m_uiScalingYAxis;
    // points of data so far
    eC_Int m_iPointsOfData;

    // points drawn
    eC_Int m_iPointsAvailible;

    // X zoom related
    eC_Value m_vZoomFactorX;
    eC_Value m_vZoomFactorY;

    /**
    The label that is shown an zero
    */
    eC_Int m_iStartingNumber;

    /// Width of lines that make up axis
    eC_Value m_vAxisLineWidth;

    /// Line width 
    eC_Value m_vDrawLineWidth;

    /// Length of sale lines
    eC_Value m_vScaleLineLength;

    /// Border size in pixel
    eC_Value m_vBorderSize;

    /// Chart x offset in pixel
    eC_Value m_vChartOffsetX;

    /// Chart y offset in pixel
    eC_Value m_vChartOffsetY;

    /// Size of the point
    eC_Value m_vPointSize;

    /// Selection offset tolerance
    eC_Value m_vPointSelectionTolerance;

    /// Bar width in pixel
    eC_Value m_vBarWidth;

    /// The size of an unit in pixel
    eC_Value m_vXUnitWidthInPx;

    /// Index of the selected chart
    eC_UInt m_uiSelectedChartIndex;

    /// Index of the selected value
    eC_UInt m_uiSelectedDataIndex;

    /// If this flag is true the selection visualization is shown
    eC_Bool m_bShowSelection;

    /// Min step size for y labels
    eC_Value m_vMinYStepSize;

    /// ranges for x and y
    eC_Value m_vMinX;
    eC_Value m_vMaxX;
    eC_Value m_vMinY;
    eC_Value m_vMaxY;
};

#endif // _GUICHART_H_
