/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef GUICALENDAR_H
#define GUICALENDAR_H

#include "GUIObject.h"
#include "GUIColorPropertyObserver.h"

#include "GUIFontResource.h"
#include "GUIImageResource.h"
#include "GUITextResource.h"
#include "GUINinePatch.h"

/** A calendar-widget which can be used to display the current date and select a new one
- background-image and marker-image can be chosen
- show also last days of previous month and first days of next month
- width and height of one entry. if set to 0 they get calculated using the current size
- weekday to start with. 0 = sunday, 1 = monday, etc.
- font and color of headline
- font and color for days of current month
- font and color for days outside current month

swipe right for previous year, left for next year
swipe down for previous month, up for next month

@brief Calendar-widget for easy access
*/
class CGUICalendar : public CGUIObject, public CGUIColorPropertyObserver
{
public:
    /** Constructor.
    @param pParent Pointer to the designated parent object.
    @param vX X-position relative to its parent object, i.e. the x-offset from the left border of the parent object
    @param vY Y-position relative to its parent object, i.e. the y-offset from the upper border of the parent object
    @param vWidth Width of the object
    @param vHeight Height of the object
    @param eID Object identifier of this object (choose NO_HANDLE if none is required).
    */
    CGUICalendar(
        CGUICompositeObject *const pParent,
        const eC_Value &vX, const eC_Value &vY,
        const eC_Value &vWidth, const eC_Value &vHeight,
        const ObjectHandle_t &eID = NO_HANDLE);

    /// Default constructor to be used by the factory.
    CGUICalendar();

    virtual ~CGUICalendar();

    virtual void OnNotification();

    virtual eC_Bool IsHighlightable() const { return false; }

#if defined(GUILIANI_STREAM_GUI)
    /** Reads all object attributes from streaming file.
    This method is called by CGUIFactoryManager after one of the registered
    factories has created an instance of this class.
    */
    virtual void ReadFromStream();
#endif

#if defined(GUILIANI_WRITE_GUI)
    /** Writes all object attributes to the streaming file. A CGUIStreamWriter
    has to be initialized first.
    @param bWriteClassID This flag is used to select if writing of ControlID,
    leading and trailing tags is performed.
    */
    virtual void WriteToStream(const eC_Bool bWriteClassID = false);
#endif

    virtual void SetWidth(const eC_Value &vX);

    virtual void SetHeight(const eC_Value &vY);

    virtual eC_Bool DoButtonDown(
        const eC_Value &vAbsX,
        const eC_Value &vAbsY);

    virtual eC_Bool DoDrag(
        const eC_Value &vDeltaX,
        const eC_Value &vDeltaY,
        const eC_Value &vAbsX,
        const eC_Value &vAbsY);

    virtual eC_Bool DoDragEnd(
        const eC_Value &vAbsX,
        const eC_Value &vAbsY);

    virtual eC_Bool DoClick(
        const eC_Value &vAbsX,
        const eC_Value &vAbsY);

    /** Set the images which are used for display
    @param eBackgroundImage background-image
    @param eMarkerImage marker-image
    */
    void SetImages(const ImageResource_t& eBackgroundImage, const ImageResource_t& eMarkerImage);

    /** Return the images which are used for display
    @param eBackgroundImage background-image
    @param eMarkerImage marker-image
    */
    void GetImages(ImageResource_t& eBackgroundImage, ImageResource_t& eMarkerImage) const;

    /** Get ninepatch of background-image
    @return ninepatch
    */
    CGUINinePatch GetBackgroundNinePatch();

    /** Get ninepatch of marker-image
    @return ninepatch
    */
    CGUINinePatch GetMarkerNinePatch();

    /** Set size of day
    @param vDayWidth width of day
    @param vDayHeight height of day
    */
    void SetDaySize(const eC_Value& vDayWidth, const eC_Value& vDayHeight);

    /** Return size of day
    @param vDayWidth width of day
    @param vDayHeight height of day
    */
    void GetDaySize(eC_Value& vDayWidth, eC_Value& vDayHeight) const;

    /** Sets the color of the primitive as a 32 Bit hexadecimal value (0xAARRGGBB).
    @param uiHeadlineColor Color of drawing */
    void SetHeadlineColor(const eC_UInt& uiHeadlineColor);

    /** Sets the color of the primitive as a property
    @param eHeadlineColor property to use */
    void SetHeadlineColor(const GlobalProperty_t& eHeadlineColor);

    /** Returns the currently set color for the headline
    @return currently set color */
    eC_UInt GetHeadlineColor() const;

    /** Sets to show the headline or not
    @param bShowHeadLine show headline
    */
    void SetShowHeadLine(const eC_Bool& bShowHeadLine);

    /** Return if headline is shown
    @return true if headline is shown
    */
    eC_Bool GetShowHeadLine() const;

    /** Set the height of the headline
    @param vHeadlineHeight
    */
    void SetHeadlineHeight(const eC_Value& vHeadlineHeight);

    /** Return the current height of the headline
    @return height of headline
    */
    eC_Value GetHeadlineHeight() const;

    /** Set font for the headline
    @param eHeadlineFont Font-ID
    */
    void SetHeadlineFont(const FontResource_t& eHeadlineFont);

    /** Return the currently set font for the headline
    @return headline-font
    */
    FontResource_t GetHeadlineFont() const;

    /** Sets the color of the active entry as a 32 Bit hexadecimal value (0xAARRGGBB).
    @param uiFontColor Color of drawing */
    void SetFontColor(const eC_UInt& uiFontColor);

    /** Sets the color of the active entry as a property
    @param eFontColor property to use */
    void SetFontColor(const GlobalProperty_t& eFontColor);

    /** Returns the currently set color for the active entry
    @return currently set color */
    eC_UInt GetFontColor() const;

    /** Set font for an entry
    @param eFont Font-ID
    */
    void SetFont(const FontResource_t& eFont);

    /** Return the currently set font for entries
    @return entry-font
    */
    FontResource_t GetFont() const;

    /** Sets the color of the inactive entries as a 32 Bit hexadecimal value (0xAARRGGBB).
    @param uiInactiveFontColor Color of drawing */
    void SetInactiveFontColor(const eC_UInt& uiInactiveFontColor);

    /** Sets the color of the inactive entries as a property
    @param eInactiveFontColor property to use */
    void SetInactiveFontColor(const GlobalProperty_t& eInactiveFontColor);

    /** Returns the currently set color for inactive fentries
    @return currently set color */
    eC_UInt GetInactiveFontColor() const;

    /** Set font for an inactive entry
    @param eInactiveFont Font-ID
    */
    void SetInactiveFont(const FontResource_t& eInactiveFont);

    /** Return the currently set font for inactive entries
    @return inactive entry-font
    */
    FontResource_t GetInactiveFont() const;

    /** Sets to show weekdays or not
    @param bShowWeekDays show weekdays
    */
    void SetShowWeekDays(const eC_Bool& bShowWeekDays);

    /** Returns if weekdays are shown
    @return true if weekdays are shown
    */
    eC_Bool GetShowWeekDays() const;

    /** Sets the first day of week
    @param ubFirstDayOfWeek first day of week (0 = sunday)
    */
    void SetFirstDayOfWeek(const eC_UByte& ubFirstDayOfWeek);

    /** Returns first day of week
    @return first day of week (0 = sunday)
    */
    eC_UByte GetFirstDayOfWeek() const;

    /** Set to fill empty spaces with days from previous or next month
    @param bFillEmptySpaces fill empty spaces
    */
    void SetFillEmptySpaces(const eC_Bool& bFillEmptySpaces);

    /** Returns if empty spaces are filled
    @return true if empty spaces are filled
    */
    eC_Bool GetFillEmptySpaces() const;

    /** Sets if touch-control is active
    this will enable the user to swipe horizontally to change years and vertically to change months
    @param bTouchControl if true touch-control is active
    */
    void SetTouchControl(const eC_Bool& bTouchControl);

    /** Return if touch-control is active
    @return true if touch-control is active
    */
    eC_Bool GetTouchControl() const;

    /** Set Text for specified month to text-ID
    @param ubIndex index of month
    @param eTextID text-id to be used for month
    */
    void SetMonthText(const eC_UByte& ubIndex, const TextResource_t& eTextID);

    /** Set Text for specified month to text-string
    @param ubIndex index of month
    @param kTextString text-string to be used for month
    */
    void SetMonthText(const eC_UByte& ubIndex, const eC_String& kTextString);

    /** Return the Text of the specified month
    @param ubIndex index of month
    @return text of month as string
    */
    eC_String GetMonthText(const eC_UByte& ubIndex) const;

    /** Set Text for specified day to text-ID
    @param ubIndex index of day
    @param eTextID text-id to be used for day
    */
    void SetDayText(const eC_UByte& ubIndex, const TextResource_t& eTextID);

    /** Set Text for specified day to text-string
    @param ubIndex index of day
    @param kTextString text-string to be used for day
    */
    void SetDayText(const eC_UByte& ubIndex, const eC_String& kTextString);

    /** Return the Text of the specified day
    @param ubIndex index of day
    @return text of day as string
    */
    eC_String GetDayText(const eC_UByte& ubIndex) const;

    /** Set the current date
    @param ubDay day
    @param ubMonth month
    @param uiYear year
    */
    void SetDate(const eC_UByte& ubDay, const eC_UByte& ubMonth, const eC_UInt& uiYear);

    /** Get the current date
    @param ubDay day
    @param ubMonth month
    @param uiYear year
    */
    void GetDate(eC_UByte& ubDay, eC_UByte& ubMonth, eC_UInt& uiYear);

    /** navigate to the previous month
    */
    void PreviousMonth();

    /** navigate to the next month
    */
    void NextMonth();

    /** navigate to the previous year
    */
    void PreviousYear();

    /** navigate to the next year
    */
    void NextYear();

protected:
    virtual eC_Bool DoDraw();

private:
    void Init();

    void DeInit();

    eC_Bool IsValidInput(const eC_UByte& ubDay, const eC_UByte& ubMonth, const eC_UInt& uiYear) const;

    void UpdateValues();

#if defined(GUILIANI_STREAM_GUI)
    void ReadTexts();
#endif

#if defined(GUILIANI_WRITE_GUI)
    void WriteTexts();
#endif

    void RefreshTexts();

    static eC_Bool IsLeapYear(eC_UInt uiYear);

    static eC_UByte CalculateNumberOfDays(eC_UByte ubMonth, eC_UInt uiYear);

    static eC_UByte CalcWeekdayFromDate(eC_UByte ubDay, eC_UByte ubMonth, eC_UInt uiYear);

private:
    static const eC_UInt INDEX_HEADLINECOLOR;
    static const eC_UInt INDEX_ACTIVECOLOR;
    static const eC_UInt INDEX_INACTIVECOLOR;

    // streamed members
    eC_UByte m_ubDay;
    eC_UByte m_ubMonth;
    eC_UInt m_uiYear;

    eC_Value m_vDayWidth;
    eC_Value m_vDayHeight;

    eC_Bool m_bShowHeadLine;
    eC_Value m_vHeadlineHeight;
    FontResource_t m_eHeadlineFontID;

    ImageResource_t m_eBackgroundImageID;
    CGUINinePatch m_kBackground;

    ImageResource_t m_eMarkerImageID;
    CGUINinePatch m_kMarker;

    FontResource_t m_eFontID;
    FontResource_t m_eInactiveFontID;

    eC_Bool m_bShowWeekDays;

    eC_UByte m_ubFirstDayOfWeek;
    eC_Bool m_bFillEmptySpaces;

    eC_Bool m_bTouchControl;

    // texts for months and days
    TextResource_t m_eTextMonths[12];
    eC_String m_kTextMonths[12];
    TextResource_t m_eTextDays[7];
    eC_String m_kTextDays[7];

    // internal members
    eC_UByte m_ubFirstWeekDayOfMonth;
    eC_UByte m_ubNumberOfDays;
    eC_Value m_vAdvanceX;
    eC_Value m_vAdvanceY;
    eC_Value m_vDragStartX;
    eC_Value m_vDragStartY;
    eC_Value m_vYOffset;
    eC_UByte m_ubStartCell;
};

#endif
