/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef SETGUIOBJECTGEOMETRY_COMMAND_H
#define SETGUIOBJECTGEOMETRY_COMMAND_H

#include "GUICommand.h"
#include "GUIEasing.h"
#include "GUIObjectHandleResource.h"

/** 
    The target object is referenced by an object ID that has to be passed to
    the constructor. When this command is executed, it tries to find the
    specified object. If no object is found, this command does nothing.
    If the object is found, the object visuals are changed according the given values.
    @brief An example for a custom command that changes the object visuals of an GUIObject.
  */
class CGUISetObjectVisualsCmd : public CGUICommand
{
public:
    /** Constructor.
        @param eTargetObject The ID of the object to be modified by this command.
        @param bChangePosition If true, the position will be changed
        @param vTargetX Target value of the x-position
        @param vTargetY Target value of the y-position
        @param bAnimatePosition If true the object will be animated to the new position
        @param uiPositionDuration Duration of the animation to change the position
        @param ePositionEasing Easing which is used for animating the position
        @param bChangeSize If true, the size will be changed
        @param vTargetWidth Target value of the width
        @param vTargetHeight Target value of the height
        @param bAnimateSize If true the object will be animated to the new size
        @param uiSizeDuration Duration of the animation to the change the size
        @param eSizeEasing Easing which is used for animating the size
        @param bChangeAlpha If true, the alpha-value will be changed
        @param ubTargetAlpha Target value of the alpha
        @param bAnimateAlpha If true the object will be animated to the new alpha-value
        @param uiAlphaDuration Duration of the animation to change the alpha-value
        @param eAlphaEasing Easing which is used for animating the alpha-value
      */
    CGUISetObjectVisualsCmd(
        ObjectHandle_t eTargetObject,
        eC_Bool bChangePosition,
        eC_Value vTargetX,
        eC_Value vTargetY,
        eC_Bool bAnimatePosition,
        eC_UInt uiPositionDuration,
        CGUIEasing::EasingType_t ePositionEasing,
        eC_Bool bChangeSize,
        eC_Value vTargetWidth,
        eC_Value vTargetHeight,
        eC_Bool bAnimateSize,
        eC_UInt uiSizeDuration,
        CGUIEasing::EasingType_t eSizeEasing,
        eC_Bool bChangeAlpha,
        eC_UByte ubTargetAlpha,
        eC_Bool bAnimateAlpha,
        eC_UInt uiAlphaDuration,
        CGUIEasing::EasingType_t eAlphaEasing);

    /// Default constructor to be used by a factory.
    CGUISetObjectVisualsCmd();

#if defined(GUILIANI_STREAM_GUI)
    /** Reads all attributes from streaming file.
        This method is called by CGUIFactoryManager after one of the registered
        factories has created an instance of this class.
    */
    virtual void ReadFromStream();
#endif

#if defined(GUILIANI_WRITE_GUI)
    /** Writes all attributes to the streaming file. A CGUIStreamWriter
        has to be initialized first.
        @param bWriteClassID This flag is used to select if writing of command
               class ID, leading and trailing tags is performed.
    */
    virtual void WriteToStream(const eC_Bool bWriteClassID=false);
#endif

protected:
    /// Implements the actual command functionality.
    void Do();

    /// Target object ID.
    ObjectHandle_t m_eTargetObject;

    eC_Bool m_bChangePosition; ///< change position
    eC_Value m_vTargetX; ///< target x-position
    eC_Value m_vTargetY; ///< target y-position
    eC_Bool m_bAnimatePosition; ///< animate position
    eC_UInt m_uiPositionDuration; ///< animation duration
    CGUIEasing::EasingType_t m_ePositionEasing; ///< animation-easing

    eC_Bool m_bChangeSize; ///< change size
    eC_Value m_vTargetWidth; ///< target width
    eC_Value m_vTargetHeight; ///< target height
    eC_Bool m_bAnimateSize; ///< animate size
    eC_UInt m_uiSizeDuration; ///< animation duration
    CGUIEasing::EasingType_t m_eSizeEasing; ///< animation-easing

    eC_Bool m_bChangeAlpha; ///< change alpha
    eC_UByte m_ubTargetAlpha; ///< target alpha-value
    eC_Bool m_bAnimateAlpha; ///< animate alpha
    eC_UInt m_uiAlphaDuration; ///< animation duration
    CGUIEasing::EasingType_t m_eAlphaEasing; ///< animation-easing
};

#endif
