/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef GUIAUTOREPEATBEHAVIOUR__H_
#define GUIAUTOREPEATBEHAVIOUR__H_

#include "GUIAnimatable.h"
#include "GUIBehaviourDecorator.h"

/** @brief Behaviour for repeating clicks on object
    Additional phases can be defined where the start-time of the phase is measured from the beginning of the previous phase
*/
class CGUIAutoRepeatBehaviour : public CGUIBehaviourDecorator, public CGUIAnimatable
{
public:
    ///Default constructor
    CGUIAutoRepeatBehaviour();

    /** Constructor
    @param bUseLongClick
    @param uiInitialDelay
    @param uiInterval
    */
    CGUIAutoRepeatBehaviour(
        const eC_Bool& bUseLongClick,
        const eC_UInt& uiInitialDelay,
        const eC_UInt& uiInterval);

    /** Add an additional autorepeat-phase
    @param uiStartTime the start-time of the new phase
    calculated from the beginning of the previous phase
    @param uiInterval the interval of the new phase
    */
    void AddAdditionalPhase(const eC_UInt& uiStartTime, const eC_UInt& uiInterval);

    /** Get number of defined additional phases
    @return number of additional phases
    */
    eC_UInt GetNumberOfAdditionalPhases() const;

#if defined(GUILIANI_STREAM_GUI)
    /** Reads all attributes from streaming file.
        This method is called by CGUIFactoryManager after one of the registered
        factories has created an instance of this class.
    */
    virtual void ReadFromStream();
#endif

#if defined(GUILIANI_WRITE_GUI)
    /** Writes all attributes to the streaming file. A CGUIStreamWriter
        has to be initialized first.
        @param bWriteClassID This flag is used to select if writing of command
               class ID, leading and trailing tags is performed.
    */
    virtual void WriteToStream(const eC_Bool bWriteClassID = false);
#endif

protected:
    virtual void DoAnimate(const eC_Value& vTimes = eC_FromInt(1));

    virtual eC_Bool DoButtonDown(
        const eC_Value &vAbsX,
        const eC_Value &vAbsY);

    virtual eC_Bool DoButtonUp(
        const eC_Value &vAbsX,
        const eC_Value &vAbsY);

    virtual eC_Bool DoClick(
        const eC_Value &vAbsX,
        const eC_Value &vAbsY);

    virtual eC_Bool DoLongClick(
        const eC_Value &vAbsX,
        const eC_Value &vAbsY);

    virtual eC_Bool DoDragEnd(
        const eC_Value &vAbsX,
        const eC_Value &vAbsY);

private:
    void EnterRepeatPhase(const eC_UInt& uiInterval);

private:
    enum AutoRepeatState_t
    {
        ARS_IDLE,
        ARS_DELAY,
        ARS_REPEAT
    };

    eC_Bool m_bUseLongClick;
    eC_UInt m_uiInitialDelay;
    eC_UInt m_uiInterval;

    AutoRepeatState_t m_eState;

    eC_UInt m_uiNumAdditionalPhases;
    eC_UInt m_uiCurrentPhase;
    eC_UInt m_uiLastPhaseStart;

    eC_TArray<eC_UInt> m_aStartTimes;
    eC_TArray<eC_UInt> m_aIntervals;

    eC_Bool m_bIgnoreNextClick;
};

#endif
