/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#if !defined GUIANIMATION_STDGUIOBJECT__H_
#define GUIANIMATION_STDGUIOBJECT__H_

#include "GUIAnimation.h"
#include "GUIEasing.h"

/**
This animation animates a standard CGUIObject object through giving access to all
important object attributes which are x position, y position, width, height and alpha value.
For all of these attributes a separate CGUIEasing functions is used.

This example moves and changes the size of an object using CGUIEasing functions.

@code

// The animation shall be done in 5 seconds.
const eC_UInt cuiDuration = 5000;

// Now create the instance of the animation.
CGUIAnimationStdGUIObject* pAnimation = new CGUIAnimationStdGUIObject(pkAnimatedObject,
CGUIEasing::EASE_LINEAR, CGUIEasing::EASE_LINEAR,
CGUIEasing::EASE_LINEAR, CGUIEasing::EASE_LINEAR,
CGUIEasing::EASE_LINEAR,
cuiDuration);

// To start the animation call the StartAnimation() method of the animation.
pAnimation->StartAnimation();

//If the instance of the animation is not available directly use the AnimationHandler.
eC_UInt uiMyAnimationID = pAnimation->GetID();

GETANIMATIONHDL.GetAnimation(uiMyAnimationID);

@endcode
@brief Generic animation of a standard object
*/
class CGUIAnimationStdGUIObject : public CGUIAnimation
{
public:

#if defined(GUILIANI_STREAM_GUI)
    CGUIAnimationStdGUIObject();
#endif
    ~CGUIAnimationStdGUIObject();

    /**
    Constructor
    @param pAnimatedObject The GUIObject that will be moved.
    @param eEasingX The easing type that is used for the XPos of the object.
    @param eEasingY The easing type that is used for the YPos of the object.
    @param eEasingWidth The easing type that is used for the width of the object.
    @param eEasingHeight The easing type that is used for the height of the object.
    @param eEasingAlpha The easing type that is used for the alpha of the object.
    @param uiDuration The total duration of the animation in milliseconds. Based on the
    distance the duration determines the animation speed.
    @param uiStepTime The time between two animation steps in milliseconds.
    */
    CGUIAnimationStdGUIObject(
        CGUIObject* pAnimatedObject,
        CGUIEasing::EasingType_t eEasingX,
        CGUIEasing::EasingType_t eEasingY,
        CGUIEasing::EasingType_t eEasingWidth,
        CGUIEasing::EasingType_t eEasingHeight,
        CGUIEasing::EasingType_t eEasingAlpha,
        eC_UInt uiDuration = 1000,
        eC_UInt uiStepTime = 25);

    /**
    Constructor
    @param pkAnimatedObject The GUIObject that will be moved.
    @param eEasing The easing type that is used for all attributes.
    @param uiDuration The total duration of the animation in milliseconds. Based on the
    distance the duration determines the animation speed.
    @param uiStepTime The time between two animation steps in milliseconds.
    */
    CGUIAnimationStdGUIObject(
        CGUIObject* pkAnimatedObject,
        CGUIEasing::EasingType_t eEasing,
        eC_UInt uiDuration = 1000,
        eC_UInt uiStepTime = 25);

    virtual void ApplyAnimation(eC_UInt uiElapsedTime);

    /** Retrieves the target values for this animation from the current status of the supplied GUIObject.
    @param pObj Object whose current attributes will serve as the new target values for this animation
    */
    void SetTargetValues(const CGUIObject* pObj);

    /** Retrieves the start values for this animation from the current status of the supplied GUIObject.
    @param pObj Object whose current attributes will serve as the start values for this animation
    */
    void SetStartValues(const CGUIObject* pObj);

    /** Get the easing type for the nth attribute
    @param uiN index
    @return easing type
    */
    CGUIEasing::EasingType_t GetEasingForNthAttribute(eC_UInt uiN);

    /** Set the easing type type for the nth attribute
    @param uiN index
    @param eEasingType easing type
    */
    void SetEasingForNthAttribute(eC_UInt uiN, CGUIEasing::EasingType_t eEasingType);

    /** Get the start value type for the nth attribute
    @param uiN index
    @return start value
    */
    eC_Value GetStartValueForNthAttribute(eC_UInt uiN);

    /** Set the start value type for the nth attribute
    @param uiN index
    @param vStart start value
    */
    void SetStartValueForNthAttribute(eC_UInt uiN, eC_Value vStart);

    /** Get the target value type for the nth attribute
    @param uiN index
    @return target value
    */
    eC_Value GetTargetValueForNthAttribute(eC_UInt uiN);

    /** Set the start value type for the nth attribute
    @param uiN index
    @param vTarget target value
    */
    void SetTargetValueForNthAttribute(eC_UInt uiN, eC_Value vTarget);

    /** Get the nth attribute
    @param uiN index
    @return attribute
    */
    eC_Value GetAnimatedObjectNthAttribute(eC_UInt uiN);

    /** Set the nth attribute
    @param uiN index
    @param vValue attribute
    */
    void SetAnimatedObjectNthAttribute(eC_UInt uiN, eC_Value vValue);

#if defined(GUILIANI_STREAM_GUI) || defined(GUILIANI_WRITE_GUI)
    static const eC_UInt STDGUIOBJECT_ANIMATION_CLASS_VERSION; ///< class-version string
    static const eC_String XMLTAG_ATTRIBUTES[]; ///< attributes-string
#endif

#ifdef GUILIANI_STREAM_GUI
    virtual void ReadFromStream();
#endif

#ifdef GUILIANI_WRITE_GUI
    virtual void WriteToStream(const eC_Bool bWriteClassID = false);
#endif

private:
    CGUIAnimatedAttribute::AnimatedAttribute_t m_eType[CGUIAnimatedAttribute::ATTR_COUNT]; ///< type of animated attribute
    eC_Bool m_bEnabled[CGUIAnimatedAttribute::ATTR_COUNT]; ///< if animation is enabled
};

#endif
